﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "dhcps_BpfManager.h"
#include "dhcps_UdpSocketManager.h"
#include "dhcps_DhcpManager.h"
#include "dhcps_LeaseTableManager.h"
#include "dhcps_Common.h"
#include "dhcps_Utils.h"

/**
 * @file
 *
 * @brief This file is the header file for the DHCP Server event
 * coordinator.
 */

namespace nn { namespace dhcps { namespace detail {

/**
 * @brief This is the event coordinator class. It receives and
 * delivers all events to respective event listeners.
 */
class Coordinator : public IEventHandler
{
public:
    /**
     * @brief This states of the @ref Coordinator.
     */
    enum class State : uint8_t
    {
        Uninitialized                 = 0,  ///< 0
        InitializingBpfManager        = 1,  ///< 1
        InitializingUdpManager        = 2,  ///< 2
        InitializingLeaseTableManager = 3,  ///< 3
        InitializingDhcpManager       = 4,  ///< 4
        Initialized                   = 5,  ///< 5
        Opening                       = 6,  ///< 6
        OpenSuccess                   = 7,  ///< 7
        Running                       = 8,  ///< 8
        Stopping                      = 9,  ///< 9
        Stopped                       = 10, ///< 10
        Error                         = 11, ///< 11
    };

    /**
     * @brief Constructor.
     */
    Coordinator() NN_NOEXCEPT;

    /**
     * @brief Destructor.
     */
    ~Coordinator() NN_NOEXCEPT;

    /**
     * @brief Initializes the @ref Coordinator for use with the
     * provided @ref LeaseTableManager
     *
     * @param[in] pLeaseTableManager The provided LeaseTableManager.
     *
     * @return On success return 0, on error return -1.
     */
    int Initialize(LeaseTableManager* pLeaseTableManager) NN_NOEXCEPT;

    /**
     * @brief Finalizes the @ref Coordinator.
     *
     * @return On success return 0, on error return -1.
     */
    int Finalize() NN_NOEXCEPT;

    /**
     * @brief Receives all @ref IEventHandler events because the
     * @ref Coordinator forwards events to all other listeners.
     *
     * @param[in] e The event to receive and forward.
     */
    virtual void OnEvent(const InternalEvent& e) NN_NOEXCEPT;

    /**
     * @brief Starts the Coordinator and DHCP server functionality.
     *
     * @details
     * - Called by the DHCP worker thread.
     * - It assumes that @a ApiLock() is being held and calls
     *  @a ApiUnlock() after it obtains @a InternalLock().
     */
    int Run() NN_NOEXCEPT;

    /**
     * @brief Calls @a InternalLock() and changes the
     * @ref Coordinator state to @ref State::Stopping. The worker
     * thread quits after the next timeout.
     */
    void ExternalStop() NN_NOEXCEPT;

    /**
     * @brief Changes the @ref Coordinator state to
     * @ref State::Stopping without taking a lock. The worker thread
     * quits after the next timeout.
     */
    void InternalStop() NN_NOEXCEPT;

private:
    /**
     * @brief The current state.
     */
    State m_State;

    /**
     * @brief The @ref BpfManager instance.
     */
    BpfManager m_BpfManager;

    /**
     * @brief The @ref UdpSocketManager instance.
     */
    UdpSocketManager m_UdpSocketManager;

    /**
     * @brief The @ref DhcpManager instance.
     */
    DhcpManager m_DhcpManager;

    /**
     * @brief A pointer to the @ref LeaseTableManager instance.
     */
    LeaseTableManager* m_pLeaseTableManager;

    /**
     * @brief Changes the state variable.
     *
     * @param[in] next The next state to transition to.
     */
    void ChangeState(State next) NN_NOEXCEPT;

    /**
     * @brief Get the desired time for the next timeout event.
     *
     * @details The Coordinator queries all the listeners for their
     * respective timeouts and selects the lowest timeout for its call
     * to @ref nn::socket::Select.
     *
     * @param[out] pOutTimeval The desired time for the next timeout event.
     */
    virtual void GetTimeout(nn::socket::TimeVal* pOutTimeval) NN_NOEXCEPT;

    /**
     * @brief Called on @ref EventType::OnFileOpen events.
     */
    void OnFileOpen() NN_NOEXCEPT;

    /**
     * @brief Called on @ref EventType::OnFileRead events.
     *
     * @param[in] pSet The set of file descriptors from @ref BpfManager
     * and @ref UdpSocketManager.
     *
     * @param[in] bpf This is the BPF file descriptor.
     *
     * @param[in] udp This is the UDP Socket Manager burner socket.
     */
    void OnFileRead(nn::socket::FdSet* pSet, int bpf, int udp) NN_NOEXCEPT;

    /**
     * @brief Handles @ref EventType::OnTimerExpired events.
     */
    void OnTimerExpired() NN_NOEXCEPT;
};

}}}; // nn::dhcps::detail
