﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "dhcps_Config.h"

//#define NN_DETAIL_DHCPS_LOG_LEVEL NN_DETAIL_DHCPS_LOG_LEVEL_DEBUG
#define NN_DETAIL_DHCPS_LOG_MODULE_NAME "ServerConfig"
#include "dhcps_Log.h"

#include <algorithm>

namespace nn { namespace dhcps { namespace detail {

namespace {

const unsigned int DefaultClientPort = 68;
const unsigned int DefaultServerPort = 67;

}; // end anonymous namespace

Config g_Config;

Config::Config()  NN_NOEXCEPT :
    m_HasGateway(false),
    m_DnsServerCount(false),
    m_pLeaseTableMemory(nullptr),
    m_LeaseTimerSeconds {},
    m_T1Ratio {},
    m_T2Ratio {},
    m_ClientUdpPort(DefaultClientPort),
    m_ServerUdpPort(DefaultServerPort),
    m_InterfaceMtu {},
    m_IpAddress {},
    m_Netmask {},
    m_Network {},
    m_Broadcast {},
    m_RangeBegin {},
    m_RangeEnd {}
{
    memset(m_InterfaceMacAddress, 0, sizeof(m_InterfaceMacAddress));
    memset(m_pInterfaceName, 0, sizeof(m_pInterfaceName));
    memset(m_pDnsServers, 0, sizeof(m_pDnsServers));
};

Result Config::FromUserConfiguration(const UserConfiguration& userConfig) NN_NOEXCEPT
{
    Result result = ResultInterfaceNameStringInvalid();
    int rc;

    if (nullptr == userConfig.pInterfaceName)
    {
        NN_DETAIL_DHCPS_LOG_DEBUG("UserConfiguration interfaceName "
                                       "string length is null.\n");
        // result already set
        goto bail;
    }
    else if (0 == (rc = (nn::util::Strnlen(userConfig.pInterfaceName,
                                           static_cast<int>(LibraryConstants::MaximumInterfaceNameLength)))))
    {
        NN_DETAIL_DHCPS_LOG_DEBUG("UserConfiguration interfaceName "
                                       "string length is 0.\n");
        // result already set
        goto bail;
    }
    else if (rc > sizeof(m_pInterfaceName))
    {
        NN_DETAIL_DHCPS_LOG_DEBUG("UserConfiguration interfaceName "
                                       "string length is too big.\n");
        // result already set
        goto bail;
    };

    memcpy(m_pInterfaceName, userConfig.pInterfaceName, rc);

    m_RangeBegin.S_addr = userConfig.begin.S_addr;
    m_RangeEnd.S_addr = userConfig.end.S_addr;

    if (nn::socket::InetNtohl(m_RangeEnd.S_addr) <
        nn::socket::InetNtohl(m_RangeBegin.S_addr))
    {
        NN_DETAIL_DHCPS_LOG_DEBUG("UserConfiguration range is invalid "
                                       "because the end is before the begin.\n");
        result = ResultRangeIsInvalid();
        goto bail;
    };

    m_pCallback = userConfig.pCallback;
    if (nullptr == m_pCallback)
    {
        NN_DETAIL_DHCPS_LOG_DEBUG("UserConfiguration pCallback is null.\n");
        result = ResultNullCallback();
        goto bail;
    };

    m_pContext = userConfig.pContext;

    m_OfferTimerSeconds = userConfig.offerSeconds;
    if (0 == m_OfferTimerSeconds)
    {
        NN_DETAIL_DHCPS_LOG_DEBUG("UserConfiguration offer timer seconds "
                                       "is zero.\n");
        result = ResultOfferTimerSecondsInvalid();
        goto bail;
    };

    m_LeaseTimerSeconds = userConfig.leaseSeconds;
    if (0 == m_LeaseTimerSeconds)
    {
        NN_DETAIL_DHCPS_LOG_DEBUG("UserConfiguration lease timer seconds "
                                       "is zero.\n");
        result = ResultLeaseTimerSecondsInvalid();
        goto bail;
    };

    m_T1Ratio = userConfig.t1Ratio;
    if (0 == m_T1Ratio)
    {
        NN_DETAIL_DHCPS_LOG_DEBUG("UserConfiguration t1 ratio is zero.\n");
        result = ResultT1RatioInvalid();
        goto bail;
    };

    m_T2Ratio = userConfig.t2Ratio;
    if (0 == m_T2Ratio)
    {
        NN_DETAIL_DHCPS_LOG_DEBUG("UserConfiguration t2 ratio is zero.\n");
        result = ResultT2RatioInvalid();
        goto bail;
    };

    if (m_T1Ratio >= m_T2Ratio)
    {
        NN_DETAIL_DHCPS_LOG_DEBUG("UserConfiguration t1 timer ratio (%f) "
                                       "is greater than or equal to the t2 "
                                       "timer ratio (%f).\n", m_T1Ratio, m_T2Ratio);
        result = ResultT1GreaterOrEqualToT2();
        goto bail;
    };

    m_HasGateway = userConfig.hasGateway;
    m_Gateway.S_addr = userConfig.gateway.S_addr;

    if (userConfig.dnsServerCount != 0)
    {
        if (userConfig.pDnsServers == nullptr)
        {
            NN_DETAIL_DHCPS_LOG_DEBUG("UserConfiguration dnsServerCount is zero "
                                           "but pDnsServers is null\n");
            result = ResultNullDnsServerArray();
            goto bail;
        };

        m_DnsServerCount = userConfig.dnsServerCount;
        for (unsigned int idx = 0; idx < m_DnsServerCount; ++idx)
        {
            m_pDnsServers[idx].S_addr = userConfig.pDnsServers[idx].S_addr;
        };
    };

    result = ResultSuccess();

bail:
    return result;
};

Result Config::AcquireNetworkValues() NN_NOEXCEPT
{
    Result result = ResultNetworkConfigurationInvalid();
    int rc = -1;
    const int MaximumInterfaces = 16;
    nn::socket::IfReq pIfreqBuffer[MaximumInterfaces];
    memset(pIfreqBuffer, 0, sizeof(pIfreqBuffer));
    nn::socket::IfReq* pIfreqPointer = nullptr;
    nn::socket::IfReq* pIfreqPointerEnd = nullptr;
    nn::socket::IfReq ifr_ioctl;
    nn::socket::IfConf ifconfig;
    int sock = -1;
    nn::socket::Errno errorNumber;
    uint32_t begin, end, net, broad, ip, gateway, mask;

    ifconfig.ifc_len = sizeof(pIfreqBuffer);
    ifconfig.ifc_buf = reinterpret_cast<caddr_t>(pIfreqBuffer);

    if (-1 == (sock = nn::socket::SocketExempt(nn::socket::Family::Af_Inet,
                                               nn::socket::Type::Sock_Dgram,
                                               nn::socket::Protocol::IpProto_Udp)))
    {
        errorNumber = nn::socket::GetLastError();
        NN_DETAIL_DHCPS_LOG_MAJOR("Socket error: %s (%d) Unable to "
                                       "configure the DHCP library.\n",
                                       strerror(static_cast<int>(errorNumber)),
                                       errorNumber);
        result = ResultSocketFailed();
        goto bail;
    }
    else if (-1 == nn::socket::Ioctl(sock,
                                     socket::IoctlCommandPrivate::SiocGIfConf,
                                     reinterpret_cast<char*>(&ifconfig),
                                     sizeof(ifconfig)))
    {
        errorNumber = nn::socket::GetLastError();
        NN_DETAIL_DHCPS_LOG_MAJOR("SiocGIfConf Ioctl Error %s (%d)\n",
                                       strerror(static_cast<int>(errorNumber)),
                                       errorNumber);
        result = ResultIoctlFailed();
        goto bail;
    }
    else if (static_cast<size_t>(ifconfig.ifc_len) < sizeof(nn::socket::IfReq))
    {
        NN_DETAIL_DHCPS_LOG_MAJOR("SiocGIfConf Ioctl Error "
                                       "ifc.ifc_len (%zu) is too small\n",
                                       ifconfig.ifc_len);
        result = ResultIoctlFailed();
        goto bail;
    };

    pIfreqPointer = pIfreqBuffer;
    pIfreqPointerEnd = reinterpret_cast<nn::socket::IfReq*>(reinterpret_cast<char*>(pIfreqBuffer) + ifconfig.ifc_len);

    // get mac, ip, and broadcast address
    while (pIfreqPointer < pIfreqPointerEnd)
    {
        size_t n;
        NN_DETAIL_DHCPS_LOG_DEBUG("interface name: %s\n", pIfreqPointer->ifr_name);

        if (0 == nn::util::Strnlen(pIfreqPointer->ifr_name, sizeof(pIfreqPointer->ifr_name)))
        {
            break;
        };

        if (0 == nn::util::Strncmp(m_pInterfaceName, pIfreqPointer->ifr_name, sizeof(pIfreqPointer->ifr_name)))
        {
            m_InterfaceMtu = pIfreqPointer->ifr_mtu;

            if (pIfreqPointer->ifr_addr.sa_family == nn::socket::Family::Af_Link)
            {
                unsigned char* ptr = reinterpret_cast<unsigned char*>(nn::socket::LlAddr(reinterpret_cast<nn::socket::SockAddrDl*>(&pIfreqPointer->ifr_addr)));
                memcpy(m_InterfaceMacAddress, ptr, static_cast<size_t>(LibraryConstants::MacAddressSize));
            }
            else if (pIfreqPointer->ifr_addr.sa_family == nn::socket::Family::Af_Inet)
            {
                nn::socket::SockAddrIn* sin = reinterpret_cast<nn::socket::SockAddrIn*>(&pIfreqPointer->ifr_addr);
                m_IpAddress.S_addr = sin->sin_addr.S_addr;

                sin = reinterpret_cast<nn::socket::SockAddrIn*>(&pIfreqPointer->ifr_broadaddr);

                memset(&ifr_ioctl, 0, sizeof(ifr_ioctl));
                nn::util::Strlcpy(ifr_ioctl.ifr_name, pIfreqPointer->ifr_name, nn::socket::If_NameSize);

                if (-1 == nn::socket::Ioctl(sock,
                                            static_cast<nn::socket::IoctlCommand>(nn::socket::IoctlCommandPrivate::SiocGIfNetmask),
                                            reinterpret_cast<char*>(&ifr_ioctl),
                                            sizeof(ifr_ioctl)))
                {
                    errorNumber = nn::socket::GetLastError();
                    NN_DETAIL_DHCPS_LOG_MAJOR("SiocGIfNetmask Ioctl Error %s (%d)\n",
                                                   strerror(static_cast<int>(errorNumber)),
                                                   errorNumber);
                    result = ResultNetmaskFailed();
                    goto bail;
                };

                sin = reinterpret_cast<nn::socket::SockAddrIn*>(&ifr_ioctl.ifr_addr);

                m_Netmask.S_addr = sin->sin_addr.S_addr;
                m_Network.S_addr = m_IpAddress.S_addr & m_Netmask.S_addr;
                m_Broadcast.S_addr = (m_IpAddress.S_addr & m_Netmask.S_addr) | ~m_Netmask.S_addr;
            };
        };

        n = pIfreqPointer->ifr_addr.sa_len + sizeof(pIfreqPointer->ifr_name);
        if (n < sizeof(*pIfreqPointer))
        {
            n = sizeof(*pIfreqPointer);
        };

        pIfreqPointer = reinterpret_cast<nn::socket::IfReq*>(reinterpret_cast<char*>(pIfreqPointer) + n);
    };

    if (0 == m_IpAddress.S_addr)
    {
        NN_DETAIL_DHCPS_LOG_DEBUG("DhcpServer: bad IP Address.\n");
        result = ResultIpInvalid();
    };

    if (0 == m_Netmask.S_addr)
    {
        NN_DETAIL_DHCPS_LOG_DEBUG("DhcpServer: bad netmask.\n");
        result = ResultNetmaskInvalid();
        rc = -1;
    };

    if (0 == m_Network.S_addr)
    {
        NN_DETAIL_DHCPS_LOG_DEBUG("DhcpServer: bad network address.\n");
        result = ResultNetworkInvalid();
        rc = -1;
    };

    if (0 == m_Broadcast.S_addr)
    {
        NN_DETAIL_DHCPS_LOG_DEBUG("DhcpServer: bad broadcast address.\n");
        result = ResultBroadcastInvalid();
        rc = -1;
    };

    ip      = nn::socket::InetNtohl(m_IpAddress.S_addr);
    net     = nn::socket::InetNtohl(m_Network.S_addr);
    gateway = nn::socket::InetNtohl(m_Gateway.S_addr);
    mask    = nn::socket::InetNtohl(m_Netmask.S_addr);
    broad   = nn::socket::InetNtohl(m_Broadcast.S_addr);
    begin   = nn::socket::InetNtohl(m_RangeBegin.S_addr);
    end     = nn::socket::InetNtohl(m_RangeEnd.S_addr);

    if (!(begin > net && begin < (net | ~mask)))
    {
        NN_DETAIL_DHCPS_LOG_DEBUG("DhcpServer: range begin not in netmask.\n");
        result = ResultRangeNotInNetwork();
        goto bail;
    };

    if (!(end >= net && end <= (net | ~mask)))
    {
        NN_DETAIL_DHCPS_LOG_DEBUG("DhcpServer: range end not in netmask.\n");
        result = ResultRangeNotInNetwork();
        goto bail;
    };

    if (ip >= begin && ip <= end)
    {
        NN_DETAIL_DHCPS_LOG_DEBUG("DhcpSerer: self ip in range.\n");
        result = ResultRangeContainsSelfIp();
        goto bail;
    };

    if (net >= begin && net <= end)
    {
        NN_DETAIL_DHCPS_LOG_DEBUG("DhcpServer: range contains network address.\n");
        result = ResultRangeContainsNetworkAddress();
        goto bail;
    };

    if (broad >= begin && broad <= end)
    {
        NN_DETAIL_DHCPS_LOG_DEBUG("DhcpSever: range contains broadcast address.\n");
        result = ResultRangeContainsBroadcastAddress();
        goto bail;
    };

    if (m_HasGateway)
    {
        if (gateway >= begin && gateway <= end)
        {
            NN_DETAIL_DHCPS_LOG_MAJOR("DhcpServer: range contains gateway address.\n");
            result = ResultRangeContainsGateway();
        };
    };

    result = ResultSuccess();

bail:

    if ( -1 != sock)
    {
        nn::socket::Close(sock);
    };

    return result;
}; //NOLINT(impl/function_size)

void Config::SetLeaseTableMemory(void* pMemoryIn, size_t size) NN_NOEXCEPT
{
    m_pLeaseTableMemory = pMemoryIn;
    m_LeaseTableMemorySize = size;
};

void Config::Print() NN_NOEXCEPT
{
#if NN_DETAIL_DHCPS_LOG_LEVEL >= NN_DETAIL_DHCPS_LOG_LEVEL_DEBUG
    char tmp[static_cast<size_t>(LibraryConstants::MacAddressStringBufferSize)];
    NN_UNUSED(tmp);

    NN_DETAIL_DHCPS_LOG_DEBUG("interfaceName: %s\n", m_pInterfaceName);
    NN_DETAIL_DHCPS_LOG_DEBUG("interfaceMacAddress: %s\n", MacAddressToString(tmp, sizeof(tmp), m_InterfaceMacAddress));
    NN_DETAIL_DHCPS_LOG_DEBUG("interfaceMtu: %d\n", m_InterfaceMtu);

    NN_DETAIL_DHCPS_LOG_DEBUG("ipAddress: %s\n", nn::socket::InetNtoa(m_IpAddress));
    NN_DETAIL_DHCPS_LOG_DEBUG("broadcast: %s\n", nn::socket::InetNtoa(m_Broadcast));
    NN_DETAIL_DHCPS_LOG_DEBUG("netmask:  %s\n", nn::socket::InetNtoa(m_Netmask));
    NN_DETAIL_DHCPS_LOG_DEBUG("network: %s\n", nn::socket::InetNtoa(m_Network));

    NN_DETAIL_DHCPS_LOG_DEBUG("clientUdpPort: %u\n", m_ClientUdpPort);
    NN_DETAIL_DHCPS_LOG_DEBUG("serverUdpPort: %u\n", m_ServerUdpPort);

    NN_DETAIL_DHCPS_LOG_DEBUG("rangeBegin: %s\n", nn::socket::InetNtoa(m_RangeBegin));
    NN_DETAIL_DHCPS_LOG_DEBUG("rangeEnd: %s\n", nn::socket::InetNtoa(m_RangeEnd));

    NN_DETAIL_DHCPS_LOG_DEBUG("offerTimerSeconds: %u\n", m_OfferTimerSeconds);
    NN_DETAIL_DHCPS_LOG_DEBUG("leaseTimerSeconds: %u\n", m_LeaseTimerSeconds);
    NN_DETAIL_DHCPS_LOG_DEBUG("t1Ratio: %f\n", m_T1Ratio);
    NN_DETAIL_DHCPS_LOG_DEBUG("t2Ratio: %f\n", m_T2Ratio);

    NN_DETAIL_DHCPS_LOG_DEBUG("pCallback: %p\n", m_pCallback);
    NN_DETAIL_DHCPS_LOG_DEBUG("pContext: %p\n", m_pContext);

    NN_DETAIL_DHCPS_LOG_DEBUG("hasGateway:  %s\n", m_HasGateway == true ? "true" : "false");
    if (m_HasGateway)
    {
        NN_DETAIL_DHCPS_LOG_DEBUG("gateway:  %s\n", nn::socket::InetNtoa(m_Gateway));
    };

    NN_DETAIL_DHCPS_LOG_DEBUG("dnsServerCount: %u\n", m_DnsServerCount);
    if (0 != m_DnsServerCount)
    {
        for (unsigned int idx = 0; idx < m_DnsServerCount; ++idx)
        {
            NN_DETAIL_DHCPS_LOG_DEBUG("dnsServers[%d]: %s.\n", idx, nn::socket::InetNtoa(m_pDnsServers[idx]));
        };
    };
#endif
};

const char* Config::GetInterfaceName() const NN_NOEXCEPT
{
    return m_pInterfaceName;
};

bool Config::GetHasGateway() const NN_NOEXCEPT
{
    return m_HasGateway;
};

nn::socket::InAddr Config::GetGateway() const NN_NOEXCEPT
{
    return m_Gateway;
};

uint32_t Config::GetDnsServerCount() const NN_NOEXCEPT
{
    return m_DnsServerCount;
};

const nn::socket::InAddr* Config::GetDnsServers() const NN_NOEXCEPT
{
    return m_pDnsServers;
};

void* Config::GetLeaseTableMemory() NN_NOEXCEPT
{
    return m_pLeaseTableMemory;
};

size_t Config::GetLeaseTableMemorySize() const NN_NOEXCEPT
{
    return m_LeaseTableMemorySize;
};

ApplicationCallback Config::GetCallback() const NN_NOEXCEPT
{
    return m_pCallback;
};

void* Config::GetContext() const NN_NOEXCEPT
{
    return m_pContext;
};

uint32_t Config::GetOfferTimerSeconds() const NN_NOEXCEPT
{
    return m_OfferTimerSeconds;
};

uint32_t Config::GetLeaseTimerSeconds() const NN_NOEXCEPT
{
    return m_LeaseTimerSeconds;
};

double Config::GetT1Ratio() const NN_NOEXCEPT
{
    return m_T1Ratio;
};

double Config::GetT2Ratio() const NN_NOEXCEPT
{
    return m_T2Ratio;
};

uint16_t Config::GetDhcpClientUdpPort() const NN_NOEXCEPT
{
    return m_ClientUdpPort;
};

uint16_t Config::GetDhcpServerUdpPort() const NN_NOEXCEPT
{
    return m_ServerUdpPort;
};

void Config::GetInterfaceMacAddress(EthernetMacAddress* pOutMac) const NN_NOEXCEPT
{
    if (pOutMac == nullptr)
    {
        NN_SDK_ASSERT(false);
        return;
    };

    memcpy(pOutMac, m_InterfaceMacAddress, sizeof(*pOutMac));
};

uint16_t Config::GetInterfaceMtu() const NN_NOEXCEPT
{
    return m_InterfaceMtu;
};

nn::socket::InAddr Config::GetIpAddress() const NN_NOEXCEPT
{
    return m_IpAddress;
};

nn::socket::InAddr Config::GetNetmask() const NN_NOEXCEPT
{
    return m_Netmask;
};

nn::socket::InAddr Config::GetNetwork() const NN_NOEXCEPT
{
    return m_Network;
};

nn::socket::InAddr Config::GetBroadcast() const NN_NOEXCEPT
{
    return m_Broadcast;
};

nn::socket::InAddr Config::GetRangeBegin() const NN_NOEXCEPT
{
    return m_RangeBegin;
};

nn::socket::InAddr Config::GetRangeEnd() const NN_NOEXCEPT
{
    return m_RangeEnd;
};

bool DetectNetworkChange() NN_NOEXCEPT
{
    Config compare;
    EthernetMacAddress a, b;
    bool different;

    std::memcpy(&compare, &g_Config, sizeof(g_Config));

    if (compare.AcquireNetworkValues().IsFailure())
    {
        return true;
    };

    compare.GetInterfaceMacAddress(&a);
    g_Config.GetInterfaceMacAddress(&b);

    different = !(0 == memcmp(a, b, sizeof(a))                                    &&
                  compare.GetInterfaceMtu()     == g_Config.GetInterfaceMtu()     &&
                  compare.GetIpAddress().S_addr == g_Config.GetIpAddress().S_addr &&
                  compare.GetBroadcast().S_addr == g_Config.GetBroadcast().S_addr &&
                  compare.GetNetmask().S_addr   == g_Config.GetNetmask().S_addr   &&
                  compare.GetNetwork().S_addr   == g_Config.GetNetwork().S_addr   );

    if (different)
    {
        compare.Print();
    };

    return different;
};

}}}; //nn::dhcps::detail
