﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "dauth_RequestParameter.h"

#include <cstring>
#include <mutex>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/crypto/crypto_Aes128CmacGenerator.h>
#include <nn/dauth/detail/dauth_Result.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/system/settings_FirmwareVersion.h>
#include <nn/spl/spl_Api.h>
#include <nn/util/util_Base64.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_TFormatString.h>

namespace nn { namespace dauth { namespace detail {
namespace {
/*
keySource: 0xde,0xd2,0x4c,0x35,0xa5,0xd8,0xc0,0xd7,0x6c,0xb8,0xd7,0x8c,0xa0,0xa5,0xa5,0x22
*/
Result ComputeCmac(void* out, size_t outSize, const void* in, size_t inSize, const KeySource& keySource) NN_NOEXCEPT
{
    NN_SDK_ASSERT(outSize == crypto::Aes128CmacGenerator::MacSize);

    static const uint8_t KekSource[] = {
        0x8b, 0xe4, 0x5a, 0xbc, 0xf9, 0x87, 0x02, 0x15, 0x23, 0xca, 0x4f, 0x5e, 0x23, 0x00, 0xdb, 0xf0,
    };
    NN_STATIC_ASSERT(sizeof(KekSource) == spl::KeySourceSize);

    int slotIndex;
    NN_RESULT_DO(spl::AllocateAesKeySlot(&slotIndex));
    NN_UTIL_SCOPE_EXIT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(spl::DeallocateAesKeySlot(slotIndex));
    };

    spl::AccessKey accessKey;
    NN_RESULT_DO(spl::GenerateAesKek(&accessKey, KekSource, sizeof(KekSource), AuthenticationKeyGeneration, 0));
    NN_RESULT_DO(spl::LoadAesKey(slotIndex, accessKey, keySource.data, sizeof(keySource.data)));
    NN_RESULT_DO(spl::ComputeCmac(out, outSize, slotIndex, in, inSize));
    NN_RESULT_SUCCESS;
}

} // ~namespace nn::dauth::detail::<anonymous>

size_t GetBase64OutputSize(size_t in) NN_NOEXCEPT
{
    NN_SDK_ASSERT(std::numeric_limits<size_t>::max() / 4 >= in);
    const auto v = in * 4;
    return (v % 3
        ? v / 3 + 1
        : v / 3);
}

Result GeneratePostData(
    char* out, size_t outSize,
    uint64_t clientId, const Challenge& challenge, const KeySource& keySource) NN_NOEXCEPT
{
    // SystemVersion
    settings::system::FirmwareVersionDigest sysVersion;
    settings::system::GetFirmwareVersionDigest(&sysVersion);

    // 入力データの作成
    static const char Format0[] = ""
        "challenge=%.*s"
        "&client_id=%016llx"
        "&key_generation=%d"
        "&system_version=%.*s";
    static const char Format1[] = ""
        "&mac=";
    static const size_t CodedMacSize = GetBase64OutputSize(crypto::Aes128CmacGenerator::MacSize);

    auto inLength0 = util::TSNPrintf(
        reinterpret_cast<char*>(out), outSize,
        Format0,
        sizeof(challenge.data), challenge.data,
        clientId,
        AuthenticationKeyGeneration,
        sizeof(sysVersion.data), sysVersion.data);
    NN_RESULT_THROW_UNLESS(inLength0 + (sizeof(Format1) - 1) + CodedMacSize < outSize, ResultInsufficientBuffer());
    std::strncpy(reinterpret_cast<char*>(out) + inLength0, Format1, (sizeof(Format1) - 1));
    auto inLength1 = inLength0 + (sizeof(Format1) - 1);

    // AES 鍵で MAC を生成する
    uint8_t cmac[crypto::Aes128CmacGenerator::MacSize] = {};
    NN_RESULT_DO(ComputeCmac(cmac, sizeof(cmac), out, inLength0, keySource));

    // エンコードして末尾に追加
    auto b64Error = util::Base64::ToBase64String(
        reinterpret_cast<char*>(out) + inLength1, outSize - inLength1,
        cmac, sizeof(cmac), util::Base64::Mode_UrlSafe);
    NN_ABORT_UNLESS(b64Error == util::Base64::Status_Success);
    NN_SDK_ASSERT(strnlen(out, outSize) < outSize);

    NN_RESULT_SUCCESS;
}

}}} // ~namespace nn::dauth::detail
