﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/dauth/detail/dauth_Result.h>
#include <nn/dauth/dauth_Result.h>
#include <nn/http/json/http_JsonErrorMap.h>
#include <nn/http/json/http_RapidJsonApi.h>
#include <nn/http/stream/http_ClientCertificate.h>
#include <nn/http/stream/http_CurlInputStream.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>

#include "dauth_AuthenticationAdaptor.h"
#include "dauth_NdasDriver.h"
#include "dauth_RequestParameter.h"

namespace nn { namespace dauth { namespace detail {
namespace {
#if defined(NN_BUILD_TARGET_PLATFORM_NX) || defined(NN_DAUTH_NETWORK_ENABLE)
#define NN_DAUTH_DENEB_ROOT "https://dauth-%.ndas.srv.nintendo.net"
const char DenebUrl[] = NN_DAUTH_DENEB_ROOT "/v3-59ed5fa1c25bb2aea8c4d73d74b919a94d89ed48d6865b728f63547943b17404/device_auth_token";
const char DenebChallengeUrl[] = NN_DAUTH_DENEB_ROOT "/v3-59ed5fa1c25bb2aea8c4d73d74b919a94d89ed48d6865b728f63547943b17404/challenge";
const char DenebEdgeTokenUrl[] = NN_DAUTH_DENEB_ROOT "/v3-59ed5fa1c25bb2aea8c4d73d74b919a94d89ed48d6865b728f63547943b17404/edge_token";
#undef NN_DAUTH_DENEB_ROOT
#else
const char DenebUrl[] = ""; // Ref. SIGLO-66856
const char DenebChallengeUrl[] = "";
const char DenebEdgeTokenUrl[] = "";
#endif

/**
 * @brief   NDAS要求通信用内部ワークバッファ構造
 */
const size_t IoBufferSizeMin = 1024u;
const size_t ClientCertSize = 4096u;
const int TokenLengthMax = (DeviceAuthenticationTokenLengthMax > EdgeTokenLengthMax ? DeviceAuthenticationTokenLengthMax : EdgeTokenLengthMax);
struct Buffer
{
    char postDataBuffer[256];
    union
    {
        struct
        {
            char clientCertBuffer[ClientCertSize];
            char stringBuffer[TokenLengthMax + 8];
            char inputBuffer[IoBufferSizeMin]; // 末尾に置くこと
        } io;
    }u;
};
NN_STATIC_ASSERT(sizeof(Buffer) <= NdasDriver::RequiredWorkBufferSize);
NN_STATIC_ASSERT(std::alignment_of<Buffer>::value <= std::alignment_of<std::max_align_t>::value);

/**
 * @brief   NDAS へのチャレンジレスポンス取得要求。
 */
Result GetKeySourceAndChallenge(
    KeySource* pOutKeySource, Challenge* pOutChallenge,
    CURL* curlHandle, void* workBuffer, size_t workBufferSize, const util::Cancelable* pCancelable) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(workBuffer) % std::alignment_of<std::max_align_t>::value == 0);
    NN_SDK_REQUIRES(workBufferSize >= sizeof(Buffer));
    auto& buffer = *reinterpret_cast<Buffer*>(workBuffer);

    // CurlInputStream 準備
    http::stream::CurlInputStream stream(curlHandle, pCancelable);
    NN_RESULT_DO(stream.Initialize());
    stream.SetStringBuffer(buffer.u.io.stringBuffer, sizeof(buffer.u.io.stringBuffer));
    stream.SetInputBuffer(buffer.u.io.inputBuffer, workBufferSize - offsetof(Buffer, u.io.inputBuffer));

    // POST データ作成
    static const char PostDataFormat[] = "key_generation=%u";
    auto l = util::SNPrintf(buffer.postDataBuffer, sizeof(buffer.postDataBuffer), PostDataFormat, AuthenticationKeyGeneration);
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(buffer.postDataBuffer));
    NN_UNUSED(l);
    NN_RESULT_DO(stream.SetHttpPost(buffer.postDataBuffer, false));

    // SSL / URL設定
    http::stream::CertIoBuffer certBuffer = {buffer.u.io.clientCertBuffer, sizeof(buffer.u.io.clientCertBuffer)};
    NN_RESULT_DO(stream.SetSslContextHandler(http::stream::SslCtxHandlerWithClientCert, &certBuffer));
    NN_RESULT_DO(stream.SetUrl(DenebChallengeUrl));

    // 通信の実行とレスポンスの適用
    ChallengeAdaptor adaptor;
    NN_RESULT_DO(stream.Open());
    NN_RESULT_DO(http::json::ImportJsonByRapidJson<http::json::DefaultJsonErrorMap>(adaptor, stream, pCancelable));
    stream.Close();
    NN_RESULT_DO(adaptor.Adapt());

    *pOutKeySource = adaptor.GetKeySourceRef();
    *pOutChallenge = adaptor.GetChallengeRef();
    NN_RESULT_SUCCESS;
}

/**
 * @brief   NDAS へのデバイス認証トークン取得要求。
 */

template <typename AdaptorType>
Result AuthenticateImpl(
    AdaptorType& adaptor,
    const char* url, uint64_t clientId, const KeySource& keySource, const Challenge& challenge,
    CURL* curlHandle, void* workBuffer, size_t workBufferSize, const util::Cancelable* pCancelable) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(workBuffer) % std::alignment_of<std::max_align_t>::value == 0);
    NN_SDK_REQUIRES(workBufferSize >= sizeof(Buffer));
    auto& buffer = *reinterpret_cast<Buffer*>(workBuffer);

    // CurlInputStream 準備
    http::stream::CurlInputStream stream(curlHandle, pCancelable);
    NN_RESULT_DO(stream.Initialize());
    stream.SetStringBuffer(buffer.u.io.stringBuffer, sizeof(buffer.u.io.stringBuffer));
    stream.SetInputBuffer(buffer.u.io.inputBuffer, workBufferSize - offsetof(Buffer, u.io.inputBuffer));

    // POST データ作成
    NN_RESULT_DO(detail::GeneratePostData(buffer.postDataBuffer, sizeof(buffer.postDataBuffer), clientId, challenge, keySource));
    auto postLength = strnlen(buffer.postDataBuffer, sizeof(buffer.postDataBuffer));
    NN_SDK_ASSERT(postLength < sizeof(buffer.postDataBuffer));
    NN_UNUSED(postLength);
    NN_RESULT_DO(stream.SetHttpPost(buffer.postDataBuffer, false));

    // SSL / URL設定
    http::stream::CertIoBuffer certBuffer = {buffer.u.io.clientCertBuffer, sizeof(buffer.u.io.clientCertBuffer)};
    NN_RESULT_DO(stream.SetSslContextHandler(http::stream::SslCtxHandlerWithClientCert, &certBuffer));
    NN_RESULT_DO(stream.SetUrl(url));

    // 通信の実行とデータの適用
    NN_RESULT_DO(stream.Open());
    NN_RESULT_DO(http::json::ImportJsonByRapidJson<http::json::DefaultJsonErrorMap>(adaptor, stream, pCancelable));
    stream.Close();
    NN_RESULT_SUCCESS;
}

Result AcquireDeviceAuthenticationTokenImpl(
    TimeSpan* pOutExpiration, int* pOutLength, char* tokenBuffer, size_t tokenBufferSize,
    uint64_t clientId, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const util::Cancelable* pCancelable) NN_NOEXCEPT
{
    KeySource keySource;
    Challenge challenge;
    NN_RESULT_DO(GetKeySourceAndChallenge(&keySource, &challenge, curlHandle, workBuffer, workBufferSize, pCancelable));

    DeviceAuthenticationAdaptor adaptor(tokenBuffer, tokenBufferSize);
    NN_RESULT_DO(AuthenticateImpl(
        adaptor,
        DenebUrl, clientId, keySource, challenge,
        curlHandle, workBuffer, workBufferSize, pCancelable));
    NN_RESULT_DO(adaptor.Adapt());

    *pOutExpiration = adaptor.GetExpiration();
    *pOutLength = adaptor.GetTokenLength();
    NN_RESULT_SUCCESS;
}

Result AcquireEdgeTokenImpl(
    TimeSpan* pOutExpiration, int* pOutLength, char* tokenBuffer, size_t tokenBufferSize,
    uint64_t clientId, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const util::Cancelable* pCancelable) NN_NOEXCEPT
{
    KeySource keySource;
    Challenge challenge;
    NN_RESULT_DO(GetKeySourceAndChallenge(&keySource, &challenge, curlHandle, workBuffer, workBufferSize, pCancelable));

    EdgeTokenAuthenticationAdaptor adaptor(tokenBuffer, tokenBufferSize);
    NN_RESULT_DO(AuthenticateImpl(
        adaptor,
        DenebEdgeTokenUrl, clientId, keySource, challenge,
        curlHandle, workBuffer, workBufferSize, pCancelable));
    NN_RESULT_DO(adaptor.Adapt());

    *pOutExpiration = adaptor.GetExpiration();
    *pOutLength = adaptor.GetTokenLength();
    NN_RESULT_SUCCESS;
}
} // ~namespace nn::dauth::detail::<anonymous>

Result NdasDriver::AcquireDeviceAuthenticationToken(
    TimeSpan* pOutExpiration, int* pOutLength, char* tokenBuffer, size_t tokenBufferSize,
    uint64_t clientId, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const util::Cancelable* pCancelable) NN_NOEXCEPT
{
    // 事前条件
    NN_SDK_REQUIRES_NOT_NULL(pOutExpiration);
    NN_SDK_REQUIRES_NOT_NULL(pOutLength);
    NN_SDK_REQUIRES_NOT_NULL(tokenBuffer);
    NN_SDK_REQUIRES(tokenBufferSize >= RequiredBufferSizeForDeviceAuthenticationToken);
    NN_SDK_REQUIRES_NOT_NULL(curlHandle);
    NN_SDK_REQUIRES_NOT_NULL(workBuffer);
    NN_SDK_REQUIRES(workBufferSize >= RequiredWorkBufferSize);

    NN_RESULT_TRY(AcquireDeviceAuthenticationTokenImpl(
        pOutExpiration, pOutLength, tokenBuffer, tokenBufferSize,
        clientId, curlHandle, workBuffer, workBufferSize, pCancelable))
        NN_RESULT_CATCH_CONVERT(http::ResultCancelled, ResultCancelled())
    NN_RESULT_END_TRY
    NN_RESULT_SUCCESS;
}

Result NdasDriver::AcquireEdgeToken(
    TimeSpan* pOutExpiration, int* pOutLength, char* tokenBuffer, size_t tokenBufferSize,
    uint64_t clientId, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const util::Cancelable* pCancelable) NN_NOEXCEPT
{
    // 事前条件
    NN_SDK_REQUIRES_NOT_NULL(pOutExpiration);
    NN_SDK_REQUIRES_NOT_NULL(pOutLength);
    NN_SDK_REQUIRES_NOT_NULL(tokenBuffer);
    NN_SDK_REQUIRES(tokenBufferSize >= RequiredBufferSizeForEdgeToken);
    NN_SDK_REQUIRES_NOT_NULL(curlHandle);
    NN_SDK_REQUIRES_NOT_NULL(workBuffer);
    NN_SDK_REQUIRES(workBufferSize >= RequiredWorkBufferSize);

    NN_RESULT_TRY(AcquireEdgeTokenImpl(
        pOutExpiration, pOutLength, tokenBuffer, tokenBufferSize,
        clientId, curlHandle, workBuffer, workBufferSize, pCancelable))
        NN_RESULT_CATCH_CONVERT(http::ResultCancelled, ResultCancelled())
    NN_RESULT_END_TRY
    NN_RESULT_SUCCESS;
}

}}} // ~namespace nn::dauth::detail
