﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "dauth_AsyncResult.h"

#include <nn/dauth/detail/dauth_Interface.sfdl.h>
#include <nn/sf/sf_ISharedObject.h>

namespace nn { namespace dauth { namespace detail {

AsyncResult::AsyncResult(detail::IAsyncResult* ptr) NN_NOEXCEPT
    : m_Ptr(ptr)
{
    NN_SDK_REQUIRES_NOT_NULL(m_Ptr);
}

AsyncResult::AsyncResult(AsyncResult&& rhs) NN_NOEXCEPT
    : m_Ptr(rhs.m_Ptr)
{
    rhs.m_Ptr = nullptr;
}

AsyncResult::~AsyncResult() NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsAvailable(), "Destroying during un-finished asynchronous function call");
    sf::ReleaseSharedObject(m_Ptr);
}

AsyncResult& AsyncResult::operator =(AsyncResult&& rhs) NN_NOEXCEPT
{
    AsyncResult tmp(std::move(rhs));
    swap(tmp, *this);
    return *this;
}

Result AsyncResult::GetResult() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Ptr);
    os::SystemEventType e;
    GetSharedSystemEvent(&e);
    os::WaitSystemEvent(&e);
    os::DestroySystemEvent(&e);
    return m_Ptr->GetResult();
}

void AsyncResult::Cancel() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Ptr);
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Ptr->Cancel());
}

bool AsyncResult::IsAvailable() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Ptr);
    bool isAvailable;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Ptr->IsAvailable(&isAvailable));
    return isAvailable;
}

void AsyncResult::GetSharedSystemEvent(os::SystemEventType* pEvent) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Ptr);
    sf::NativeHandle readable;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Ptr->GetSystemEvent(&readable));
    os::AttachReadableHandleToSystemEvent(pEvent, readable.GetOsHandle(), readable.IsManaged(), os::EventClearMode_ManualClear);
    readable.Detach();
}

void swap(AsyncResult& o0, AsyncResult& o1) NN_NOEXCEPT
{
    std::swap(o0.m_Ptr, o1.m_Ptr);
}

}}} // ~namespace nn::dauth::detail
