﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/dauth/dauth_Api.h>
#include <nn/dauth/dauth_ApiPrivate.h>

#include <cstring>
#include <mutex>

#include <nn/dauth/dauth_ServiceResource.h>
#include <nn/dauth/detail/dauth_Interface.sfdl.h>
#include <nn/dauth/detail/dauth_Result.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_HipcSimpleClientSessionManager.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/util/util_Optional.h>

#include "detail/dauth_AsyncResult.h"

namespace nn { namespace dauth {
namespace {
sf::SharedPointer<detail::IService> (*g_Initializer)() = nullptr;
os::SdkRecursiveMutexType g_Lock = NN_OS_SDK_RECURSIVE_MUTEX_INITIALIZER();
sf::SharedPointer<detail::IService> g_Ptr = nullptr;

sf::SharedPointer<detail::IService> GetServicePointerViaHipc() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    std::lock_guard<decltype(g_Lock)> lock(g_Lock);

    // セッションマネージャ
    NN_FUNCTION_LOCAL_STATIC(sf::HipcSimpleClientSessionManager, s_HipcDomain);

    // セッションアロケータ
    static const size_t SizeOfHeapToAcquireObject = 1024u;
    struct DauthServiceAllocatorTag {};
    typedef sf::ExpHeapStaticAllocator<SizeOfHeapToAcquireObject, DauthServiceAllocatorTag> Allocator;

    // HIPC 経由で取得したサービスオブジェクトのポインタ
    typedef sf::SharedPointer<detail::IService> ServicePointer;
    NN_FUNCTION_LOCAL_STATIC(ServicePointer, s_PtrHipc, = nullptr);

    if (!s_PtrHipc)
    {
        Allocator::Initialize(lmem::CreationOption_NoOption);
        NN_ABORT_UNLESS_RESULT_SUCCESS((s_HipcDomain.InitializeByName<detail::IService, Allocator::Policy>(&s_PtrHipc, ServiceName)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(s_HipcDomain.SetSessionCount(1));
    }
    return s_PtrHipc;

#elif defined(NN_BUILD_CONFIG_OS_WIN)
    return nullptr;
#else
#error "Unsupported OS specified"
#endif
}

sf::SharedPointer<detail::IService> GetServicePointer() NN_NOEXCEPT
{
    std::lock_guard<decltype(g_Lock)> lock(g_Lock);
    if (!g_Ptr)
    {
        g_Ptr = g_Initializer
            ? g_Initializer()
            : GetServicePointerViaHipc();
        NN_ABORT_UNLESS_NOT_NULL(g_Ptr);
    }
    return g_Ptr;
}

detail::AsyncResult EnsureAuthenticationTokenCacheAsync(uint64_t clientId, bool refresh) NN_NOEXCEPT
{
    sf::SharedPointer<detail::IAsyncResult> pAsync;
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetServicePointer()->EnsureAuthenticationTokenCacheAsync(&pAsync, clientId, refresh));
    return detail::AsyncResult(pAsync.Detach());
}

Result LoadAuthenticationTokenCache(TimeSpan* pOutExpiration, int* pOutLength, char* buffer, size_t bufferSize, uint64_t clientId) NN_NOEXCEPT
{
    int64_t expirationSec;
    NN_RESULT_DO(GetServicePointer()->LoadAuthenticationTokenCache(
        &expirationSec,
        sf::OutBuffer(buffer, bufferSize),
        clientId));

    *pOutExpiration = TimeSpan::FromSeconds(expirationSec);
    *pOutLength = static_cast<int>(strnlen(buffer, bufferSize));
    NN_RESULT_SUCCESS;
}

detail::AsyncResult EnsureEdgeTokenCacheAsync(uint64_t clientId) NN_NOEXCEPT
{
    sf::SharedPointer<detail::IAsyncResult> pAsync;
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetServicePointer()->EnsureEdgeTokenCacheAsync(&pAsync, clientId));
    return detail::AsyncResult(pAsync.Detach());
}

Result LoadEdgeTokenCache(TimeSpan* pOutExpiration, int* pOutLength, char* buffer, size_t bufferSize, uint64_t clientId) NN_NOEXCEPT
{
    int64_t expirationSec;
    NN_RESULT_DO(GetServicePointer()->LoadEdgeTokenCache(
        &expirationSec,
        sf::OutBuffer(buffer, bufferSize),
        clientId));

    *pOutExpiration = TimeSpan::FromSeconds(expirationSec);
    *pOutLength = static_cast<int>(strnlen(buffer, bufferSize));
    NN_RESULT_SUCCESS;
}
} // ~namespace nn::dauth::<anonymous>

//!-------------------------------------------------------------------------------
void InitializeWith(sf::SharedPointer<detail::IService>&& ptr) NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(sf::SharedPointer<detail::IService>, s_Ptr);

    s_Ptr = std::move(ptr);
    DebugSetServiceInitializer([]()->sf::SharedPointer<detail::IService>{ return s_Ptr; });
}

void DebugSetServiceInitializer(sf::SharedPointer<detail::IService> (*initializer)()) NN_NOEXCEPT
{
    std::lock_guard<decltype(g_Lock)> lock(g_Lock);
    g_Initializer = initializer;
}

void DebugResetServiceAccessor() NN_NOEXCEPT
{
    std::lock_guard<decltype(g_Lock)> lock(g_Lock);
    g_Ptr = nullptr;
}

//!-------------------------------------------------------------------------------

Result AcquireDeviceAuthenticationToken(
    TimeSpan* pOutExpiration, int* pOutTokenLength, char* buffer, size_t bufferSize,
    uint64_t clientId, bool ignoreCache,
    util::Cancelable* pCancelable) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(bufferSize >= RequiredBufferSizeForDeviceAuthenticationToken);
    NN_RESULT_DO(EnsureDeviceAuthenticationTokenCache(clientId, ignoreCache, pCancelable));
    NN_RESULT_DO(LoadAuthenticationTokenCache(pOutExpiration, pOutTokenLength, buffer, bufferSize, clientId));
    NN_RESULT_SUCCESS;
}

Result EnsureDeviceAuthenticationTokenCache(uint64_t clientId, bool refresh, util::Cancelable* pCancelable) NN_NOEXCEPT
{
    auto async = EnsureAuthenticationTokenCacheAsync(clientId, refresh);
    util::optional<detail::CancelAttachment> pListener;
    if (pCancelable)
    {
        pListener.emplace(*pCancelable, async);
    }
    return async.GetResult();
}

Result AcquireEdgeToken(
    TimeSpan* pOutExpiration, int* pOutTokenLength, char* buffer, size_t bufferSize,
    uint64_t clientId,
    util::Cancelable* pCancelable) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(bufferSize >= RequiredBufferSizeForEdgeToken);
    NN_RESULT_DO(EnsureEdgeTokenCache(clientId, pCancelable));
    NN_RESULT_DO(LoadEdgeTokenCache(pOutExpiration, pOutTokenLength, buffer, bufferSize, clientId));
    NN_RESULT_SUCCESS;
}

Result EnsureEdgeTokenCache(uint64_t clientId, util::Cancelable* pCancelable) NN_NOEXCEPT
{
    auto async = EnsureEdgeTokenCacheAsync(clientId);
    util::optional<detail::CancelAttachment> pListener;
    if (pCancelable)
    {
        pListener.emplace(*pCancelable, async);
    }
    return async.GetResult();
}

}} // ~namespace nn::dauth
