﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/i2c/i2c.h>
#include <nn/gpio/gpio.h>
#include <nn/cpad/cpad.h>

namespace nn {
namespace cpad {
namespace detail {

class Cpad
{
    NN_DISALLOW_COPY(Cpad);
    NN_DISALLOW_MOVE(Cpad);

public:
    Cpad()  NN_NOEXCEPT :
        m_IsCpadInitialized(false),
        m_IsDeviceInitialized(false)
        {
            m_StickOriginL.x = m_StickOriginR.x = OriginX;
            m_StickOriginL.y = m_StickOriginR.y = OriginY;
        }
    ~Cpad() NN_NOEXCEPT {}

    void   InitializeCpad() NN_NOEXCEPT;
    Result GetCpadState(CpadState* pOutState) NN_NOEXCEPT;

    bool   HasCpadChange(const CpadState *pInStates, int statesCount) const NN_NOEXCEPT;
    bool   HasCpadInput(CpadState state) const NN_NOEXCEPT;
    bool   CalibrateCpad() NN_NOEXCEPT;

private:
    Result MapCpadResult(nn::Result i2cError) NN_NOEXCEPT;
    Result GetCpadRegister(Bit8* pOutBuf, nn::Bit8 startRegister, size_t outBufSize) NN_NOEXCEPT;
    Result GetRawCpadState(CpadState* pOutState) NN_NOEXCEPT;
    Result InitializeDevice() NN_NOEXCEPT;
    void   ParseInput( CpadState* pOutState, const Bit8* pInputBuf, int dataFormat) const NN_NOEXCEPT;

private:
    nn::i2c::I2cSession        m_I2cSession;
    nn::gpio::GpioPadSession   m_GpioSession;
    nn::cpad::CpadStickState   m_StickOriginL;
    nn::cpad::CpadStickState   m_StickOriginR;

    bool                       m_IsCpadInitialized;
    bool                       m_IsDeviceInitialized;

    static const int DataFormat                          = 3;   // DataFormat3 を指定
    static const int DeviceId                            = 1;   // クラシックコントローラの device Id
    static const int ReadIntervalMicroSeconds            = 150; // 150 [us]
    static const int AnalogStickChangeDetectionThreshold = 2;   // アナログスティックの入力変化検知の閾値
    static const nn::i2c::I2cDevice CpadName             = nn::i2c::I2cDevice::I2cDevice_ClassicController;
    static const nn::i2c::TransactionOption InOption     = static_cast<nn::i2c::TransactionOption>( nn::i2c::TransactionOption_StartCondition | nn::i2c::TransactionOption_StopCondition );
    static const int OriginX                             = 128; // デフォルトの原点 X 座標
    static const int OriginY                             = 128; // デフォルトの原点 Y 座標
};

} // detail
} // cpad
} // nn
