﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/os/os_Event.h>

#include <nn/codec/server/codec_HardwareOpusDecoderHipcServer.h>
#include <nn/codec/server/codec_HardwareOpusDecoderManager.h>

#include "../detail/codec_HardwareOpusDecoderManagerImpl-spec.NX.h"

namespace nn { namespace codec { namespace server {

bool HardwareOpusDecoder::Hardware::g_IsInitialized(false);
bool HardwareOpusDecoder::Service::g_IsInitialized(false);
bool HardwareOpusDecoder::Service::g_IsListening(false);
bool HardwareOpusDecoder::Service::g_IsFinalizing(false);
bool HardwareOpusDecoder::Service::g_IsReadyForSleep(false);
nn::os::EventType HardwareOpusDecoder::Service::g_ActionWaiter;

/*------------------------------------------------------------------------
 * Hardware (Internal Resources)
 *------------------------------------------------------------------------*/
void HardwareOpusDecoder::Hardware::Initialize() NN_NOEXCEPT
{
    NN_SDK_ASSERT( !g_IsInitialized );
    if ( g_IsInitialized )
    {
        return ;
    }
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::codec::detail::HardwareOpusDecoderManagerImpl::InitializeAdspInternal());
#endif // defined(NN_BUILD_CONFIG_HARDWARE_NX)
    g_IsInitialized = true;
}

void HardwareOpusDecoder::Hardware::Finalize() NN_NOEXCEPT
{
    NN_SDK_ASSERT( g_IsInitialized );
    if ( !g_IsInitialized )
    {
        return ;
    }
    g_IsInitialized = false;
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::codec::detail::HardwareOpusDecoderManagerImpl::FinalizeAdspInternal());
#endif // defined(NN_BUILD_CONFIG_HARDWARE_NX)
}

/*------------------------------------------------------------------------
 * Service (External Interface)
 *------------------------------------------------------------------------*/
void HardwareOpusDecoder::Service::Initialize() NN_NOEXCEPT
{
    NN_SDK_ASSERT( !g_IsInitialized );
    if ( g_IsInitialized )
    {
        return ;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::codec::detail::HardwareOpusDecoderManagerImpl::InitializeInternal());
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    InitializeHardwareOpusDecoderServer();
    StartHardwareOpusDecoderServer();
#else
#if defined(NN_BUILD_CONFIG_OS_WIN32)
    // ToDo
#else
#error Define any OS.
#endif // defined(NN_BUILD_CONFIG_OS_WIN32)
#endif // defined(NN_BUILD_CONFIG_OS_HORIZON)
    nn::os::InitializeEvent(&g_ActionWaiter, false, nn::os::EventClearMode_AutoClear);

    g_IsInitialized = true;
}

void HardwareOpusDecoder::Service::DoListening() NN_NOEXCEPT
{
    NN_SDK_ASSERT( g_IsInitialized );
    NN_SDK_ASSERT( !g_IsListening );
    if ( g_IsListening )
    {
        return ;
    }
    g_IsListening = true;
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    LoopHardwareOpusDecoderServer();
#else
#if defined(NN_BUILD_CONFIG_OS_WIN32)
    // ToDo
#else
#error Define any OS.
#endif // defined(NN_BUILD_CONFIG_OS_WIN32)
#endif // defined(NN_BUILD_CONFIG_OS_HORIZON)
}

void HardwareOpusDecoder::Service::RequestStopListening() NN_NOEXCEPT
{
    NN_SDK_ASSERT( g_IsInitialized );
    NN_SDK_ASSERT( g_IsListening );
    if ( !g_IsListening )
    {
        return ;
    }
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    RequestStopHardwareOpusDecoderServer();
#else
#if defined(NN_BUILD_CONFIG_OS_WIN32)
    // ToDo
#else
#error Define any OS.
#endif // defined(NN_BUILD_CONFIG_OS_WIN32)
#endif // defined(NN_BUILD_CONFIG_OS_HORIZON)
    g_IsListening = false;
}

void HardwareOpusDecoder::Service::Finalize() NN_NOEXCEPT
{
    NN_SDK_ASSERT( g_IsInitialized );
    if ( !g_IsInitialized )
    {
        return ;
    }
    g_IsInitialized = false;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::codec::detail::HardwareOpusDecoderManagerImpl::FinalizeInternal());
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    FinalizeHardwareOpusDecoderServer();
#else
#if defined(NN_BUILD_CONFIG_OS_WIN32)
    // ToDo
#else
#error Define any OS.
#endif // defined(NN_BUILD_CONFIG_OS_WIN32)
#endif // defined(NN_BUILD_CONFIG_OS_HORIZON)
}

void HardwareOpusDecoder::Service::Exit() NN_NOEXCEPT
{
    RequestStopListening();
    // 終了フラグを立てて先に進める。
    g_IsFinalizing = true;
    nn::os::SignalEvent(&g_ActionWaiter);
}

void HardwareOpusDecoder::Service::Sleep() NN_NOEXCEPT
{
    if (g_IsReadyForSleep)
    {
        return ;
    }
    RequestStopListening();
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::codec::detail::HardwareOpusDecoderManagerImpl::SleepInternal());
#endif // defined(NN_BUILD_CONFIG_HARDWARE_NX)
    g_IsReadyForSleep = true;
}

void HardwareOpusDecoder::Service::Wait() NN_NOEXCEPT
{
    nn::os::WaitEvent(&g_ActionWaiter);
}

void HardwareOpusDecoder::Service::Wake() NN_NOEXCEPT
{
    if (!g_IsReadyForSleep)
    {
        return ;
    }
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::codec::detail::HardwareOpusDecoderManagerImpl::WakeInternal());
    StartHardwareOpusDecoderServer();
#endif // defined(NN_BUILD_CONFIG_HARDWARE_NX)
    nn::os::SignalEvent(&g_ActionWaiter);
    g_IsReadyForSleep = false;
}

}}} // namespace nn::codec::server
