﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <cerrno>

#include <nn/nn_SdkAssert.h>
#include <nn/codec/detail/codec_Log.h>
#include <nn/result/result_ErrorResult.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_BitUtil.h>

#include <nn/dd.h>
#include <nn/codec/codec_OpusCommon.h>
#include <nn/codec/detail/codec_HardwareOpusCommon.h>
#include <nn/codec/codec_Result.private.h>

#include <nne/audio/audio_Iova.h>

#include "opus.h"
#include "codec_HardwareOpusDecoderImpl-spec.NX.h"
#include "codec_HardwareOpusDecoderManagerImpl-spec.NX.h"

#define DEBUG_LOG(...)

#ifndef DEBUG_LOG
#define DEBUG_LOG(...) NN_DETAIL_CODEC_TRACE("[HW OPUS] " __VA_ARGS__)
#endif

namespace nn {
namespace codec {
namespace detail {

namespace {

    const int ChannelCountMaximum = 2;   // stereo

    const uint MailboxErrorCountLimitOnSending = 1024;   // TBD
    const uint MailboxErrorCountLimitOnReceiving = 1024; // TBD
    const int64_t MailboxIntervalOnSending = 1000;       // in msec (TBD)
    const int64_t MailboxIntervalOnSendingInternal = 1;  // in msec (TBD)
    const int64_t MailboxIntervalOnReceiving = 1000;     // in msec (TBD)

    NN_DD_ALIGNAS_DEVICE_ADDRESS_SPACE_MEMORY nn::codec::dsp::SharedMemory g_SharedMemoryArea;

    nn::codec::dsp::SharedMemory* GetSharedMemory() NN_NOEXCEPT
    {
        return &g_SharedMemoryArea;
    }

    size_t GetAlignedSize(size_t size) NN_NOEXCEPT
    {
        const size_t alignedSize = static_cast<size_t>(nn::dd::DeviceAddressSpaceMemoryRegionAlignment);
        return (size + alignedSize - 1) & ~(alignedSize - 1);
    }

    nn::Result ConvertOpusErrorsToNnResults(int errorCode) NN_NOEXCEPT
    {
        switch(errorCode)
        {
        case OPUS_OK:
            {
                NN_RESULT_SUCCESS;
                break;
            }
        case OPUS_BAD_ARG:
            {
                return ResultOpusInvalidArguments();
            }
            break;
        case OPUS_BUFFER_TOO_SMALL:
            {
                return ResultOpusBufferTooSmall();
            }
            break;
        case OPUS_INTERNAL_ERROR:
            {
                return ResultOpusInternalError();
            }
            break;
        case OPUS_INVALID_PACKET:
            {
                return ResultOpusInvalidPacket();
            }
            break;
        case OPUS_UNIMPLEMENTED:
            {
                return ResultOpusUnimplemented();
            }
            break;
        case OPUS_INVALID_STATE:
            {
                return ResultOpusInvalidState();
            }
            break;
        case OPUS_ALLOC_FAIL:
            {
                return ResultOpusAllocationFailure();
            }
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
        return ResultOpusUnexpectedError();
    }

    uint32_t ConvertSmallerTypeFromAddress(const void* address) NN_NOEXCEPT
    {
        return static_cast<uint32_t>(reinterpret_cast<uintptr_t>(address));
    }

} // anonymous

// 静的クラスの非公開変数
nn::os::Mutex HardwareOpusDecoderManagerImpl::m_Mutex(false);
dsp::MboxWrapper HardwareOpusDecoderManagerImpl::m_Mailbox;
DeviceBuffer HardwareOpusDecoderManagerImpl::m_SharedMemory;
size_t       HardwareOpusDecoderManagerImpl::m_RequiredWorkBufferSize[2] = { 0, };
uint         HardwareOpusDecoderManagerImpl::m_ErrorCountOfSending(0);
uint         HardwareOpusDecoderManagerImpl::m_ErrorCountOfReceiving(0);
bool         HardwareOpusDecoderManagerImpl::m_IsAdspInitialized(false);
bool         HardwareOpusDecoderManagerImpl::m_IsInitialized(false);
std::array<HardwareOpusDecoderImpl*, HardwareOpusDecoderManagerImpl::DecoderCountMax> HardwareOpusDecoderManagerImpl::m_DecoderList;

nn::Result HardwareOpusDecoderManagerImpl::InitializeInternal() NN_NOEXCEPT
{
    if ( IsInitialized() )
    {
        NN_RESULT_SUCCESS;
    }
    for (auto pDecoder = m_DecoderList.begin(); pDecoder != m_DecoderList.end(); ++pDecoder)
    {
        *pDecoder = nullptr;
    }
    m_IsInitialized = true;
    NN_RESULT_SUCCESS;
}

nn::Result HardwareOpusDecoderManagerImpl::FinalizeInternal() NN_NOEXCEPT
{
    if ( !IsInitialized() )
    {
        NN_RESULT_SUCCESS;
    }
    m_IsInitialized = false;
    NN_RESULT_SUCCESS;
}

nn::Result HardwareOpusDecoderManagerImpl::InitializeAdspInternal() NN_NOEXCEPT
{
    if (IsAdspInitialized())
    {
        NN_RESULT_SUCCESS;
    }
    if ( false == m_Mailbox.Open(AdspMailboxId_HardwareOpus_BidirectionalPath) )
    {
        NN_RESULT_THROW(ResultIpcCannotOpen());
    }
    DEBUG_LOG("MailboxId=%d\n", m_Mailbox.GetMailboxId());

    NN_UTIL_SCOPE_EXIT
    {
        if (!IsAdspInitialized())
        {
            m_Mailbox.Close();
            if ( m_SharedMemory.IsAvailable() )
            {
                nne::audio::iova::Unmap(
                    m_SharedMemory.GetVirtualAddress() ,
                    m_SharedMemory.GetSize() ,
                    m_SharedMemory.GetDeviceAddress());
            }
        }
    };
    NN_SDK_ASSERT( m_Mailbox.GetLastError() == dsp::MboxWrapper::MailboxStatusNoError ); // TBD
    // 共有メモリを SMMU に設定する。
    // SMMU への Map は、プロセスの起動時に一回だけ実施すれば良いので、Sleep/Wake の度に呼ばれる
    // InitializeAdspInternal() とは別の関数に配置したい。
    // しかし、適切な箇所がないのでここに配置しておく。
    // ToDo:
    //     可能なら、以下の処理ブロックを InitializeAdspInternal() 外に配置する。
    //     付随して、FinalizeAdspInternal() からも、DeviceSpace 関連の処理を追い出す。
    {
        const size_t alignedSize = GetAlignedSize(sizeof(g_SharedMemoryArea));
        const uint32_t deviceAddress = MapMemoryOnSmmu(&g_SharedMemoryArea, alignedSize);
        NN_SDK_ASSERT_ALIGNED(alignedSize, dsp::OneCacheLineExpected);
        NN_SDK_ASSERT_ALIGNED(deviceAddress, dsp::OneCacheLineExpected);
        NN_SDK_ASSERT_ALIGNED(deviceAddress, nn::dd::DeviceAddressSpaceMemoryRegionAlignment);

        m_SharedMemory.SetParameters(&g_SharedMemoryArea, deviceAddress, alignedSize);
        DEBUG_LOG("Output buffer: %p, %08llx, %d\n",
            m_SharedMemory.GetVirtualAddress(), m_SharedMemory.GetDeviceAddress(), m_SharedMemory.GetSize());
        memset(&g_SharedMemoryArea, 0x0, sizeof(g_SharedMemoryArea));
        nn::dd::StoreDataCache(&g_SharedMemoryArea, sizeof(g_SharedMemoryArea));
    }
    //----------------------------------------------------------------------------------------------
    // 送信 (ADSP 側でスレッド生成)
    NN_RESULT_DO(TryingSendMessage(OpusManagerCommand_Start, MailboxIntervalOnSending));
    // 共有メモリのデバイスアドレス
    uint32_t address = m_SharedMemory.GetDeviceAddress();
    uint32_t msg = (address >> 16) & 0xffff;
    NN_RESULT_DO(TryingSendMessage(msg, MailboxIntervalOnSending));
    msg = address & 0xffff;
    NN_RESULT_DO(TryingSendMessage(msg, MailboxIntervalOnSending));

    // 受信
    uint32_t message = 0;
    NN_RESULT_DO(WaitForMessage(&message, MailboxIntervalOnReceiving));
    DEBUG_LOG("Finish: status=%08x\n", message);
    NN_RESULT_THROW_UNLESS(message == OpusManagerResponse_Start, ResultCoprocessorUnexpectedResponse());
    nn::dd::InvalidateDataCache(GetSharedMemory()->from, sizeof(nn::codec::dsp::ArgumentType) * nn::codec::dsp::ArgumentCount);
    NN_RESULT_THROW_UNLESS(GetSharedMemory()->from[0] == OpusManagerResult_Success, ResultCoprocessorInitializationFailure());

    // デコードに必要なワークサイズを取得する。
    for ( int channel = 0 ; channel < ChannelCountMaximum; ++channel )
    {
        NN_RESULT_DO(GetWorkBufferSizeInternal( &m_RequiredWorkBufferSize[channel], channel + 1 ));
    }
    m_IsAdspInitialized = true;

    NN_RESULT_SUCCESS;
}

nn::Result HardwareOpusDecoderManagerImpl::FinalizeAdspInternal() NN_NOEXCEPT
{
    if (!IsAdspInitialized())
    {
        NN_RESULT_SUCCESS;
    }
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    m_IsAdspInitialized = false;
    NN_RESULT_DO(TryingSendMessage(OpusManagerCommand_Exit, MailboxIntervalOnSending));
    uint32_t message = 0;
    NN_RESULT_DO(WaitForMessage(&message, MailboxIntervalOnReceiving));
    DEBUG_LOG(" Finish: status=%08x\n", message);
    NN_RESULT_THROW_UNLESS(message == OpusManagerResponse_Exit, ResultCoprocessorUnexpectedResponse());
    nn::dd::InvalidateDataCache(GetSharedMemory()->from, sizeof(nn::codec::dsp::ArgumentType) * nn::codec::dsp::ArgumentCount);
    NN_RESULT_THROW_UNLESS(GetSharedMemory()->from[0] == OpusManagerResult_Success, ResultCoprocessorFinalizationFailure());

    m_Mailbox.Close();
    // Unmap
    nne::audio::iova::Unmap(
        m_SharedMemory.GetVirtualAddress() ,
        m_SharedMemory.GetSize() ,
        m_SharedMemory.GetDeviceAddress());

    NN_RESULT_SUCCESS;
}

nn::Result HardwareOpusDecoderManagerImpl::GetWorkBufferSizeInternal(size_t* pSize, int channelCount) NN_NOEXCEPT
{
    DEBUG_LOG("Send message: %d\n", OpusDecoderCommand_GetWorkSize);
    NN_RESULT_DO(TryingSendMessage(OpusDecoderCommand_GetWorkSize, MailboxIntervalOnSending)); // 問い合わせ
    NN_RESULT_DO(TryingSendMessage(channelCount, MailboxIntervalOnSending)); // 引数にチャンネル数を与える

    uint32_t msg = 0;
    DEBUG_LOG("Waiting for a response from ADSP ...\n");
    NN_RESULT_DO(WaitForMessage(&msg, MailboxIntervalOnReceiving));
    NN_RESULT_THROW_UNLESS(msg == OpusDecoderResponse_GetWorkSize, ResultCoprocessorUnexpectedResponse());
    // サイズを受け取る。
    nn::dd::InvalidateDataCache(GetSharedMemory()->from, sizeof(nn::codec::dsp::ArgumentType) * nn::codec::dsp::ArgumentCount);
    const auto size = static_cast<std::size_t>(GetSharedMemory()->from[0]);
    NN_RESULT_THROW_UNLESS(size > 0, ResultCoprocessorUnexpectedReturnValue());
    DEBUG_LOG("WorkSize=%d Byte (channelCount=%d)\n", size, channelCount);

    *pSize = size;

    NN_RESULT_SUCCESS;
}

nn::Result HardwareOpusDecoderManagerImpl::SleepInternal() NN_NOEXCEPT
{
    for (auto pDecoder = m_DecoderList.begin(); pDecoder != m_DecoderList.end(); ++pDecoder)
    {
        if (nullptr != *pDecoder)
        {
            (*pDecoder)->Sleep();
        }
    }
    NN_RESULT_SUCCESS;
}

nn::Result HardwareOpusDecoderManagerImpl::WakeInternal() NN_NOEXCEPT
{
    for (auto pDecoder = m_DecoderList.begin(); pDecoder != m_DecoderList.end(); ++pDecoder)
    {
        if (nullptr != *pDecoder)
        {
            (*pDecoder)->Wake();
        }
    }
    NN_RESULT_SUCCESS;
}

nn::Result HardwareOpusDecoderManagerImpl::Initialize(
    int sampleRate,
    int channelCount,
    uintptr_t workBufferAddress,
    size_t workBufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsInitialized() );

    // ToDo: lock を外せるか、要検討
    //   * lock を追加せずともテストはパス (2016/6/3 時点)
    //     ==> ADSP 側がシングルスレッドでメッセージを受け付けており、CPU 側でのメールボックス送信が
    //         たまたま期待通りの順序で処理されているため、だと思われる。
    //   * たとえばメールボックスの仕組みの変更、本ソースファイルのリファクタリングなどで、
    //     たまたま保たれている順序が崩れる可能性がある (特に前者は制御できない)。
    //   * よって、安全側 (lock_guard 有) に倒しておく。
    //   * Finalize()、DecodeInterleaved() の lock_guard も検討対象
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    GetSharedMemory()->to[0] = workBufferAddress; // ワーク領域のデバイスアドレス (必ず送る)
    GetSharedMemory()->to[1] = workBufferSize;    // ワーク領域サイズ
    GetSharedMemory()->to[2] = sampleRate;        // 標本化周波数とチャンネル数
    GetSharedMemory()->to[3] = channelCount;      // チャンネル数
    nn::dd::StoreDataCache(GetSharedMemory()->to, sizeof(nn::codec::dsp::ArgumentType) * nn::codec::dsp::ArgumentCount);
    // 初期化要求
    NN_RESULT_DO(TryingSendMessage(OpusDecoderCommand_Initialize, MailboxIntervalOnSending));

    // 応答を待つ。
    uint32_t resp = 0;
    NN_RESULT_DO(WaitForMessage(&resp, MailboxIntervalOnReceiving));
    NN_RESULT_THROW_UNLESS(resp == OpusDecoderResponse_Initialize, ResultCoprocessorUnexpectedResponse());
    nn::dd::InvalidateDataCache(GetSharedMemory()->from, sizeof(nn::codec::dsp::ArgumentType) * nn::codec::dsp::ArgumentCount);
    return ConvertOpusErrorsToNnResults(GetSharedMemory()->from[0]);
}

nn::Result HardwareOpusDecoderManagerImpl::Finalize(
    uintptr_t workBufferAddress,
    size_t workBufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsInitialized() );

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    GetSharedMemory()->to[0] = workBufferAddress; // ワーク領域のデバイスアドレス (必ず送る)
    GetSharedMemory()->to[1] = workBufferSize;    // ワーク領域サイズ
    nn::dd::StoreDataCache(GetSharedMemory()->to, sizeof(nn::codec::dsp::ArgumentType) * nn::codec::dsp::ArgumentCount);
    NN_RESULT_DO(TryingSendMessage(OpusDecoderCommand_Finalize, MailboxIntervalOnSending));
    // 応答を待つ。
    uint32_t resp = 0;
    NN_RESULT_DO(WaitForMessage(&resp, MailboxIntervalOnReceiving));
    NN_RESULT_THROW_UNLESS(resp == OpusDecoderResponse_Finalize, ResultCoprocessorUnexpectedResponse());
    nn::dd::InvalidateDataCache(GetSharedMemory()->from, sizeof(nn::codec::dsp::ArgumentType) * nn::codec::dsp::ArgumentCount);
    return ConvertOpusErrorsToNnResults(GetSharedMemory()->from[0]);
}

size_t HardwareOpusDecoderManagerImpl::GetWorkBufferSize(int channelCount) NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsInitialized() );
    size_t size = 0;
    if ( 0 < channelCount && channelCount <= ChannelCountMaximum )
    {
        size = m_RequiredWorkBufferSize[channelCount - 1];
    }
    return size;
}

nn::Result HardwareOpusDecoderManagerImpl::DecodeInterleaved(
    int* pOutputSampleCount,
    int16_t* outputBuffer,
    size_t outputBufferSize,
    int channelCount,
    const uint8_t* inputBuffer,
    size_t inputBufferSize,
    uintptr_t workBufferAddress,
    nn::TimeSpan* pPerf,
    bool isResetRequested) NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsInitialized() );

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    GetSharedMemory()->to[0] = workBufferAddress;                            // ワーク領域のデバイスアドレス (必ず送る)
    GetSharedMemory()->to[1] = ConvertSmallerTypeFromAddress(inputBuffer);   // 入力バッファのデバイスアドレス
    GetSharedMemory()->to[2] = inputBufferSize;                              // 入力パケットのサイズ
    GetSharedMemory()->to[3] = ConvertSmallerTypeFromAddress(outputBuffer) ; // 出力バッファのデバイスアドレス
    GetSharedMemory()->to[4] = outputBufferSize;                             // 出力バッファのサイズ
    GetSharedMemory()->to[5] = 0;                                            // FinalRange (Not in use)
    GetSharedMemory()->to[6] = isResetRequested ? 1u : 0u;                   // デコーダコンテキストのリセット要求

    nn::dd::StoreDataCache(GetSharedMemory()->to, sizeof(nn::codec::dsp::ArgumentType) * nn::codec::dsp::ArgumentCount);
    // デコード要求を送信
    NN_RESULT_DO(TryingSendMessage(OpusDecoderCommand_DecodeOneFrame, MailboxIntervalOnSending));

    // 応答を待つ。
    uint32_t response = 0;
    NN_RESULT_DO(WaitForMessage(&response, MailboxIntervalOnReceiving));
    NN_RESULT_THROW_UNLESS(response == OpusDecoderResponse_DecodeOneFrame, ResultCoprocessorUnexpectedResponse());

    nn::dd::InvalidateDataCache(GetSharedMemory()->from, sizeof(nn::codec::dsp::ArgumentType) * nn::codec::dsp::ArgumentCount);
    const int outputSampleCount = GetSharedMemory()->from[1];
    const int status = GetSharedMemory()->from[0];
    const auto perf = nn::TimeSpan::FromMicroSeconds(GetSharedMemory()->from[2]);
    DEBUG_LOG("status=%d\n", status);
    DEBUG_LOG("sample=%d\n", outputSampleCount);
    DEBUG_LOG("perf=%lld [us]\n", perf.GetMicroSeconds());
    NN_RESULT_DO(ConvertOpusErrorsToNnResults(status));
    // outputSampleCount > 0 でないなら OPUS_OK 以外のステートが返ってきており、
    // 上の NN_RESULT_DO で result を返しているはず。
    NN_SDK_ASSERT( outputSampleCount > 0, "[codec] error: outputSampleCount = %d\n", outputSampleCount );

    *pOutputSampleCount = outputSampleCount;
    if (pPerf)
    {
        *pPerf = perf;
    }

    NN_RESULT_SUCCESS;
}

nn::Result HardwareOpusDecoderManagerImpl::InitializeForMultiStream(
    int sampleRate,
    int channelCount,
    int totalStreamCount,
    int stereoStreamCount,
    const uint8_t channelMapping[],
    uintptr_t workBufferAddress,
    size_t workBufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsInitialized() );

    // ToDo: lock を外せるか、要検討
    //   * lock を追加せずともテストはパス (2016/6/3 時点)
    //     ==> ADSP 側がシングルスレッドでメッセージを受け付けており、CPU 側でのメールボックス送信が
    //         たまたま期待通りの順序で処理されているため、だと思われる。
    //   * たとえばメールボックスの仕組みの変更、本ソースファイルのリファクタリングなどで、
    //     たまたま保たれている順序が崩れる可能性がある (特に前者は制御できない)。
    //   * よって、安全側 (lock_guard 有) に倒しておく。
    //   * Finalize()、DecodeInterleaved() の lock_guard も検討対象
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    GetSharedMemory()->to[0] = workBufferAddress; // ワーク領域のデバイスアドレス (必ず送る)
    GetSharedMemory()->to[1] = workBufferSize;    // ワーク領域サイズ
    GetSharedMemory()->to[2] = sampleRate;        // 標本化周波数とチャンネル数
    GetSharedMemory()->to[3] = channelCount;      // チャンネル数
    GetSharedMemory()->to[4] = totalStreamCount;
    GetSharedMemory()->to[5] = stereoStreamCount;
    {
        auto map = reinterpret_cast<uint8_t*>(&GetSharedMemory()->to[6]);
        for (auto i = 0; i < channelCount; ++i)
        {
            map[i] = channelMapping[i];
        }
    }
    nn::dd::StoreDataCache(GetSharedMemory()->to, sizeof(nn::codec::dsp::ArgumentType) * nn::codec::dsp::ArgumentCount);
    // 初期化要求
    NN_RESULT_DO(TryingSendMessage(OpusDecoderCommand_InitializeForMultiStream, MailboxIntervalOnSending));

    // 応答を待つ。
    uint32_t resp = 0;
    NN_RESULT_DO(WaitForMessage(&resp, MailboxIntervalOnReceiving));
    NN_RESULT_THROW_UNLESS(resp == OpusDecoderResponse_InitializeForMultiStream, ResultCoprocessorUnexpectedResponse());
    nn::dd::InvalidateDataCache(GetSharedMemory()->from, sizeof(nn::codec::dsp::ArgumentType) * nn::codec::dsp::ArgumentCount);
    return ConvertOpusErrorsToNnResults(GetSharedMemory()->from[0]);
}

nn::Result HardwareOpusDecoderManagerImpl::FinalizeForMultiStream(
    uintptr_t workBufferAddress,
    size_t workBufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsInitialized() );

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    GetSharedMemory()->to[0] = workBufferAddress; // ワーク領域のデバイスアドレス (必ず送る)
    GetSharedMemory()->to[1] = workBufferSize;    // ワーク領域サイズ
    nn::dd::StoreDataCache(GetSharedMemory()->to, sizeof(nn::codec::dsp::ArgumentType) * nn::codec::dsp::ArgumentCount);
    NN_RESULT_DO(TryingSendMessage(OpusDecoderCommand_FinalizeForMultiStream, MailboxIntervalOnSending));
    // 応答を待つ。
    uint32_t resp = 0;
    NN_RESULT_DO(WaitForMessage(&resp, MailboxIntervalOnReceiving));
    NN_RESULT_THROW_UNLESS(resp == OpusDecoderResponse_FinalizeForMultiStream, ResultCoprocessorUnexpectedResponse());
    nn::dd::InvalidateDataCache(GetSharedMemory()->from, sizeof(nn::codec::dsp::ArgumentType) * nn::codec::dsp::ArgumentCount);
    return ConvertOpusErrorsToNnResults(GetSharedMemory()->from[0]);
}

nn::Result HardwareOpusDecoderManagerImpl::DecodeInterleavedForMultiStream(
    int* pOutputSampleCount,
    int16_t* outputBuffer,
    size_t outputBufferSize,
    int channelCount,
    const uint8_t* inputBuffer,
    size_t inputBufferSize,
    uintptr_t workBufferAddress,
    nn::TimeSpan* pPerf,
    bool isResetRequested) NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsInitialized() );

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    GetSharedMemory()->to[0] = workBufferAddress;                            // ワーク領域のデバイスアドレス (必ず送る)
    GetSharedMemory()->to[1] = ConvertSmallerTypeFromAddress(inputBuffer);   // 入力バッファのデバイスアドレス
    GetSharedMemory()->to[2] = inputBufferSize;                              // 入力パケットのサイズ
    GetSharedMemory()->to[3] = ConvertSmallerTypeFromAddress(outputBuffer) ; // 出力バッファのデバイスアドレス
    GetSharedMemory()->to[4] = outputBufferSize;                             // 出力バッファのサイズ
    GetSharedMemory()->to[5] = 0;                                            // FinalRange (Not in use)
    GetSharedMemory()->to[6] = isResetRequested ? 1u : 0u;                   // デコーダコンテキストのリセット要求

    nn::dd::StoreDataCache(GetSharedMemory()->to, sizeof(nn::codec::dsp::ArgumentType) * nn::codec::dsp::ArgumentCount);
    // デコード要求を送信
    NN_RESULT_DO(TryingSendMessage(OpusDecoderCommand_DecodeOneFrameForMultiStream, MailboxIntervalOnSending));

    // 応答を待つ。
    uint32_t response = 0;
    NN_RESULT_DO(WaitForMessage(&response, MailboxIntervalOnReceiving));
    NN_RESULT_THROW_UNLESS(response == OpusDecoderResponse_DecodeOneFrameForMultiStream, ResultCoprocessorUnexpectedResponse());

    nn::dd::InvalidateDataCache(GetSharedMemory()->from, sizeof(nn::codec::dsp::ArgumentType) * nn::codec::dsp::ArgumentCount);
    const int outputSampleCount = GetSharedMemory()->from[1];
    const int status = GetSharedMemory()->from[0];
    const auto perf = nn::TimeSpan::FromMicroSeconds(GetSharedMemory()->from[2]);
    DEBUG_LOG("status=%d\n", status);
    DEBUG_LOG("sample=%d\n", outputSampleCount);
    DEBUG_LOG("perf=%lld [us]\n", perf.GetMicroSeconds());
    NN_RESULT_DO(ConvertOpusErrorsToNnResults(status));
    // outputSampleCount > 0 でないなら OPUS_OK 以外のステートが返ってきており、
    // 上の NN_RESULT_DO で result を返しているはず。
    NN_SDK_ASSERT( outputSampleCount > 0, "[codec] error: outputSampleCount = %d\n", outputSampleCount );

    *pOutputSampleCount = outputSampleCount;
    if (pPerf)
    {
        *pPerf = perf;
    }


    NN_RESULT_SUCCESS;
}

nn::Result HardwareOpusDecoderManagerImpl::GetWorkBufferSizeForMultiStreamInternal(size_t* pSize, int totalStreamCount, int stereoStreamCount) NN_NOEXCEPT
{
    DEBUG_LOG("Send message: %d\n", OpusDecoderCommand_GetWorkSizeForMultiStream);

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    GetSharedMemory()->to[0] = totalStreamCount;
    GetSharedMemory()->to[1] = stereoStreamCount;

    nn::dd::StoreDataCache(GetSharedMemory()->to, sizeof(nn::codec::dsp::ArgumentType) * nn::codec::dsp::ArgumentCount);

    NN_RESULT_DO(TryingSendMessage(OpusDecoderCommand_GetWorkSizeForMultiStream, MailboxIntervalOnSending)); // 問い合わせ

    uint32_t msg = 0;
    DEBUG_LOG("Waiting for a response from ADSP ...\n");
    NN_RESULT_DO(WaitForMessage(&msg, MailboxIntervalOnReceiving));
    NN_RESULT_THROW_UNLESS(msg == OpusDecoderResponse_GetWorkSizeForMultiStream, ResultCoprocessorUnexpectedResponse());
    // サイズを受け取る。
    nn::dd::InvalidateDataCache(GetSharedMemory()->from, sizeof(nn::codec::dsp::ArgumentType) * nn::codec::dsp::ArgumentCount);
    const auto size = static_cast<std::size_t>(GetSharedMemory()->from[0]);
    NN_RESULT_THROW_UNLESS(size > 0, ResultCoprocessorUnexpectedReturnValue());
    DEBUG_LOG("WorkSize=%d Byte (totalStreamCount=%d, stereoStreamCount=%d)\n", size, totalStreamCount, stereoStreamCount);

    *pSize = size;

    NN_RESULT_SUCCESS;
}

size_t HardwareOpusDecoderManagerImpl::GetWorkBufferSizeForMultiStream(int totalStreamCount, int stereoStreamCount) NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsInitialized() );

    // TODO: 初期化時にキャッシュする？
    size_t size = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetWorkBufferSizeForMultiStreamInternal(&size, totalStreamCount, stereoStreamCount));

    return size;
}

bool HardwareOpusDecoderManagerImpl::RegisterDecoder(const HardwareOpusDecoderImpl* decoder) NN_NOEXCEPT
{
    for (auto pDecoder = m_DecoderList.begin(); pDecoder != m_DecoderList.end(); ++pDecoder)
    {
        if (nullptr == *pDecoder)
        {
            *pDecoder = const_cast<HardwareOpusDecoderImpl*>(decoder);
            return true;
        }
    }
    return false;
}

void HardwareOpusDecoderManagerImpl::UnregisterDecoder(const HardwareOpusDecoderImpl* decoder) NN_NOEXCEPT
{
    for (auto pDecoder = m_DecoderList.begin(); pDecoder != m_DecoderList.end(); ++pDecoder)
    {
        if(decoder == *pDecoder)
        {
            *pDecoder = nullptr;
            return;
        }
    }
    NN_SDK_ASSERT(true);
}

uint32_t HardwareOpusDecoderManagerImpl::MapMemoryOnSmmu(
    const void* virtualAddress,
    size_t size) NN_NOEXCEPT
{
    NN_SDK_ASSERT( nn::util::is_aligned(size, nn::dd::DeviceAddressSpaceMemoryRegionAlignment) );
    uint32_t deviceAddress = nne::audio::iova::Map(virtualAddress, size);
    NN_SDK_ASSERT( nn::util::is_aligned(deviceAddress, nn::dd::DeviceAddressSpaceMemoryRegionAlignment) );
    return deviceAddress;
}

nn::Result HardwareOpusDecoderManagerImpl::MapMemoryOnAdsp(
    uint32_t deviceAddress,
    size_t size) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    GetSharedMemory()->to[0] = deviceAddress; // ワーク領域のデバイスアドレス
    GetSharedMemory()->to[1] = size;          // ワーク領域のサイズ
    nn::dd::StoreDataCache(GetSharedMemory()->to, sizeof(nn::codec::dsp::ArgumentType) * nn::codec::dsp::ArgumentCount);
    NN_RESULT_DO(TryingSendMessage(OpusDecoderCommand_MapMemory, MailboxIntervalOnSending));
    // 応答を待つ。
    uint32_t resp = 0;
    NN_RESULT_DO(WaitForMessage(&resp, MailboxIntervalOnReceiving));
    NN_RESULT_THROW_UNLESS(resp == OpusDecoderResponse_MapMemory, ResultCoprocessorUnexpectedResponse());
    NN_RESULT_SUCCESS;
}

nn::Result HardwareOpusDecoderManagerImpl::Map(
    uint32_t* pOutDeviceAddress,
    const void* virtualAddress,
    size_t size) NN_NOEXCEPT
{
    *pOutDeviceAddress = MapMemoryOnSmmu(virtualAddress, size);
    return MapMemoryOnAdsp(*pOutDeviceAddress, size);
}

void HardwareOpusDecoderManagerImpl::UnmapMemoryFromSmmu(
    const void* virtualAddress,
    size_t size,
    uint32_t deviceAddress) NN_NOEXCEPT
{
    NN_SDK_ASSERT( nn::util::is_aligned(size, nn::dd::DeviceAddressSpaceMemoryRegionAlignment) );
    nne::audio::iova::Unmap(virtualAddress, size, deviceAddress);
}

nn::Result HardwareOpusDecoderManagerImpl::UnmapMemoryFromAdsp(
    uint32_t deviceAddress,
    size_t size) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    GetSharedMemory()->to[0] = deviceAddress; // ワーク領域のデバイスアドレス
    GetSharedMemory()->to[1] = size;          // ワーク領域のサイズ
    nn::dd::StoreDataCache(GetSharedMemory()->to, sizeof(nn::codec::dsp::ArgumentType) * nn::codec::dsp::ArgumentCount);
    NN_RESULT_DO(TryingSendMessage(OpusDecoderCommand_UnmapMemory, MailboxIntervalOnSending));
    // 応答を待つ。
    uint32_t resp = 0;
    NN_RESULT_DO(WaitForMessage(&resp, MailboxIntervalOnReceiving));
    NN_RESULT_THROW_UNLESS(resp == OpusDecoderResponse_UnmapMemory, ResultCoprocessorUnexpectedResponse());
    NN_RESULT_SUCCESS;
}

nn::Result HardwareOpusDecoderManagerImpl::Unmap(
    const void* virtualAddress,
    size_t size,
    uint32_t deviceAddress) NN_NOEXCEPT
{
    auto result = UnmapMemoryFromAdsp(deviceAddress, size);
    UnmapMemoryFromSmmu(virtualAddress, size, deviceAddress);
    return result;
}

nn::Result HardwareOpusDecoderManagerImpl::WaitForMessage(uint32_t* pMessage, int64_t intervalInMilli) NN_NOEXCEPT
{
    NN_SDK_ASSERT( m_Mailbox.IsAvailable() );

    int _msg = 0;
    m_Mailbox.ReceiveData(&_msg, intervalInMilli);
    *pMessage = static_cast<uint32_t>(_msg);
    return MailboxErrorHanderOfReceiving(m_Mailbox.GetLastError());
}

nn::Result HardwareOpusDecoderManagerImpl::MailboxErrorHanderOfReceiving(int errorCode) NN_NOEXCEPT
{
    switch(errorCode)
    {
    case dsp::MboxWrapper::MailboxStatusNoError:
        break;
    case -ETIMEDOUT:
        {
            NN_RESULT_THROW(ResultIpcTimeoutDuringReceiving());
        }
        break;
    case -EINVAL:
        {
            NN_RESULT_THROW(ResultIpcInvalidArguments());
        }
        break;
    case -EPERM:
        {
            NN_RESULT_THROW(ResultIpcOperationNotPermitted());
        }
        break;
    case -ENOMEM:
        {
            NN_RESULT_THROW(ResultIpcNoSlot());
        }
        break;
    default:
        {
            ++m_ErrorCountOfReceiving;
            if ( m_ErrorCountOfReceiving > MailboxErrorCountLimitOnReceiving )
            {
                NN_RESULT_THROW(ResultIpcTooManyErrorsDuringReceiving());
            }
        }
        break;
    }
    NN_RESULT_SUCCESS;
}

nn::Result HardwareOpusDecoderManagerImpl::TryingSendMessage(uint32_t msg, int64_t intervalInMilli) NN_NOEXCEPT
{
    NN_SDK_ASSERT( m_Mailbox.IsAvailable() );
    const int64_t retryCountMaximum = intervalInMilli / MailboxIntervalOnSendingInternal;
    int64_t retryCount;
    int status = dsp::MboxWrapper::MailboxStatusNoError;
    for (retryCount = 0; retryCount < retryCountMaximum; ++retryCount)
    {
        if (true == m_Mailbox.SendData(static_cast<int>(msg), MailboxIntervalOnSendingInternal))
        {
            retryCount = retryCountMaximum; // exit
        }
        status = m_Mailbox.GetLastError();
    }
    return MailboxErrorHanderOfSending(status);
}

nn::Result HardwareOpusDecoderManagerImpl::MailboxErrorHanderOfSending(int errorCode) NN_NOEXCEPT
{
    switch(errorCode)
    {
    case dsp::MboxWrapper::MailboxStatusNoError:
        break;
    case -ETIMEDOUT: return ResultIpcTimeoutDuringSending();
        {
            NN_RESULT_THROW(ResultIpcTimeoutDuringSending());
        }
        break;
    case -EINVAL:
        {
            NN_RESULT_THROW(ResultIpcInvalidArguments());
        }
        break;
    case -EPERM:
        {
            NN_RESULT_THROW(ResultIpcOperationNotPermitted());
        }
        break;
    case -ENOMEM:
        {
            NN_RESULT_THROW(ResultIpcNoSlot());
        }
        break;
    default:
        {
            ++m_ErrorCountOfSending;
            if ( m_ErrorCountOfSending > MailboxErrorCountLimitOnSending )
            {
                NN_RESULT_THROW(ResultIpcTooManyErrorsDuringSending());
            }
        }
        break;
    }
    NN_RESULT_SUCCESS;
}

}  // namespace detail
}  // namespace codec
}  // namespace nn
