﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>

#include <nn/codec/codec_OpusCommon.h>
#include <nn/codec/codec_OpusEncoderTypes.h>
#include <nn/codec/detail/codec_OpusEncoderTypesInternal.h>

#include "opus.h"
#include "opus_private.h"

namespace nn {
namespace codec {
namespace detail {

int GetOpusApplicationValue(OpusApplication application) NN_NOEXCEPT
{
    switch(application)
    {
    case OpusApplication_Audio:
        {
            return OPUS_APPLICATION_AUDIO;
        }
    case OpusApplication_Voip:
        {
            return OPUS_APPLICATION_VOIP;
        }
    case OpusApplication_RestrictedLowDelay:
        {
            return OPUS_APPLICATION_RESTRICTED_LOWDELAY;
        }
    default: NN_UNEXPECTED_DEFAULT;
    }
}

int GetOpusForceModeValue(OpusCodingMode forceMode) NN_NOEXCEPT
{
    switch(forceMode)
    {
    case OpusCodingMode_Auto:
        {
            return OPUS_AUTO;
        }
    case OpusCodingMode_Silk:
    case OpusCodingMode_Hybrid:
        {
            return MODE_SILK_ONLY;
        }
    case OpusCodingMode_Celt:
        {
            return MODE_CELT_ONLY;
        }
    default: NN_UNEXPECTED_DEFAULT;
    }
}

int GetOpusBandwidthValue(OpusBandwidth bandwidth) NN_NOEXCEPT
{
    switch(bandwidth)
    {
    case OpusBandwidth_Auto:
        {
            return OPUS_AUTO;
        }
    case OpusBandwidth_NarrowBand:
        {
            return OPUS_BANDWIDTH_NARROWBAND;
        }
    case OpusBandwidth_MediumBand:
        {
            return OPUS_BANDWIDTH_MEDIUMBAND;
        }
    case OpusBandwidth_WideBand:
        {
            return OPUS_BANDWIDTH_WIDEBAND;
        }
    case OpusBandwidth_SuperWideBand:
        {
            return OPUS_BANDWIDTH_SUPERWIDEBAND;
        }
    case OpusBandwidth_FullBand:
        {
            return OPUS_BANDWIDTH_FULLBAND;
        }
    default: NN_UNEXPECTED_DEFAULT;
    }
}


int GetOpusWaveformTypeValue(OpusWaveformType waveformType) NN_NOEXCEPT
{
    switch(waveformType)
    {
    case OpusWaveformType_Auto:
        {
            return OPUS_AUTO;
        }
    case OpusWaveformType_Voice:
        {
            return OPUS_SIGNAL_VOICE;
        }
    case OpusWaveformType_Music:
        {
            return OPUS_SIGNAL_MUSIC;
        }
    default: NN_UNEXPECTED_DEFAULT;
    }
}

int GetOpusFrameSizeValue(OpusFrameSize frameSize) NN_NOEXCEPT
{
    switch(frameSize)
    {
    case OpusFrameSize_Argument:
        {
            return OPUS_FRAMESIZE_ARG;
        }
    case OpusFrameSize_2500Us:
        {
            return OPUS_FRAMESIZE_2_5_MS;
        }
    case OpusFrameSize_5000Us:
        {
            return OPUS_FRAMESIZE_5_MS;
        }
    case OpusFrameSize_10000Us:
        {
            return OPUS_FRAMESIZE_10_MS;
        }
    case OpusFrameSize_20000Us:
        {
            return OPUS_FRAMESIZE_20_MS;
        }
    case OpusFrameSize_40000Us:
        {
            return OPUS_FRAMESIZE_40_MS;
        }
    case OpusFrameSize_60000Us:
        {
            return OPUS_FRAMESIZE_60_MS;
        }
    case OpusFrameSize_Variable:
        {
            return OPUS_FRAMESIZE_VARIABLE;
        }
    default: NN_UNEXPECTED_DEFAULT;
    }
}

void OpusEncoderControl(::OpusEncoder* pOpusEncoder, int request, ...) NN_NOEXCEPT
{
    va_list arg;
    va_start(arg, request);
    intptr_t argument = (request & 1) == 0 ?
        va_arg(arg, opus_int32) : reinterpret_cast<intptr_t>(va_arg(arg, opus_int32*));
    const int status = opus_encoder_ctl(pOpusEncoder, request, argument);
    NN_UNUSED(status);
    NN_SDK_ASSERT(OPUS_OK == status);
    va_end(arg);
}

/**
 * @brief       opus_encoder_ctl() の SET リクエストで設定した値が設定されているか確認する関数です。
 *
 * @details     OpusEncoderControl() の SET リクエスト直後に呼び出す検証目的の関数です。
 *              検証は Develop ビルドでおこない、Release ではコードを生成しないようにします (NN_SDK_ASSERT のラッパという位置づけ)。
 */
void VerifyOpusEncoderControlValue(::OpusEncoder* pOpusEncoder, int getRequest, int correctValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT(getRequest & 1, "Illegal GET Request ID: %d", getRequest); // 奇数である (GET リクエスト)
    NN_UNUSED(pOpusEncoder);
    NN_UNUSED(correctValue);
    NN_UNUSED(getRequest);
    int actualValue = 0;
    NN_UNUSED(actualValue);
#if !defined(NN_SDK_BUILD_RELEASE)
    OpusEncoderControl(pOpusEncoder, getRequest, &actualValue);
#endif // !defined(NN_SDK_BUILD_RELEASE)
    NN_SDK_ASSERT(correctValue == actualValue, "Unmatched: %d (correct) <==> %d (actual) (GET request ID=%d)", correctValue, actualValue, getRequest);
}

// Set/GetXxxInternal は、Internal を除外したメンバ関数の内部実装である。
// Internal 版は、IsInitialized() を事前条件に含めない点だけが非 Internal 版と違う。
// 内部実装の都合のための Internal 関数なので、NN_SDK_ASSERT を使用している。
void SetBitRateInternal(::OpusEncoder* pOpusEncoder, int bitRate, int channelCount) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOpusEncoder);
    NN_SDK_ASSERT_MINMAX(bitRate, GetOpusBitRateMin(channelCount), GetOpusBitRateMax(channelCount)); // チャンネル数が考慮されていない不完全な ASSERT
    NN_UNUSED(channelCount);
    OpusEncoderControl(pOpusEncoder, OPUS_SET_BITRATE(bitRate));
    VerifyOpusEncoderControlValue(pOpusEncoder, OPUS_GET_BITRATE_REQUEST, bitRate);

}

int GetBitRateInternal(::OpusEncoder* pOpusEncoder) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOpusEncoder);
    int bitRate;
    OpusEncoderControl(pOpusEncoder, OPUS_GET_BITRATE(&bitRate));
    return bitRate;
}

void SetBitRateControlInternal(::OpusEncoder* pOpusEncoder, OpusBitRateControl bitRateControl) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOpusEncoder);
    // Cvbr at default
    int vbr = 1;
    int cvbr = 1;
    if (bitRateControl == OpusBitRateControl_Vbr)
    {
        cvbr = 0;
    }
    else if (bitRateControl == OpusBitRateControl_Cbr)
    {
        vbr = 0;
    }
    OpusEncoderControl(pOpusEncoder, OPUS_SET_VBR(vbr));
    OpusEncoderControl(pOpusEncoder, OPUS_SET_VBR_CONSTRAINT(cvbr));
    VerifyOpusEncoderControlValue(pOpusEncoder, OPUS_GET_VBR_REQUEST, vbr);
    VerifyOpusEncoderControlValue(pOpusEncoder, OPUS_GET_VBR_CONSTRAINT_REQUEST, cvbr);
}

void SetApplicationInternal(::OpusEncoder* pOpusEncoder, OpusApplication application) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOpusEncoder);
    NN_SDK_ASSERT(
        application == OpusApplication_Audio
        || application == OpusApplication_Voip
        || application == OpusApplication_RestrictedLowDelay);

    const int value = GetOpusApplicationValue(application);
    OpusEncoderControl(pOpusEncoder, OPUS_SET_APPLICATION(value));
    VerifyOpusEncoderControlValue(pOpusEncoder, OPUS_GET_APPLICATION_REQUEST, value);
}

void SetBandwidthInternal(::OpusEncoder* pOpusEncoder, OpusBandwidth bandwidth) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOpusEncoder);
    NN_SDK_ASSERT(
        bandwidth == OpusBandwidth_Auto
        || bandwidth == OpusBandwidth_NarrowBand
        || bandwidth == OpusBandwidth_MediumBand
        || bandwidth == OpusBandwidth_WideBand
        || bandwidth == OpusBandwidth_SuperWideBand
        || bandwidth == OpusBandwidth_FullBand);

    const int value = GetOpusBandwidthValue(bandwidth);
    OpusEncoderControl(pOpusEncoder, OPUS_SET_BANDWIDTH(value));
    // SET は st->user_bandwidth に設定する、GET は st->bandwidth を取得するので、確認できない。
    // VerifyOpusEncoderControlValue(pOpusEncoder, OPUS_GET_BANDWIDTH_REQUEST, value);
}

void SetForceModeInternal(::OpusEncoder* pOpusEncoder, OpusCodingMode codingMode) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOpusEncoder);
    NN_SDK_ASSERT(
        codingMode == OpusCodingMode_Auto
        || codingMode == OpusCodingMode_Silk
        || codingMode == OpusCodingMode_Celt
        || codingMode == OpusCodingMode_Hybrid);

    const int value = GetOpusForceModeValue(codingMode);
    OpusEncoderControl(pOpusEncoder, OPUS_SET_FORCE_MODE(value));
    VerifyOpusEncoderControlValue(pOpusEncoder, OPUS_GET_FORCE_MODE_REQUEST, value);
}

void SetSignalInternal(::OpusEncoder* pOpusEncoder, OpusWaveformType waveformType) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOpusEncoder);
    NN_SDK_ASSERT(
        waveformType == OpusWaveformType_Auto
        || waveformType == OpusWaveformType_Music
        || waveformType == OpusWaveformType_Voice);

    const int value = GetOpusWaveformTypeValue(waveformType);
    OpusEncoderControl(pOpusEncoder, OPUS_SET_SIGNAL(value));
    VerifyOpusEncoderControlValue(pOpusEncoder, OPUS_GET_SIGNAL_REQUEST, value);
}

void SetComplexityInternal(::OpusEncoder* pOpusEncoder, int complexity) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOpusEncoder);
    NN_SDK_ASSERT_MINMAX(complexity, 0, 10);

    const int value = complexity;
    OpusEncoderControl(pOpusEncoder, OPUS_SET_COMPLEXITY(value));
    VerifyOpusEncoderControlValue(pOpusEncoder, OPUS_GET_COMPLEXITY_REQUEST, value);
}

void SetLsbDepthInternal(::OpusEncoder* pOpusEncoder, int depth) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOpusEncoder);
    NN_SDK_ASSERT_MINMAX(depth, 8, 24);

    const int value = depth;
    OpusEncoderControl(pOpusEncoder, OPUS_SET_LSB_DEPTH(value));
    VerifyOpusEncoderControlValue(pOpusEncoder, OPUS_GET_LSB_DEPTH_REQUEST, value);
}

void SetFrameSizeTypeInternal(::OpusEncoder* pOpusEncoder, OpusFrameSize frameSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOpusEncoder);
    NN_SDK_ASSERT(
        frameSize == OpusFrameSize_Argument
        || frameSize == OpusFrameSize_2500Us
        || frameSize == OpusFrameSize_5000Us
        || frameSize == OpusFrameSize_10000Us
        || frameSize == OpusFrameSize_20000Us
        || frameSize == OpusFrameSize_40000Us
        || frameSize == OpusFrameSize_60000Us
        || frameSize == OpusFrameSize_Variable);

    const int value = GetOpusFrameSizeValue(frameSize);
    OpusEncoderControl(pOpusEncoder, OPUS_SET_EXPERT_FRAME_DURATION(value));
    VerifyOpusEncoderControlValue(pOpusEncoder, OPUS_GET_EXPERT_FRAME_DURATION_REQUEST, value);
}

int GetLookAheadInternal(::OpusEncoder* pOpusEncoder) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOpusEncoder);
    int lookAhead;
    OpusEncoderControl(pOpusEncoder, OPUS_GET_LOOKAHEAD(&lookAhead));
    return lookAhead;
}


uint32_t GetFinalRangeInternal(::OpusEncoder* pOpusEncoder) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOpusEncoder);
    uint32_t finalRange;
    OpusEncoderControl(pOpusEncoder, OPUS_GET_FINAL_RANGE(&finalRange));
    return finalRange;
}

}}}  // nn::codec::detail
