﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <cstring>

#include <nn/nn_SdkAssert.h>
#include <nn/codec/codec_OpusEncoder.h>
#include <nn/codec/codec_OpusUtility.h>

#include <nn/codec/detail/codec_OpusPacketInternal.h>
#include <nn/codec/detail/codec_OpusEncoderTypesInternal.h>
#include <nn/codec/detail/codec_OpusCommonInternal.h>

#include "opus.h"
#include "opus_private.h"

#include "codec_OpusUtilityInternal.h"

namespace nn {
namespace codec {
namespace {

// opus_demo の初期値に合わせておく
const int OpusDefaultComplexity = 10;

// opus_demo の初期値に合わせておく
// https://www.opus-codec.org/docs/opus_api-1.1.2.pdf には、以下の記載がある。
//   When using opus_encode() instead of opus_encode_float(), or when libopus is compiled for fixed-point,
//    the encoder uses the minimum of the value set here and the value 16.
// NintendoSDK 0.9.0 時点では libopus-1.1 を使用しており、上記の記載と使用している libopus との関係
// は直接的にはないかも知れないが、留意しておくべき内容と考え、上記の抜粋をここに付しておく。
const int OpusDefaultLsbDepth = 16;

}

// OpusEncoder クラスメンバ関数
OpusEncoder::OpusEncoder() NN_NOEXCEPT
    : m_Encoder()
{
}

OpusEncoder::~OpusEncoder() NN_NOEXCEPT
{
}

OpusResult OpusEncoder::Initialize(int sampleRate, int channelCount, void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!IsInitialized());
    return InitializeOpusEncoder(&m_Encoder, sampleRate, channelCount, buffer, size);
}

OpusResult OpusEncoder::Initialize(int sampleRate, int channelCount, OpusApplication application, void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!IsInitialized());
    return InitializeOpusEncoder(&m_Encoder, sampleRate, channelCount, application, buffer, size);
}

size_t GetOpusEncoderWorkBufferSize(int sampleRate, int channelCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(sampleRate == 48000 || sampleRate == 24000 || sampleRate == 16000 || sampleRate == 12000 || sampleRate == 8000);
    NN_SDK_REQUIRES(channelCount >= 1 && channelCount <= 2);

    NN_UNUSED(sampleRate);

    return opus_encoder_get_size(channelCount);
}

OpusResult InitializeOpusEncoder(OpusEncoderType* pOutEncoder, int sampleRate, int channelCount, OpusApplication application, void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutEncoder);
    NN_SDK_REQUIRES(sampleRate == 48000 || sampleRate == 24000 || sampleRate == 16000 || sampleRate == 12000 || sampleRate == 8000);
    NN_SDK_REQUIRES(channelCount >= 1 && channelCount <= 2);
    NN_SDK_REQUIRES(application == OpusApplication_Audio || application == OpusApplication_Voip || application == OpusApplication_RestrictedLowDelay);
    NN_SDK_REQUIRES(buffer != nullptr);
    NN_SDK_REQUIRES(size >= GetOpusEncoderWorkBufferSize(sampleRate, channelCount));

    if (sampleRate != 48000 && sampleRate != 24000 && sampleRate != 16000 && sampleRate != 12000 && sampleRate != 8000)
    {
        return OpusResult_InvalidSampleRate;
    }
    if (channelCount != 1 && channelCount != 2)
    {
        return OpusResult_InvalidChannelCount;
    }
    if (buffer == nullptr || size < GetOpusEncoderWorkBufferSize(sampleRate, channelCount))
    {
        return OpusResult_InvalidWorkBuffer;
    }
    auto pOpusEncoder = static_cast<::OpusEncoder*>(buffer);

    if (opus_encoder_init(pOpusEncoder, sampleRate, channelCount, detail::GetOpusApplicationValue(application)) != OPUS_OK)
    {
        return OpusResult_InternalError;
    }

    pOutEncoder->_sampleRate = sampleRate;
    pOutEncoder->_channelCount = channelCount;
    pOutEncoder->_buffer = buffer;
    pOutEncoder->_size = size;
    pOutEncoder->_handle = pOpusEncoder;

    // opus_encoder から得た初期値 (サンプルレート、チャンネル数、前回のフレームサイズに基づく) を
    // 初期値とし、opus_encoder に設定しなおす。
    // 上記の "opus_encoder から得た初期値" は、フレームサイズが変化する場合には固定値ではないことに注意
    // 設定し直しは、ビットレートを固定するための操作。
    // ToDo: 動的ビットレート計算モードに入れる API が必要かどうか、検討して対応する。
    //       SDK ユーザからの要望がなければ、対応する必要はないと思う。
    pOutEncoder->_bitRate = detail::GetBitRateInternal(pOpusEncoder);
    detail::SetBitRateInternal(pOpusEncoder, pOutEncoder->_bitRate, pOutEncoder->_channelCount);

    // デフォルトは Vbr とする (初期化処理直後は Cvbr になっている)。
    pOutEncoder->_bitRateControl = OpusBitRateControl_Vbr;
    detail::SetBitRateControlInternal(pOpusEncoder, pOutEncoder->_bitRateControl);

    // 初期値を、opus_demo のものに合わせる (opus_demo のエンコード機能を照合データとしているため)。
    detail::SetComplexityInternal(pOpusEncoder, OpusDefaultComplexity);
    detail::SetLsbDepthInternal(pOpusEncoder, OpusDefaultLsbDepth);
    detail::SetFrameSizeTypeInternal(pOpusEncoder, OpusFrameSize_Argument);
    detail::VerifyOpusEncoderControlValue(pOpusEncoder, OPUS_GET_FORCE_MODE_REQUEST, OPUS_AUTO);
    detail::VerifyOpusEncoderControlValue(pOpusEncoder, OPUS_GET_SIGNAL_REQUEST, OPUS_AUTO);

    pOutEncoder->_codingMode = OpusCodingMode_Auto;
    pOutEncoder->_waveformType = OpusWaveformType_Auto;

    nn::os::InitializeMutex(&pOutEncoder->_mutex, false, 0);

    return OpusResult_Success;
}

OpusResult InitializeOpusEncoder(OpusEncoderType* pEncoder, int sampleRate, int channelCount, void* buffer, size_t size) NN_NOEXCEPT
{
    return InitializeOpusEncoder(pEncoder, sampleRate, channelCount, OpusApplication_RestrictedLowDelay, buffer, size);
}

void FinalizeOpusEncoder(OpusEncoderType* pEncoder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEncoder);
    NN_SDK_REQUIRES_NOT_NULL(pEncoder->_handle);

    nn::os::LockMutex(&pEncoder->_mutex);

    pEncoder->_sampleRate = 0;
    pEncoder->_channelCount = 0;
    pEncoder->_buffer = nullptr;
    pEncoder->_size = 0;
    pEncoder->_handle = nullptr;

    nn::os::UnlockMutex(&pEncoder->_mutex);
    nn::os::FinalizeMutex(&pEncoder->_mutex);
}

int GetOpusEncoderSampleRate(const OpusEncoderType* pEncoder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEncoder);
    NN_SDK_REQUIRES_NOT_NULL(pEncoder->_handle);
    return pEncoder->_sampleRate;
}

int GetOpusEncoderChannelCount(const OpusEncoderType* pEncoder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEncoder);
    NN_SDK_REQUIRES_NOT_NULL(pEncoder->_handle);
    return pEncoder->_channelCount;
}

OpusResult EncodeOpusInterleaved(OpusEncoderType* pEncoder, size_t* pOutputSize, void* outputBuffer, size_t outputBufferSize, const int16_t* inputBuffer, int inputSampleCountPerChannel) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEncoder);
    NN_SDK_REQUIRES_NOT_NULL(pEncoder->_handle);
    NN_SDK_REQUIRES_NOT_NULL(pOutputSize);
    NN_SDK_REQUIRES_NOT_NULL(outputBuffer);
    NN_SDK_REQUIRES_NOT_NULL(inputBuffer);
    NN_SDK_REQUIRES(
        inputSampleCountPerChannel * 400 == GetOpusEncoderSampleRate(pEncoder)
        || inputSampleCountPerChannel * 200 == GetOpusEncoderSampleRate(pEncoder)
        || inputSampleCountPerChannel * 100 == GetOpusEncoderSampleRate(pEncoder)
        || inputSampleCountPerChannel *  50 == GetOpusEncoderSampleRate(pEncoder) );

    if ( (inputSampleCountPerChannel * 400 == GetOpusEncoderSampleRate(pEncoder) && GetOpusEncoderCodingMode(pEncoder) == OpusCodingMode_Silk)
        || (inputSampleCountPerChannel * 200 == GetOpusEncoderSampleRate(pEncoder) && GetOpusEncoderCodingMode(pEncoder) == OpusCodingMode_Silk) )
    {
        return OpusResult_UnsupportedSampleCount;
    }

    int ret = 0;
    // Lock の範囲のためのブロック化
    {
        // SetBitRate()/SetBitRateControl()/Finalize() と干渉しないように Lock する。
        nn::os::LockMutex(&pEncoder->_mutex);
        // opus_encode が排他的に呼び出されるように Lock する。
        std::lock_guard<nn::os::Mutex> globalLock(detail::GetOpusGlobalMutex());
        ret = opus_encode(
            static_cast<::OpusEncoder*>(pEncoder->_handle),
            inputBuffer,
            inputSampleCountPerChannel,
            static_cast<uint8_t*>(outputBuffer) + detail::OpusPacketInternal::HeaderSize,
            static_cast<int>(outputBufferSize));
        nn::os::UnlockMutex(&pEncoder->_mutex);
    }
    if (ret < 0)
    {
        if (ret == OPUS_BUFFER_TOO_SMALL)
        {
            return OpusResult_InsufficientOpusBuffer;
        }
        else // その他は内部エラー
        {
            return OpusResult_InternalError;
        }
    }
    // 生の Opus パケットにヘッダを付与する (ToDo: 生の Opus パケットの正しい呼び方を要調査)。
    // ToDo:
    // OpusDecoder が前提としている Opus ヘッダ構成 (opus_demo のもの) を元にヘッダを構成する。
    // RFC6716 によると、range は decoder/encoder の検証のための一致確認に用いられるようす。
    // なので、テストで用いればよく、Opus パケットに付与して通信オーバーヘッド
    // を上げるべきではない。
    // SIGLO-30925 にて、Opus ヘッダ構成を見直す。
    const uint32_t packetSize = ret;
    const uint32_t finalRange = detail::GetFinalRangeInternal(static_cast<::OpusEncoder*>(pEncoder->_handle));
    detail::OpusPacketInternal::Header header =
    {
        packetSize,
        finalRange
    };
    detail::OpusPacketInternal::SetHeaderInPacket(static_cast<uint8_t*>(outputBuffer), header);

    // パケットサイズの書き戻し
    *pOutputSize = packetSize + detail::OpusPacketInternal::HeaderSize;

    return OpusResult_Success;
}

void SetOpusEncoderBitRate(OpusEncoderType* pEncoder, int bitRate) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEncoder);
    NN_SDK_REQUIRES_NOT_NULL(pEncoder->_handle);
    NN_SDK_REQUIRES_MINMAX(bitRate, GetOpusBitRateMin(GetOpusEncoderChannelCount(pEncoder)), GetOpusBitRateMax(GetOpusEncoderChannelCount(pEncoder)));
    if (pEncoder->_bitRate != bitRate)
    {
        // Encode()/EncodeInterleaved() と干渉しないように Lock する。
        nn::os::LockMutex(&pEncoder->_mutex);
        detail::SetBitRateInternal(static_cast<::OpusEncoder*>(pEncoder->_handle), bitRate, GetOpusEncoderChannelCount(pEncoder));
        pEncoder->_bitRate = bitRate;
        nn::os::UnlockMutex(&pEncoder->_mutex);
    }
}

int GetOpusEncoderBitRate(const OpusEncoderType* pEncoder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEncoder);
    NN_SDK_REQUIRES_NOT_NULL(pEncoder->_handle);
    return pEncoder->_bitRate;
}

void SetOpusEncoderBitRateControl(OpusEncoderType* pEncoder, OpusBitRateControl bitRateControl) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEncoder);
    NN_SDK_REQUIRES_NOT_NULL(pEncoder->_handle);
    NN_SDK_REQUIRES(bitRateControl == OpusBitRateControl_Vbr || bitRateControl == OpusBitRateControl_Cvbr || bitRateControl == OpusBitRateControl_Cbr);
    if (pEncoder->_bitRateControl != bitRateControl)
    {
        // Encode()/EncodeInterleaved() と干渉しないように Lock する。
        nn::os::LockMutex(&pEncoder->_mutex);
        detail::SetBitRateControlInternal(static_cast<::OpusEncoder*>(pEncoder->_handle), bitRateControl);
        pEncoder->_bitRateControl = bitRateControl;
        nn::os::UnlockMutex(&pEncoder->_mutex);
    }
}

OpusBitRateControl GetOpusEncoderBitRateControl(const OpusEncoderType* pEncoder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEncoder);
    NN_SDK_REQUIRES_NOT_NULL(pEncoder->_handle);
    return pEncoder->_bitRateControl;
}

int GetOpusEncoderPreSkipSampleCount(const OpusEncoderType* pEncoder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEncoder);
    NN_SDK_REQUIRES_NOT_NULL(pEncoder->_handle);
    return detail::GetLookAheadInternal(static_cast<::OpusEncoder*>(pEncoder->_handle));
}

void SetOpusEncoderWaveformType(OpusEncoderType* pEncoder, OpusWaveformType waveformType) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEncoder);
    NN_SDK_REQUIRES_NOT_NULL(pEncoder->_handle);
    NN_SDK_REQUIRES(waveformType == OpusWaveformType_Auto || waveformType == OpusWaveformType_Music || waveformType == OpusWaveformType_Voice);

    if (pEncoder->_waveformType != waveformType)
    {
        // Encode()/EncodeInterleaved() と干渉しないように Lock する。
        nn::os::LockMutex(&pEncoder->_mutex);
        auto pOpusEncoder = static_cast<::OpusEncoder*>(pEncoder->_handle);
        detail::SetSignalInternal(pOpusEncoder, OpusWaveformType_Music);
        pEncoder->_waveformType = waveformType;
        nn::os::UnlockMutex(&pEncoder->_mutex);
    }
}

OpusWaveformType GetOpusEncoderWaveformType(const OpusEncoderType* pEncoder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEncoder);
    NN_SDK_REQUIRES_NOT_NULL(pEncoder->_handle);
    return pEncoder->_waveformType;
}

void BindOpusEncoderCodingMode(OpusEncoderType* pEncoder, OpusCodingMode codingMode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEncoder);
    NN_SDK_REQUIRES_NOT_NULL(pEncoder->_handle);
    NN_SDK_REQUIRES(codingMode == OpusCodingMode_Auto || codingMode == OpusCodingMode_Celt || codingMode == OpusCodingMode_Silk);

    if (pEncoder->_codingMode != codingMode)
    {
        const auto bandwidth = codingMode == OpusCodingMode_Silk ?
            OpusBandwidth_WideBand : OpusBandwidth_Auto;
        const auto application = codingMode == OpusCodingMode_Silk ?
            OpusApplication_Voip : OpusApplication_RestrictedLowDelay;
        const auto forceMode = codingMode == OpusCodingMode_Silk ?
            OpusCodingMode_Silk : OpusCodingMode_Auto;
        auto pOpusEncoder = static_cast<::OpusEncoder*>(pEncoder->_handle);
        // Encode()/EncodeInterleaved() と干渉しないように Lock する。
        nn::os::LockMutex(&pEncoder->_mutex);
        // nn::codec::OpusEncoder の符号化モードは、opus バンド幅と opus 強制モード指定 (Auto/Silk/Celt) で制御している。
        detail::SetForceModeInternal(pOpusEncoder, forceMode);
        detail::SetBandwidthInternal(pOpusEncoder, bandwidth);
        detail::SetApplicationInternal(pOpusEncoder, application);
        pEncoder->_codingMode = codingMode;
        nn::os::UnlockMutex(&pEncoder->_mutex);
    }
}

OpusCodingMode GetOpusEncoderCodingMode(const OpusEncoderType* pEncoder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEncoder);
    NN_SDK_REQUIRES_NOT_NULL(pEncoder->_handle);
    return pEncoder->_codingMode;
}

int CalculateOpusEncoderFrameSampleCount(const OpusEncoderType* pEncoder, int frame) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEncoder);
    NN_SDK_REQUIRES_NOT_NULL(pEncoder->_handle);
    return CalculateOpusFrameSampleCount(frame, GetOpusEncoderSampleRate(pEncoder));
}

}  // namespace codec
}  // namespace nn
