﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_ErrorResult.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/util/util_BitUtil.h>

#include <nn/codec/codec_Result.private.h>

#include "codec_CreateHardwareOpusDecoderManager.h"
#include "codec_HardwareOpusDecoderManagerImpl.h"
#include "codec_HardwareOpusDecoderImpl.h"

namespace nn { namespace codec {

HardwareOpusDecoderManagerImpl::HardwareOpusDecoderManagerImpl() NN_NOEXCEPT
{
    m_HeapHandle = nn::lmem::CreateExpHeap(&m_HeapBuffer, sizeof(m_HeapBuffer), nn::lmem::CreationOption_NoOption);
    m_Allocator.Attach(m_HeapHandle);
}

HardwareOpusDecoderManagerImpl::~HardwareOpusDecoderManagerImpl() NN_NOEXCEPT
{
    nn::lmem::DestroyExpHeap(m_HeapHandle);
}

nn::Result HardwareOpusDecoderManagerImpl::OpenHardwareOpusDecoder(
    nn::sf::Out<nn::sf::SharedPointer<detail::IHardwareOpusDecoder>> outHardwareOpusDecoder,
    const detail::HardwareOpusDecoderParameterInternal& parameter,
    nn::sf::NativeHandle sfHandle,
    int workBufferSize) NN_NOEXCEPT
{
    typedef nn::sf::ObjectFactory<MyAllocator::Policy> Factory;
    auto p = Factory::CreateSharedEmplaced<detail::IHardwareOpusDecoder, HardwareOpusDecoderImpl>(&m_Allocator);
    NN_RESULT_THROW_UNLESS(p != nullptr, ResultCpuOutOfResource());
    NN_RESULT_DO(Factory::GetEmplacedImplPointer<HardwareOpusDecoderImpl>(p)->Initialize(parameter, sfHandle, workBufferSize));

    *outHardwareOpusDecoder = std::move(p);
    NN_RESULT_SUCCESS;
}

nn::Result HardwareOpusDecoderManagerImpl::GetWorkBufferSize(
    nn::sf::Out<int> outSize,
    const detail::HardwareOpusDecoderParameterInternal& parameter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(parameter.channelCount == 1 || parameter.channelCount == 2);
    NN_SDK_REQUIRES(parameter.sampleRate == 48000 || parameter.sampleRate == 24000 || parameter.sampleRate == 16000 || parameter.sampleRate == 12000 || parameter.sampleRate == 8000);

    const auto size =
        detail::HardwareOpusDecoderManagerImpl::GetWorkBufferSize(parameter.channelCount)
        + detail::HardwareOpusDecoderImpl::GetInputBufferSize(parameter.sampleRate, 1, parameter.channelCount == 1 ? 0 : 1)
        + detail::HardwareOpusDecoderImpl::GetOutputBufferSize(parameter.sampleRate, parameter.channelCount);
    *outSize = size;
    NN_RESULT_SUCCESS;
}

nn::Result HardwareOpusDecoderManagerImpl::OpenHardwareOpusDecoderForMultiStream(
    nn::sf::Out<nn::sf::SharedPointer<detail::IHardwareOpusDecoder>> outHardwareOpusDecoder,
    const detail::HardwareOpusMultiStreamDecoderParameterInternal& parameter,
    nn::sf::NativeHandle sfHandle,
    int workBufferSize) NN_NOEXCEPT
{
    typedef nn::sf::ObjectFactory<MyAllocator::Policy> Factory;
    auto p = Factory::CreateSharedEmplaced<detail::IHardwareOpusDecoder, HardwareOpusDecoderImpl>(&m_Allocator);
    NN_RESULT_THROW_UNLESS(p != nullptr, ResultCpuOutOfResource());
    NN_RESULT_DO(Factory::GetEmplacedImplPointer<HardwareOpusDecoderImpl>(p)->InitializeForMultiStream(parameter, sfHandle, workBufferSize));

    *outHardwareOpusDecoder = std::move(p);
    NN_RESULT_SUCCESS;
}

nn::Result HardwareOpusDecoderManagerImpl::GetWorkBufferSizeForMultiStream(
    nn::sf::Out<int> outSize,
    const detail::HardwareOpusMultiStreamDecoderParameterInternal& parameter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(parameter.sampleRate == 48000 || parameter.sampleRate == 24000 || parameter.sampleRate == 16000 || parameter.sampleRate == 12000 || parameter.sampleRate == 8000);
    NN_SDK_REQUIRES(parameter.channelCount >= 1 && parameter.channelCount <= OpusStreamCountMax);
    NN_SDK_REQUIRES(parameter.totalStreamCount + parameter.stereoStreamCount <= parameter.channelCount);
    NN_SDK_REQUIRES(parameter.totalStreamCount >= 1 && parameter.totalStreamCount <= OpusStreamCountMax);
    NN_SDK_REQUIRES(parameter.stereoStreamCount >= 0 && parameter.stereoStreamCount <= parameter.totalStreamCount);
    const auto size =
        detail::HardwareOpusDecoderManagerImpl::GetWorkBufferSizeForMultiStream(parameter.totalStreamCount, parameter.stereoStreamCount)
        + detail::HardwareOpusDecoderImpl::GetInputBufferSize(parameter.sampleRate, parameter.totalStreamCount, parameter.stereoStreamCount)
        + detail::HardwareOpusDecoderImpl::GetOutputBufferSize(parameter.sampleRate, parameter.channelCount);
    *outSize = size;
    NN_RESULT_SUCCESS;
}

}}  // namespace nn::codec
