﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>  // TODO: should remove malloc/free
#include "codec_AdpcmEncoderEstimate.h"

namespace nn { namespace codec {

namespace {

const double Epsilon = 1e-30;

}

void SplitCodebook(double** codebook, double* dir, int order, int n_entries, double delta)
{
    // Double the size of the codebook by splitting each entry
    // along the direction 'dir'
    for (auto i = 0; i < n_entries; ++i)
    {
        for (auto j = 0; j <= order; ++j)
        {
            codebook[i + n_entries][j] = codebook[i][j] + delta * dir[j];
        }
    }
}

void RefineCodebook(double** codebook, int order, int n_entries, double** training, int nframes, int iterations, double converge)
{
    // Standard Lloyd algorithm for iterative refinement of the codebook
    int     *count;
    double  **centroids;
    double  *ac;
    double  e2, min, dist;

    centroids = (double **) malloc(n_entries * sizeof(double *));

    for (auto i = 0; i < n_entries; i++)
    {
        centroids[i] = (double *) malloc((order + 1) * sizeof(double));
    }

    count   = (int *) malloc(n_entries * sizeof(int));
    ac      = (double *) malloc((order + 1) * sizeof(double));

    for (auto iter = 0; iter < iterations; iter++)
    {
        // Go through each training vector and find the nearest neighbour
        for (auto j = 0; j < n_entries; j++)
        {
            count[j] = 0;

            for (auto i = 0; i <= order; i++)
            {
                centroids[j][i] = 0.0;
            }
        }

        for (auto tv = 0; tv < nframes; tv++)
        {
            int ne  = 0;
            min = 1e30;

            for (auto i = 0; i < n_entries; i++)
            {
                if (order == 2)
                {
                    dist = CalculateModelDistanceOrder2(codebook[i], training[tv], order);
                }
                else
                {
                    dist = CalculateModelDistance(codebook[i], training[tv], order);
                }

                if (dist<min)
                {
                    min = dist;
                    ne = i;
                }
            }

            // Add the autocorrelation of this training vector to the centroid
            count[ne] += 1;

            CalculateAutoCorrelationFromTap(training[tv], order, ac);

            for (auto i = 0; i <= order; i++)
            {
                centroids[ne][i] += ac[i];
            }
        }

        // Get the average
        for (auto i = 0; i < n_entries; i++)
        {
            if (count[i] > 0)
            {
                for (auto j = 0; j <= order; j++)
                {
                    centroids[i][j] = centroids[i][j] / count[i];
                }
            }
        }

        // Redefine the codebook
        for (auto i = 0; i < n_entries; i++)
        {
            ExecuteDurbin(centroids[i], order, ac, codebook[i], &e2);

            // Stabilize - could put this in durbin
            for (auto j = 1; j <= order; j++)
            {
                if (ac[j] >= 1.0)
                {
                    ac[j] = 1.0 - Epsilon;
                }

                if (ac[j] <= -1.0)
                {
                    ac[j] = -1.0 + Epsilon;
                }
            }

            CalculateTapFromReflection(ac, codebook[i], order);
        }
    }

    free(count);

    for (auto i = 0; i < n_entries; i++)
    {
        free(centroids[i]);
    }

    free(centroids);
    free(ac);
}

}}  // namespace nn::codec
