﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os/os_Config.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/os/os_MutexTypes.h>

#include <cec_TransmitterFiniteStateMachine.h>
#include <cec_Manager.h>
#include "cec_DetailLog.h"
#include "cec_Utilities.h"

namespace nn { namespace cec { namespace detail {

/*
 * Random Number Generator
 * It is designed to generate a Pseudo Random Number in the range of
 * [0, 0x7FFFFFFF], that is, a 31bit integer.
 */
typedef struct {
    uint32_t    current;
    uint32_t    minValue;
    uint32_t    maxValue;
} PseudoRandomNumberGeneratorStateType;

namespace {

    const uint32_t  CecRngConstantModulus     = 0x7FFFFFFF;
    const uint32_t  CecRngConstantMultiplier  = 48271;

    const uint32_t  RngQuotient = CecRngConstantModulus / CecRngConstantMultiplier;
    const uint32_t  RngRemainder = CecRngConstantModulus % CecRngConstantMultiplier;
    const uint32_t  CecParameterMinBroadcastDelayMicroSeconds  = 150 * 1000;
    const uint32_t  CecParameterMaxBroadcastDelayMicroSeconds  = 750 * 1000;

    const int   CecCancelFlushListDelayInMilliseconds = 50;

    PseudoRandomNumberGeneratorStateType s_CecRng =
    {
        0xCECBABE,
        CecParameterMinBroadcastDelayMicroSeconds,
        CecParameterMaxBroadcastDelayMicroSeconds
    };

    struct SleepThreadState
    {
        struct SleepThreadState*    pNext;
        os::TimerEventType          timerEvent;
        bool                        wasCancelled;
    };

    struct SleepingThreadList
    {
        struct SleepThreadState*    pHead;
        os::MutexType               listMutex;
    } s_SleepingThreadList;

    struct CecCancelState
    {
        bool                        cancelApiCall;
        os::MutexType               cancelMutex;
    } s_CecCancelState;

    void    CecCancelAddSleepingThreadToList(struct SleepThreadState* pState) NN_NOEXCEPT
    {
        struct SleepThreadState**   ppCurrent;

        os::LockMutex(&s_SleepingThreadList.listMutex);
        ppCurrent = &s_SleepingThreadList.pHead;
        pState->pNext = nullptr;
        while(*ppCurrent != nullptr)
        {
            ppCurrent = &(*ppCurrent)->pNext;
        }
        *ppCurrent = pState;
        os::UnlockMutex(&s_SleepingThreadList.listMutex);
    }

    void    CecCancelDeleteSleepingThreadFromList(struct SleepThreadState* pState) NN_NOEXCEPT
    {
        struct SleepThreadState**   ppCurrent;

        os::LockMutex(&s_SleepingThreadList.listMutex);
        ppCurrent = &s_SleepingThreadList.pHead;
        while(*ppCurrent != nullptr)
        {
            if(*ppCurrent == pState)
            {
                *ppCurrent = pState->pNext;
                pState->pNext = nullptr;
                break;
            }
            ppCurrent = &(*ppCurrent)->pNext;
        }
        os::UnlockMutex(&s_SleepingThreadList.listMutex);
    }

    void    CecCancelWakeAllSleepingThreads() NN_NOEXCEPT
    {
        struct SleepThreadState**   ppCurrent;

        os::LockMutex(&s_SleepingThreadList.listMutex);
        ppCurrent = &s_SleepingThreadList.pHead;
        while(*ppCurrent != nullptr)
        {
            NN_CEC_INFO("Waking sleeping thread\n");
            (*ppCurrent)->wasCancelled = true;
            os::SignalTimerEvent(&(*ppCurrent)->timerEvent);
            ppCurrent = &(*ppCurrent)->pNext;
        }
        os::UnlockMutex(&s_SleepingThreadList.listMutex);
    }
}

void    CecCancelInitialize() NN_NOEXCEPT
{
    os::InitializeMutex(&s_SleepingThreadList.listMutex, false, 0);
    s_SleepingThreadList.pHead = nullptr;
    os::InitializeMutex(&s_CecCancelState.cancelMutex, false, 0);
    s_CecCancelState.cancelApiCall = false;
}

void    CecCancelFinalize() NN_NOEXCEPT
{
    bool    canExit;

    os::LockMutex(&s_CecCancelState.cancelMutex);
    CecCancelWakeAllSleepingThreads();
    s_CecCancelState.cancelApiCall = true;
    os::UnlockMutex(&s_CecCancelState.cancelMutex);
    do
    {
        os::LockMutex(&s_SleepingThreadList.listMutex);
        canExit = (s_SleepingThreadList.pHead == nullptr);
        os::UnlockMutex(&s_SleepingThreadList.listMutex);
        if(!canExit)
        {
            os::SleepThread(TimeSpan::FromMilliSeconds(CecCancelFlushListDelayInMilliseconds));
        }
    } while(!canExit);
    os::FinalizeMutex(&s_CecCancelState.cancelMutex);
    os::FinalizeMutex(&s_SleepingThreadList.listMutex);
    s_CecCancelState.cancelApiCall = false;
}

bool    CecCancelSetCancelState(os::MutexType* pApiMutex) NN_NOEXCEPT
{
    bool    rval;

    NN_CEC_INFO("%s: Entry\n", NN_CURRENT_FUNCTION_NAME);
    CecManagerSequenceCancellationHandler();
    os::LockMutex(&s_CecCancelState.cancelMutex);
    if(os::TryLockMutex(pApiMutex))
    {
        NN_CEC_INFO("%s: No API in progress\n", NN_CURRENT_FUNCTION_NAME);
        os::UnlockMutex(pApiMutex);
        rval = false;
    }
    else
    {
        NN_CEC_INFO("%s: Canceling API in progress\n", NN_CURRENT_FUNCTION_NAME);
        CecTransmitSetCancelState();
        s_CecCancelState.cancelApiCall = true;
        CecCancelWakeAllSleepingThreads();
        rval = true;
    }
    os::UnlockMutex(&s_CecCancelState.cancelMutex);
    NN_CEC_INFO("%s: Leave %d\n", NN_CURRENT_FUNCTION_NAME, rval);
    return rval;
}

bool    CecCancelClearCancelState() NN_NOEXCEPT
{
    bool    rval;

    os::LockMutex(&s_CecCancelState.cancelMutex);
    rval = s_CecCancelState.cancelApiCall;
    s_CecCancelState.cancelApiCall = false;
    CecTransmitClearCancelState();
    os::UnlockMutex(&s_CecCancelState.cancelMutex);
    return rval;
}

bool    CecCancelSleepThreadMilliseconds(uint32_t sleepTimeInMilliseconds)
{
    struct SleepThreadState state;
    bool                    rval;

    os::LockMutex(&s_CecCancelState.cancelMutex);
    rval = s_CecCancelState.cancelApiCall;
    if(!rval)
    {
        state.wasCancelled = false;
        os::InitializeTimerEvent(&state.timerEvent, os::EventClearMode_AutoClear);
        os::StartOneShotTimerEvent(&state.timerEvent, TimeSpan::FromMilliSeconds(sleepTimeInMilliseconds));
        CecCancelAddSleepingThreadToList(&state);
    }
    os::UnlockMutex(&s_CecCancelState.cancelMutex);
    if(!rval)
    {
        os::WaitTimerEvent(&state.timerEvent);
        os::LockMutex(&s_CecCancelState.cancelMutex);
        CecCancelDeleteSleepingThreadFromList(&state);
        os::UnlockMutex(&s_CecCancelState.cancelMutex);
        rval = state.wasCancelled;
        os::FinalizeTimerEvent(&state.timerEvent);
    }
    if(rval)
    {
        NN_CEC_INFO("Canceled Sleep Thread\n");
    }
    return rval;
}

void    CecStepPseudoRandomNumberGenerator() NN_NOEXCEPT
{
    uint32_t    nextRnd;

    nextRnd = CecRngConstantMultiplier * (s_CecRng.current % RngQuotient) -
              RngRemainder * (s_CecRng.current / RngQuotient);
    if(nextRnd > CecRngConstantModulus)
    {
        s_CecRng.current = nextRnd + CecRngConstantModulus;
    }
    else
    {
        s_CecRng.current = nextRnd;
    }
}

uint32_t    CecComputePseudoRandomBroadcastDelayInMicroSeconds() NN_NOEXCEPT
{
    uint32_t    rval;

    rval = (s_CecRng.current % (s_CecRng.maxValue - s_CecRng.minValue)) + s_CecRng.minValue;
    CecStepPseudoRandomNumberGenerator();
    return rval;
}

uint64_t    CecComputeSixtyFourBitHashValue(uint8_t* pBuffer, uint32_t countBytes) NN_NOEXCEPT
{
    uint64_t    rval=0;

    return rval;
}

}
}   // namespace cec
}   // namespace nn
