﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os/os_Config.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/os/os_MutexTypes.h>
#include <nn/os/os_SdkThread.h>
#include <nn/nn_SdkAssert.h>

#include "cec_DetailLog.h"
#include "cec_Error.h"
#include "cec_FiniteStateMachine.h"
#include "cec_TransmitterFiniteStateMachine.h"
#include "cec_LowLevelInterface.h"
#include "cec_Receiver.h"
#include "cec_Utilities.h"

namespace nn { namespace cec { namespace detail {

typedef struct {
    uint32_t                flags;
    nn::os::MutexType       flagMutex;
    nn::os::MutexType       apiMutex;
    nn::os::TimerEventType  timerEvent;
    FiniteStateMachineType  fsmState;
} TransmitStateType;

typedef struct {
    uint8_t*                pBuffer;
    uint8_t                 count;
    TransmitReturnValueType transmitterReturnValue;
} TransmitArgumentsType;

namespace {

/* this should be an enum */
const FiniteStateMachineStateType TransmitterStateInit = 0;
const FiniteStateMachineStateType TransmitterStateRun = 1;
const FiniteStateMachineStateType TransmitterStateNumber = 2;

/* this should be an enum */
const FiniteStateMachineInputType TransmitterInputTransmit = 0;
const FiniteStateMachineInputType TransmitterInputReset = 1;
const FiniteStateMachineInputType TransmitterInputNumber = 2;

const FiniteStateMachineActionValueType TransmitterActionStartTimer = 0;
const FiniteStateMachineActionValueType TransmitterActionClearTimer = 1;
const FiniteStateMachineActionValueType TransmitterActionAbort = 2;
const FiniteStateMachineActionValueType TransmitterActionTransmit = 3;
const FiniteStateMachineActionValueType TransmitterActionWait = 4;
const FiniteStateMachineActionValueType TransmitterActionNumber = 5;

const FiniteStateMachineStateType s_TransmitterStateMatrix[TransmitterStateNumber * TransmitterInputNumber] =
{
    TransmitterStateRun, TransmitterStateInit,
    TransmitterStateRun, TransmitterStateInit
};

const FiniteStateMachineActionType s_TransmitterActionMatrix[TransmitterStateNumber * TransmitterInputNumber] =
{
    {{TransmitterActionTransmit, TransmitterActionStartTimer, FsmParamNotAnAction, FsmParamNotAnAction}},
    {{TransmitterActionClearTimer, TransmitterActionAbort, FsmParamNotAnAction, FsmParamNotAnAction}},
    {{TransmitterActionWait, TransmitterActionTransmit, TransmitterActionStartTimer, FsmParamNotAnAction}},
    {{TransmitterActionClearTimer, TransmitterActionAbort, FsmParamNotAnAction, FsmParamNotAnAction}}
};

const uint32_t TransmitterStateFlagTransmitEnabledBit = (1 << 0);
const uint32_t TransmitterStateFlagCancelledBit = (1 << 1);


TransmitStateType   s_TransmitterState;

const int32_t  TransmitterParameterMaximumNumberOfTimeoutInPolling = 1;
const int32_t  TransmitterParameterPollingAttempts      = 3;
const uint32_t TransmitterParameterPollSignalFreeTimeMs = 20;
const int32_t  TransmitterParameterBestEffortNakRetries = 5;
const int32_t  TransmitterParameterBestEffortErrorRetries = 2;

bool    TransmitterIsTransactionCancelled(TransmitStateType* pTransmitterState) NN_NOEXCEPT
{
    bool    rval;

    os::LockMutex(&pTransmitterState->flagMutex);
    rval = !!(pTransmitterState->flags & TransmitterStateFlagCancelledBit);
    os::UnlockMutex(&pTransmitterState->flagMutex);
    return rval;
}

void TransmitterActionHandlerStartTimer(FiniteStateMachineActionValueType actCode, void* pContext, void* pExtra) NN_NOEXCEPT
{
    TransmitStateType*      pTransmitterState;
    int64_t                 delayNs=0;

    pTransmitterState = static_cast<TransmitStateType*>(pContext);
    delayNs = CecReceiveParamFrameToFrameDelayNanoSeconds + CecReceiveParamCradleFirmwareBugWorkAroundNanoSeconds;
    nn::os::StartOneShotTimerEvent(&pTransmitterState->timerEvent, nn::TimeSpan::FromNanoSeconds(delayNs));
}

void TransmitterActionHandlerClearTimer(FiniteStateMachineActionValueType actCode, void* pContext, void* pExtra) NN_NOEXCEPT
{
    TransmitStateType*  pTransmitterState;

    pTransmitterState = static_cast<TransmitStateType*>(pContext);
    nn::os::StopTimerEvent(&pTransmitterState->timerEvent);
    nn::os::SignalTimerEvent(&pTransmitterState->timerEvent);
}

void TransmitterActionHandlerAbort(FiniteStateMachineActionValueType actCode, void* pContext, void* pExtra) NN_NOEXCEPT
{
    TransmitArgumentsType*  pArgs;

    pArgs = static_cast<TransmitArgumentsType*>(pExtra);
    if(pArgs != nullptr)
    {
        pArgs->transmitterReturnValue.ackNak = TransmitAckNakValueOther;
        pArgs->transmitterReturnValue.error = CecTransmitAbortError;
    }
}

void TransmitterActionHandlerXmit(FiniteStateMachineActionValueType actCode, void* pContext, void* pExtra) NN_NOEXCEPT
{
    TransmitStateType*      pTransmitterState;
    TransmitArgumentsType*  pArgs;

    pTransmitterState = static_cast<TransmitStateType*>(pContext);
    pArgs = static_cast<TransmitArgumentsType*>(pExtra);
    if(pArgs != nullptr)
    {
        if(TransmitterIsTransactionCancelled(pTransmitterState))
        {
            pArgs->transmitterReturnValue.ackNak = TransmitAckNakValueAck;
            pArgs->transmitterReturnValue.error = CecManagerImplCancelledError;
        }
        else
        {
            CecLowLevelInterfaceTransmitData(&pArgs->transmitterReturnValue, pArgs->pBuffer, pArgs->count);
        }
    }
}

void TransmitterActionHandlerWait(FiniteStateMachineActionValueType actCode, void* pContext, void* pExtra) NN_NOEXCEPT
{
    TransmitStateType*  pTransmitterState;

    pTransmitterState = static_cast<TransmitStateType*>(pContext);
    if(!TransmitterIsTransactionCancelled(pTransmitterState))
    {
        nn::os::WaitTimerEvent(&pTransmitterState->timerEvent);
    }
}

const FiniteStateMachineActionHandlerFunctionPointerType s_TransmitterActionHandlers[TransmitterActionNumber] =
{
    TransmitterActionHandlerStartTimer,
    TransmitterActionHandlerClearTimer,
    TransmitterActionHandlerAbort,
    TransmitterActionHandlerXmit,
    TransmitterActionHandlerWait
};

int32_t TransmitDataPacket(TransmitReturnValueType* pTransmitReturnValue, uint8_t* pBuffer, uint8_t count) NN_NOEXCEPT
{
    int32_t                 rval = CecNoError;
    TransmitArgumentsType   xmitArgs;

    NN_SDK_ASSERT(s_TransmitterState.flags & TransmitterStateFlagTransmitEnabledBit);
    if(count <= CecReceiveParamMaxMessageLength)
    {
        xmitArgs.transmitterReturnValue.error = CecNoError;
        xmitArgs.transmitterReturnValue.ackNak = TransmitAckNakValueOther;
        xmitArgs.pBuffer = pBuffer;
        xmitArgs.count = count;
        CecFiniteStateMachineStep(&s_TransmitterState.fsmState, TransmitterInputTransmit, (void*)&xmitArgs);
        *pTransmitReturnValue = xmitArgs.transmitterReturnValue;
    }
    else
    {
        rval = CecTransmitInvalidArgError;
        pTransmitReturnValue->error = CecTransmitInvalidArgError;
        pTransmitReturnValue->ackNak = TransmitAckNakValueOther;
    }
    return rval;
}

}

void CecTransmitInit() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!(s_TransmitterState.flags & TransmitterStateFlagTransmitEnabledBit));
    CecFiniteStateMachineInit(&s_TransmitterState.fsmState,
                              TransmitterStateNumber,
                              TransmitterActionNumber,
                              TransmitterInputNumber,
                              s_TransmitterStateMatrix,
                              s_TransmitterActionMatrix,
                              s_TransmitterActionHandlers,
                              static_cast<void*>(&s_TransmitterState));
    nn::os::InitializeMutex(&s_TransmitterState.apiMutex, false, 0);
    nn::os::InitializeMutex(&s_TransmitterState.flagMutex, false, 0);
    nn::os::InitializeTimerEvent(&s_TransmitterState.timerEvent, nn::os::EventClearMode_AutoClear);
    s_TransmitterState.flags = TransmitterStateFlagTransmitEnabledBit;
}

void    CecTransmitShutdown() NN_NOEXCEPT
{
    NN_SDK_ASSERT(s_TransmitterState.flags & TransmitterStateFlagTransmitEnabledBit);
    CecFiniteStateMachineStep(&s_TransmitterState.fsmState, TransmitterInputReset, nullptr);
    nn::os::FinalizeMutex(&s_TransmitterState.apiMutex);
    nn::os::FinalizeMutex(&s_TransmitterState.flagMutex);
    nn::os::FinalizeTimerEvent(&s_TransmitterState.timerEvent);
    s_TransmitterState.flags &= ~TransmitterStateFlagTransmitEnabledBit;
    s_TransmitterState.flags &= ~TransmitterStateFlagCancelledBit;
}

void    CecTransmitSetCancelState() NN_NOEXCEPT
{
    NN_SDK_ASSERT(s_TransmitterState.flags & TransmitterStateFlagTransmitEnabledBit);
    os::LockMutex(&s_TransmitterState.flagMutex);
    s_TransmitterState.flags |= TransmitterStateFlagCancelledBit;
    os::SignalTimerEvent(&s_TransmitterState.timerEvent);
    os::UnlockMutex(&s_TransmitterState.flagMutex);
}

void    CecTransmitClearCancelState() NN_NOEXCEPT
{
    NN_SDK_ASSERT(s_TransmitterState.flags & TransmitterStateFlagTransmitEnabledBit);
    os::LockMutex(&s_TransmitterState.flagMutex);
    s_TransmitterState.flags &= ~TransmitterStateFlagCancelledBit;
    os::UnlockMutex(&s_TransmitterState.flagMutex);
}

int32_t CecTransmitPollingRequest(TransmitReturnValueType* pReturnValue, uint8_t pollValue) NN_NOEXCEPT
{
    int32_t rval;
    int32_t attempts;
    int32_t maxNumberOfTimeout;

    NN_SDK_ASSERT(s_TransmitterState.flags & TransmitterStateFlagTransmitEnabledBit);
    attempts = TransmitterParameterPollingAttempts;
    maxNumberOfTimeout = TransmitterParameterMaximumNumberOfTimeoutInPolling;
    nn::os::LockMutex(&s_TransmitterState.apiMutex);
    do
    {
        attempts--;
        CecLowLevelInterfaceTransmitData(pReturnValue, &pollValue, 1);
        rval = pReturnValue->error;
        if(pReturnValue->error == CecNoError)
        {
            if(pReturnValue->ackNak == TransmitAckNakValueAck)
            {
                break;
            }
            else if(pReturnValue->ackNak == TransmitAckNakValueNak)
            {
                if(CecCancelSleepThreadMilliseconds(TransmitterParameterPollSignalFreeTimeMs))
                {
                    rval = CecManagerImplCancelledError;
                    pReturnValue->error = CecManagerImplCancelledError;
                    break;
                }
                else
                {
                    rval = CecTransmitNakError;
                }
            }
        }
        else if(pReturnValue->error == CecDisplayPortInterfaceTimeoutError)
        {
            if(maxNumberOfTimeout <= 0)
            {
                break;
            }
            maxNumberOfTimeout--;
            attempts++;
        }
        else
        {
            break;
        }
    } while(attempts > 0);
    nn::os::UnlockMutex(&s_TransmitterState.apiMutex);
    if(rval == CecManagerImplCancelledError)
    {
        NN_CEC_INFO("Canceled poll request\n");
    }
    return rval;
}

int32_t CecTransmitBestEffort(uint8_t* pBuffer, uint8_t count, bool isDirectAddressed) NN_NOEXCEPT
{
    int32_t                 rval = CecNoError;
    TransmitReturnValueType transmitterReturnValue;
    uint32_t                delayUs;
    uint32_t                attempts=0;

    NN_SDK_ASSERT(s_TransmitterState.flags & TransmitterStateFlagTransmitEnabledBit);
    nn::os::LockMutex(&s_TransmitterState.apiMutex);
    while(true)
    {
        bool    exitLoop = false;

        attempts++;
        rval = TransmitDataPacket(&transmitterReturnValue, pBuffer, count);
        switch(transmitterReturnValue.ackNak) {
            case TransmitAckNakValueAck:
                exitLoop = true;
                break;
            case TransmitAckNakValueNak:
                exitLoop = (attempts > TransmitterParameterBestEffortNakRetries);
                break;
            case TransmitAckNakValueOther:
                exitLoop = (attempts > TransmitterParameterBestEffortErrorRetries);
                break;
            default:
                exitLoop = true;
                break;
        }
        if(exitLoop)
        {
            break;
        }
        delayUs = CecComputePseudoRandomBroadcastDelayInMicroSeconds();
        if(isDirectAddressed)
        {
            // Empirically derived value, not required by CEC spec.
            delayUs /= 4;
        }
        if(CecCancelSleepThreadMilliseconds(delayUs / 1000))
        {
            rval = CecManagerImplCancelledError;
            break;
        }
    }
    if(rval == CecNoError)
    {
        rval = (transmitterReturnValue.ackNak == TransmitAckNakValueAck) ? transmitterReturnValue.error :
                                                                           CecTransmitCannotSendError;
    }
    if(rval == CecManagerImplCancelledError)
    {
        NN_CEC_INFO("Canceled best effort transmit\n");
    }
    nn::os::UnlockMutex(&s_TransmitterState.apiMutex);
    return rval;
}

}
}   // namespace cec
}   // namespace nn
