﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os/os_Config.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/os.h>
#include <nn/os/os_MutexTypes.h>
#include <nn/nn_SdkAssert.h>

#include "cec_Error.h"
#include "cec_FiniteStateMachine.h"

namespace nn { namespace cec { namespace detail {

void CecFiniteStateMachineInit(FiniteStateMachineType* pFsmState,
                               FiniteStateMachineStateType numStates,
                               FiniteStateMachineActionValueType numActions,
                               FiniteStateMachineInputType numInputs,
                               const FiniteStateMachineStateType* pStateMatrix,
                               const FiniteStateMachineActionType* pActionMatrix,
                               const FiniteStateMachineActionHandlerFunctionPointerType* pActionHandlers,
                               void* pContext) NN_NOEXCEPT
{
    NN_SDK_ASSERT(pFsmState != nullptr);
    NN_SDK_ASSERT(pStateMatrix != nullptr);
    NN_SDK_ASSERT(pActionMatrix != nullptr);
    NN_SDK_ASSERT(pActionHandlers != nullptr);
    NN_SDK_ASSERT(numStates <= FsmParamMaxStates);
    NN_SDK_ASSERT(numActions <= FsmParamMaxActions);
    NN_SDK_ASSERT(numInputs < FsmParamMaxInputs);
    pFsmState->fsmState = FsmParamDefaultState;
    pFsmState->fsmNumState = numStates;
    pFsmState->fsmNumAction = numActions;
    pFsmState->fsmNumInput = numInputs;
    pFsmState->fsmStateMatrix = pStateMatrix;
    pFsmState->fsmActionMatrix = pActionMatrix;
    pFsmState->fsmActionHandlers = pActionHandlers;
    pFsmState->fsmClientContext = pContext;
}

void CecFiniteStateMachineStep(FiniteStateMachineType* pFsmState, FiniteStateMachineInputType inputValue, void* pExtra) NN_NOEXCEPT
{
    FiniteStateMachineStateType           next;
    uint32_t            index;
    FiniteStateMachineActionType          actions;
    FiniteStateMachineActionHandlerFunctionPointerType   pActionHandlerFunction;

    NN_SDK_ASSERT((pFsmState != nullptr) && (inputValue <= FsmParamMaxInputs));
    index = static_cast<uint32_t>(pFsmState->fsmState);
    index *= static_cast<uint32_t>(pFsmState->fsmNumInput);
    index += static_cast<uint32_t>(inputValue);
    next = pFsmState->fsmStateMatrix[index];
    NN_SDK_ASSERT(next < pFsmState->fsmNumState);
    actions = pFsmState->fsmActionMatrix[index];
    for(int loopIdx=0; loopIdx<FiniteStateMachineConstantMaxActionCount; loopIdx++)
    {
        index = static_cast<uint32_t>(actions.FsmActions[loopIdx]);
        if(index < static_cast<uint32_t>(pFsmState->fsmNumAction))
        {
            pActionHandlerFunction = pFsmState->fsmActionHandlers[index];
            if(pActionHandlerFunction != nullptr)
            {
                pActionHandlerFunction(static_cast<FiniteStateMachineActionValueType>(index),
                                       pFsmState->fsmClientContext, pExtra);
            }
        }
        else
        {
            break;
        }
    }
    pFsmState->fsmState = next;
}

}
}   // namespace cec
}   // namespace nn
