﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/sf/sf_Types.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_HipcClientProxyByName.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/nn_Abort.h>
#include <nn/diag/detail/diag_DetailStructuredSdkLog.h>

#include <nn/cec/cec_Api.h>

#include <cec_ICecManager.h>
#include <cec_ServerName.h>
#include <cec_ManagerImpl.h>
#include <cec_Error.h>

#define NN_CEC_CLIENT_INFO(...)    NN_DETAIL_STRUCTURED_SDK_LOG(cec:Client, Info, 0, ##__VA_ARGS__)

namespace nn { namespace cec {

namespace {

struct CreateCecServerByHipcTag;
typedef nn::sf::ExpHeapStaticAllocator<1024 * 16, CreateCecServerByHipcTag> CecMainAllocator;
typedef nn::sf::ExpHeapStaticAllocator<1024 * 16, CreateCecServerByHipcTag> CecCancelAllocator;

nn::Result DecodeError(int32_t errorValue) NN_NOEXCEPT
{
    nn::Result  result;

    switch(errorValue)
    {
        case CecNoError:
            result = ResultSuccess();
            break;
        case CecManagerImplOperationFailed:
            result = ResultOperationFailed();
            break;
        case CecManagerImplSubsystemDisabled:
            result = ResultSubsystemDisabled();
            break;
        case CecManagerImplSubsystemSuspended:
            result = ResultSubsystemSuspended();
            break;
        case CecManagerImplCommandExecutionFailed:
            result = ResultCommandExecutionFailed();
            break;
        case CecManagerImplInvalidOperand:
            result = ResultInvalidOperand();
            break;
        case CecManagerImplTimeoutError:
            result = ResultTimeout();
            break;
        case CecManagerImplFeatureAbortError:
            result = ResultFeatureAbort();
            break;
        case CecManagerImplCancelledError:
            result = ResultCanceled();
            break;
        default:
            result = ResultOperationFailed();
            break;
    }
    return result;
}

class CecMainAllocatorInitializer
{
public:
    CecMainAllocatorInitializer() NN_NOEXCEPT
    {
        CecMainAllocator::Initialize(nn::lmem::CreationOption_NoOption);
    }
} s_CecMainAllocatorInitializer;

class CecCancelAllocatorInitializer
{
public:
    CecCancelAllocatorInitializer() NN_NOEXCEPT
    {
        CecCancelAllocator::Initialize(nn::lmem::CreationOption_NoOption);
    }
} s_CecCancelAllocatorInitializer;

const int32_t   MaxEventCount = 10;

class CecShimState
{
    public:
        CecShimState() NN_NOEXCEPT
        {
            int32_t index;

            for(index = 0; index < MaxEventCount; index++)
            {
                m_pCallbackSystemEvent[index] = nullptr;
                m_SystemEventHandle[index] = ~0;
            }
            m_CecManager.Reset();
            m_RefCount = 0;
            nn::os::InitializeMutex(&m_Mutex, false, 0);
            NN_CEC_CLIENT_INFO("%s: leave\n", NN_CURRENT_FUNCTION_NAME);
        }
        ~CecShimState() NN_NOEXCEPT
        {
            nn::os::FinalizeMutex(&m_Mutex);
            NN_CEC_CLIENT_INFO("%s: leave\n", NN_CURRENT_FUNCTION_NAME);
        }
        int AddReference(nn::os::SystemEventType* pAsynchronousSystemEvent) NN_NOEXCEPT
        {
            nn::Result              result;
            nn::sf::NativeHandle    eventHandle;
            int                     rval;
            int32_t                 index;

            nn::os::LockMutex(&m_Mutex);
            if(m_RefCount == 0)
            {
                NN_SDK_ASSERT(!m_CecManager);
                auto valueAndResult = nn::sf::CreateHipcProxyByName<ICecManager, CecMainAllocator::Policy>(CecServiceName);
                NN_ABORT_UNLESS_RESULT_SUCCESS(valueAndResult);
                m_CecManager = *valueAndResult;
                NN_SDK_ASSERT(m_CecManager);
            }
            index = AllocateSystemEventPointer(pAsynchronousSystemEvent);
            if(index != -1)
            {
                rval = m_CecManager->RegisterCallback(&eventHandle, &m_SystemEventHandle[index]);
                NN_SDK_ASSERT(rval == CecNoError);
                nn::os::AttachReadableHandleToSystemEvent(pAsynchronousSystemEvent, eventHandle.GetOsHandle(),
                                                          eventHandle.IsManaged(),
                                                          nn::os::EventClearMode_AutoClear);
                eventHandle.Detach();
                m_RefCount++;
            }
            else if(m_RefCount == 0)
            {
                m_CecManager.Reset();
            }
            rval = (index == -1) ? index : m_RefCount;
            nn::os::UnlockMutex(&m_Mutex);
            NN_CEC_CLIENT_INFO("%s: leave %d\n", NN_CURRENT_FUNCTION_NAME, rval);
            return rval;
        }
        int DelReference(nn::os::SystemEventType* pAsynchronousSystemEvent) NN_NOEXCEPT
        {
            int rval;
            int32_t index;

            nn::os::LockMutex(&m_Mutex);
            index = FindSystemEventPointer(pAsynchronousSystemEvent);
            if(index != -1)
            {
                Result  result;

                result = m_CecManager->TriggerSystemEvent(m_SystemEventHandle[index]);
                NN_ABORT_UNLESS(result.IsSuccess());
                rval = m_CecManager->UnregisterCallback(m_SystemEventHandle[index]);
                NN_SDK_ASSERT(rval == CecNoError);
                DeleteSystemEventPointer(pAsynchronousSystemEvent);
                m_RefCount--;
                if(m_RefCount == 0)
                {
                    NN_SDK_ASSERT(m_CecManager);
                    m_CecManager.Reset();
                }
            }
            rval = (index == -1) ? index : m_RefCount;
            nn::os::UnlockMutex(&m_Mutex);
            NN_CEC_CLIENT_INFO("%s: leave %d\n", NN_CURRENT_FUNCTION_NAME, rval);
            return rval;
        }
        nn::Result PerformActionClient(server::ActionType action, server::ShimDataTransferType* pShimData) NN_NOEXCEPT
        {
            int32_t rval;
            nn::Result  result;

            NN_SDK_ASSERT(m_CecManager);
            NN_CEC_CLIENT_INFO("%s: Enter\n", NN_CURRENT_FUNCTION_NAME);
            rval = m_CecManager->PerformAction(action, pShimData->ullArray[0], pShimData->ullArray[1]);
            result = DecodeError(rval);
            NN_CEC_CLIENT_INFO("%s: rval %X result value %d\n", NN_CURRENT_FUNCTION_NAME, rval, result.GetDescription());
            return result;
        }
        nn::Result QueryStateClient(server::QueryType query, server::ShimDataTransferType* pOutShimData) NN_NOEXCEPT
        {
            int32_t rval;
            nn::Result  result;

            NN_SDK_ASSERT(m_CecManager);
            NN_CEC_CLIENT_INFO("%s: Enter\n", NN_CURRENT_FUNCTION_NAME);
            rval = m_CecManager->QueryState(query, &pOutShimData->ullArray[0], &pOutShimData->ullArray[1]);
            result = DecodeError(rval);
            NN_CEC_CLIENT_INFO("%s: rval %X result value %d\n", NN_CURRENT_FUNCTION_NAME, rval, result.GetDescription());
            return result;
        }
        nn::sf::HipcRef<ICecManager> DuplicateSessionForCancel() NN_NOEXCEPT
        {
            nn::sf::HipcRef<ICecManager> p;
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_CecManager.template CloneSession<CecCancelAllocator::Policy>(&p, 1));
            return p;
        }
        nn::sf::HipcRef<ICecManager>& GetCecManagerObject() NN_NOEXCEPT
        {
            return m_CecManager;
        }

    private:
        int32_t FindSystemEventPointer(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
        {
            int32_t index;
            int32_t rval = -1;

            for(index = 0; index < MaxEventCount; index++)
            {
                if(m_pCallbackSystemEvent[index] == pSystemEvent)
                {
                    rval = index;
                    break;
                }
            }
            return rval;
        }
        int32_t DeleteSystemEventPointer(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
        {
            int32_t index;
            int32_t rval = -1;

            for(index = 0; index < MaxEventCount; index++)
            {
                if(m_pCallbackSystemEvent[index] == pSystemEvent)
                {
                    m_pCallbackSystemEvent[index] = nullptr;
                    rval = index;
                    break;
                }
            }
            return rval;
        }
        int32_t AllocateSystemEventPointer(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
        {
            int32_t index;
            int32_t rval = -1;

            for(index = 0; index < MaxEventCount; index++)
            {
                if(m_pCallbackSystemEvent[index] == nullptr)
                {
                    m_pCallbackSystemEvent[index] = pSystemEvent;
                    rval = index;
                    break;
                }
            }
            return rval;
        }

        nn::os::SystemEventType*            m_pCallbackSystemEvent[MaxEventCount];
        uint32_t                            m_SystemEventHandle[MaxEventCount];
        nn::sf::HipcRef<ICecManager>        m_CecManager;
        int                                 m_RefCount;
        nn::os::MutexType                   m_Mutex;

} s_CecShimState;

class CecCancelState
{
    public:
        CecCancelState() NN_NOEXCEPT
        {
            m_CecCancel = nullptr;
            m_RefCount = 0;
            nn::os::InitializeMutex(&m_Mutex, false, 0);
            NN_CEC_CLIENT_INFO("%s: leave\n", NN_CURRENT_FUNCTION_NAME);
        }
        ~CecCancelState() NN_NOEXCEPT
        {
            nn::os::FinalizeMutex(&m_Mutex);
            NN_CEC_CLIENT_INFO("%s: leave\n", NN_CURRENT_FUNCTION_NAME);
        }
        int AddReference() NN_NOEXCEPT
        {
            nn::Result              result;
            nn::sf::NativeHandle    eventHandle;
            int                     rval;

            nn::os::LockMutex(&m_Mutex);
            if(m_RefCount == 0)
            {
                NN_SDK_ASSERT(!m_CecCancel);
                m_CecCancel = s_CecShimState.DuplicateSessionForCancel();
                NN_SDK_ASSERT(m_CecCancel);
            }
            m_RefCount++;
            rval = m_RefCount;
            nn::os::UnlockMutex(&m_Mutex);
            NN_CEC_CLIENT_INFO("%s: leave %d\n", NN_CURRENT_FUNCTION_NAME, rval);
            return rval;
        }
        int DelReference() NN_NOEXCEPT
        {
            int rval;

            nn::os::LockMutex(&m_Mutex);
            m_RefCount--;
            if(m_RefCount == 0)
            {
                NN_SDK_ASSERT(m_CecCancel);
                m_CecCancel = nullptr;
            }
            rval = m_RefCount;
            nn::os::UnlockMutex(&m_Mutex);
            NN_CEC_CLIENT_INFO("%s: leave %d\n", NN_CURRENT_FUNCTION_NAME, rval);
            return rval;
        }
        nn::Result CancelClient(bool* pOutApiWasCanceled) NN_NOEXCEPT
        {
            nn::Result  result;
            int         rval;

            NN_SDK_ASSERT(m_CecCancel);
            NN_CEC_CLIENT_INFO("%s: Enter\n", NN_CURRENT_FUNCTION_NAME);
            rval = m_CecCancel->CancelCurrentCall(pOutApiWasCanceled);
            result = DecodeError(rval);
            NN_CEC_CLIENT_INFO("%s: rval %X result value %d\n", NN_CURRENT_FUNCTION_NAME, rval, result.GetDescription());
            return result;
        }
        nn::Result OnEventClient(BusEventType* pEvent) NN_NOEXCEPT
        {
            int32_t     rval;
            uint32_t    event;
            nn::Result  result;
            uint32_t    count;
            uint64_t    ullArray[2];

            NN_SDK_ASSERT(m_CecCancel);
            NN_CEC_CLIENT_INFO("%s: Enter\n", NN_CURRENT_FUNCTION_NAME);
            rval = m_CecCancel->OnSystemEvent(&event, &count, &ullArray[0], &ullArray[1]);
            result = DecodeError(rval);
            NN_CEC_CLIENT_INFO("%s: rval %X result value %d\n", NN_CURRENT_FUNCTION_NAME, rval, result.GetDescription());
            *pEvent = static_cast<BusEventType>(event);
            return result;
        }
    private:
        nn::sf::SharedPointer<ICecManager>  m_CecCancel;
        int                                 m_RefCount;
        nn::os::MutexType                   m_Mutex;
} s_CecCancelState;

}

nn::Result Initialize(nn::os::SystemEventType* pOutAsynchronousSystemEvent) NN_NOEXCEPT
{
    int rval;
    nn::Result  result;

    NN_SDK_REQUIRES_NOT_NULL(pOutAsynchronousSystemEvent);
    rval = s_CecShimState.AddReference(pOutAsynchronousSystemEvent);
    s_CecCancelState.AddReference();
    result = (rval == -1) ? ResultOperationFailed() : ResultSuccess();
    return result;
}

nn::Result Finalize(nn::os::SystemEventType* pAsynchronousSystemEvent) NN_NOEXCEPT
{
    int rval;
    nn::Result  result;

    NN_SDK_REQUIRES_NOT_NULL(pAsynchronousSystemEvent);
    rval = s_CecShimState.DelReference(pAsynchronousSystemEvent);
    result = (rval == -1) ? ResultOperationFailed() : ResultSuccess();
    s_CecCancelState.DelReference();
    return result;
}

nn::Result PerformOneTouchPlay() NN_NOEXCEPT
{
    server::ShimDataTransferType    shimData;

    shimData.integerValue = -1;
    return(s_CecShimState.PerformActionClient(server::ActionType_OneTouchPlay, &shimData));
}

nn::Result PerformImageViewOn() NN_NOEXCEPT
{
    server::ShimDataTransferType    shimData;

    shimData.integerValue = -1;
    return(s_CecShimState.PerformActionClient(server::ActionType_ImageViewOn, &shimData));
}

nn::Result PerformActiveSource() NN_NOEXCEPT
{
    server::ShimDataTransferType    shimData;

    shimData.integerValue = -1;
    return(s_CecShimState.PerformActionClient(server::ActionType_ActiveSource, &shimData));
}

nn::Result SetActiveSourceState() NN_NOEXCEPT
{
    server::ShimDataTransferType    shimData;

    shimData.integerValue = -1;
    return(s_CecShimState.PerformActionClient(server::ActionType_SetActiveSourceState, &shimData));
}

nn::Result PerformGoStandby(bool tvOnly) NN_NOEXCEPT
{
    server::ShimDataTransferType    shimData;

    shimData.booleanValue = tvOnly;
    return(s_CecShimState.PerformActionClient(server::ActionType_GoStandby, &shimData));
}

nn::Result PerformInactiveSource() NN_NOEXCEPT
{
    server::ShimDataTransferType    shimData;

    shimData.integerValue = -1;
    return(s_CecShimState.PerformActionClient(server::ActionType_InactiveSource, &shimData));
}

nn::Result PerformSendDeviceMenuStateCommand(bool activate) NN_NOEXCEPT
{
    server::ShimDataTransferType    shimData;

    shimData.booleanValue = activate;
    return(s_CecShimState.PerformActionClient(server::ActionType_DeviceMenuCommand, &shimData));
}

nn::Result PerformSendRemoteControlCommand(bool press, RemoteControlCommand remoteCommandValue) NN_NOEXCEPT
{
    server::ShimDataTransferType    shimData;

    shimData.byteArray[0] = static_cast<uint8_t>(press);
    shimData.byteArray[1] = static_cast<uint8_t>(remoteCommandValue);
    return(s_CecShimState.PerformActionClient(server::ActionType_RemoteControlCommand, &shimData));
}

nn::Result SetOnScreenString(const char* string) NN_NOEXCEPT
{
    server::ShimDataTransferType    shimData;
    uint8_t                         maxCount = detail::CecReceiveParamMaxMessageLength;
    uint8_t                         index = 0;
    char                            currentCharacter;

    do
    {
        currentCharacter = string[index];
        shimData.byteArray[index] = currentCharacter;
        index++;
    } while((currentCharacter != '\0') && (index < maxCount));
    return(s_CecShimState.PerformActionClient(server::ActionType_SetOnScreenDisplay, &shimData));
}

nn::Result SetPowerState(PowerState powerStateValue) NN_NOEXCEPT
{
    server::ShimDataTransferType    shimData;

    shimData.powerStateValue = powerStateValue;
    return(s_CecShimState.PerformActionClient(server::ActionType_SetPowerState, &shimData));
}

nn::Result SuspendManager() NN_NOEXCEPT
{
    server::ShimDataTransferType    shimData;

    shimData.integerValue = -1;
    return(s_CecShimState.PerformActionClient(server::ActionType_SuspendManager, &shimData));
}

nn::Result RestartManager() NN_NOEXCEPT
{
    server::ShimDataTransferType    shimData;

    shimData.integerValue = -1;
    return(s_CecShimState.PerformActionClient(server::ActionType_RestartManager, &shimData));
}

nn::Result CancelCurrentApiCall(bool* pOutIsCanceled) NN_NOEXCEPT
{
    return s_CecCancelState.CancelClient(pOutIsCanceled);
}

nn::Result IsStarted(bool* pOutIsStarted) NN_NOEXCEPT
{
    server::ShimDataTransferType    shimData;
    nn::Result                      result;

    NN_SDK_REQUIRES_NOT_NULL(pOutIsStarted);
    result = s_CecShimState.QueryStateClient(server::QueryType_IfStarted, &shimData);
    *pOutIsStarted = shimData.booleanValue;
    return result;
}

nn::Result GetPowerState(PowerState* pOutPowerState) NN_NOEXCEPT
{
    server::ShimDataTransferType    shimData;
    nn::Result                      result;

    NN_SDK_REQUIRES_NOT_NULL(pOutPowerState);
    result = s_CecShimState.QueryStateClient(server::QueryType_PowerState, &shimData);
    *pOutPowerState = shimData.powerStateValue;
    return result;
}

nn::Result GetTvPowerState(PowerState* pOutPowerState) NN_NOEXCEPT
{
    server::ShimDataTransferType    shimData;
    nn::Result                      result;

    NN_SDK_REQUIRES_NOT_NULL(pOutPowerState);
    result = s_CecShimState.QueryStateClient(server::QueryType_TvPowerState, &shimData);
    *pOutPowerState = shimData.powerStateValue;
    return result;
}

nn::Result IsTvResponsive(bool* pOutIsSuccessfullyPinged) NN_NOEXCEPT
{
    server::ShimDataTransferType    shimData;
    nn::Result                      result;

    NN_SDK_REQUIRES_NOT_NULL(pOutIsSuccessfullyPinged);
    result = s_CecShimState.QueryStateClient(server::QueryType_CanPingTv, &shimData);
    *pOutIsSuccessfullyPinged = shimData.booleanValue;
    return result;
}

nn::Result GetConnectionState(ConnectionState* pOutConnectionState) NN_NOEXCEPT
{
    server::ShimDataTransferType    shimData;
    nn::Result                      result;

    NN_SDK_REQUIRES_NOT_NULL(pOutConnectionState);
    result = s_CecShimState.QueryStateClient(server::QueryType_ConnectionState, &shimData);
    *pOutConnectionState = shimData.connectionStateValue;
    return result;
}

nn::Result GetHpdState(bool* pOutHpdState) NN_NOEXCEPT
{
    server::ShimDataTransferType    shimData;
    nn::Result                      result;

    NN_SDK_REQUIRES_NOT_NULL(pOutHpdState);
    result = s_CecShimState.QueryStateClient(server::QueryType_HpdState, &shimData);
    *pOutHpdState = shimData.booleanValue;
    return result;
}

nn::Result IsEnabled(bool* pOutIsEnabled) NN_NOEXCEPT
{
    server::ShimDataTransferType    shimData;
    nn::Result                      result;

    NN_SDK_REQUIRES_NOT_NULL(pOutIsEnabled);
    result = s_CecShimState.QueryStateClient(server::QueryType_IfEnabled, &shimData);
    *pOutIsEnabled = shimData.booleanValue;
    return result;
}

nn::Result IsActiveSource(bool* pOutIsActiveSource) NN_NOEXCEPT
{
    server::ShimDataTransferType    shimData;
    nn::Result                      result;

    NN_SDK_REQUIRES_NOT_NULL(pOutIsActiveSource);
    result = s_CecShimState.QueryStateClient(server::QueryType_IfActiveSource, &shimData);
    *pOutIsActiveSource = shimData.booleanValue;
    return result;
}

nn::Result GetBusEventType(BusEventType* pOutEventType) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutEventType);
    return(s_CecCancelState.OnEventClient(pOutEventType));
}

nn::Result GetHdcpServiceObject(nn::sf::SharedPointer<nn::hdcp::detail::IHdcpController>* pOutValue) NN_NOEXCEPT
{
    return s_CecShimState.GetCecManagerObject()->GetHdcpServiceObject(pOutValue);
}

}}
