﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   UAC attach implementation
 */

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Assert.h>

#include <nn/cduac/cduac_Spec.h>
#include <nn/cduac/cduac_Api.h>

namespace nn {
namespace cduac {

//////////////////////////////////////////////////////////////////////////////
Result Parser::CreateInterfaceProfile(InterfaceProfile* pOutInterfaceProfile, Host* pHost, nn::usb::InterfaceQueryOutput* pInInterfaceQueryOutput) NN_NOEXCEPT
{
    NN_SDK_ASSERT(pOutInterfaceProfile);
    NN_SDK_ASSERT(pHost);
    NN_SDK_ASSERT(pInInterfaceQueryOutput);

    memset(m_CtrlBuffer, 0, 4096);
    memset(pOutInterfaceProfile, 0, sizeof(InterfaceProfile));
    memcpy(&pOutInterfaceProfile->usbInterfaceParams, pInInterfaceQueryOutput, sizeof(nn::usb::InterfaceQueryOutput));

    Result result = m_CtrlSession.Initialize(pHost->GetUsbHost(), pInInterfaceQueryOutput->ifProfile.handle);

    if (result.IsSuccess())
    {
        result = ParseDescriptors(pOutInterfaceProfile, pInInterfaceQueryOutput);

        // Print warning if we truncated descriptor buffer
        if (result.IsSuccess())
        {
            if (pOutInterfaceProfile->descriptorsSize >= DescriptorCountMax)
            {
                NN_SDK_LOG("Warning: InterfaceProfile descriptors truncated %d >= %d\n", pOutInterfaceProfile->descriptorsSize, DescriptorCountMax);
            }
        }

        m_CtrlSession.Finalize();
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
AudioControlInputTerminal* Parser::GetInputTerminal(InterfaceProfile* pInterfaceProfile) NN_NOEXCEPT
{
    uint8_t *p = pInterfaceProfile->descriptors;

    while (p)
    {
        if (GetDescriptorLength(p) == sizeof(AudioControlInputTerminal))
        {
            if (GetDescriptorType(p) == DescriptorType::DescriptorType_Interface)
            {
                if (GetAudioDescriptorSubType(p) == AudioControl_InputTerminal)
                {
                    return reinterpret_cast<AudioControlInputTerminal*>(p);
                }
            }
        }

        p = GetNextDescriptor(p);
    }

    return NULL;
}


//////////////////////////////////////////////////////////////////////////////
AudioControlOutputTerminal* Parser::GetOutputTerminal(InterfaceProfile* pInterfaceProfile) NN_NOEXCEPT
{
    uint8_t *p = pInterfaceProfile->descriptors;

    while (p)
    {
        if (GetDescriptorLength(p) == sizeof(AudioControlOutputTerminal))
        {
            if (GetDescriptorType(p) == DescriptorType::DescriptorType_Interface)
            {
                if (GetAudioDescriptorSubType(p) == AudioControl_OutputTerminal)
                {
                    return reinterpret_cast<AudioControlOutputTerminal*>(p);
                }
            }
        }

        p = GetNextDescriptor(p);
    }

    return NULL;
}


//////////////////////////////////////////////////////////////////////////////
AudioControlFeatureUnit* Parser::GetFeatureUnit(InterfaceProfile* pInterfaceProfile) NN_NOEXCEPT
{
    uint8_t *p = pInterfaceProfile->descriptors;

    while (p)
    {
        if (GetDescriptorType(p) == DescriptorType::DescriptorType_Interface)
        {
            if (GetAudioDescriptorSubType(p) == AudioControl_FeatureUnit)
            {
                return reinterpret_cast<AudioControlFeatureUnit*>(p);
            }
        }

        p = GetNextDescriptor(p);
    }

    return NULL;
}


//////////////////////////////////////////////////////////////////////////////
nn::usb::UsbInterfaceDescriptor* Parser::GetInterface(InterfaceProfile* pInterfaceProfile, uint8_t bAlternateSetting) NN_NOEXCEPT
{
    uint8_t *p = pInterfaceProfile->descriptors;

    while (p)
    {
        if (IsUsbInterfaceDescriptor(p))
        {
            if (p[3] == bAlternateSetting)
            {
                return reinterpret_cast<nn::usb::UsbInterfaceDescriptor*>(p);
            }
        }

        p = GetNextDescriptor(p);
    }

    return NULL;
}


//////////////////////////////////////////////////////////////////////////////
AudioStreamingGeneral* Parser::GetAudioStreamingGeneral(InterfaceProfile* pInterfaceProfile, uint8_t bAlternateSetting) NN_NOEXCEPT
{
    uint8_t *p = reinterpret_cast<uint8_t*>(GetInterface(pInterfaceProfile, bAlternateSetting));

    while (p)
    {
        if (GetDescriptorType(p) == DescriptorType::DescriptorType_Interface)
        {
            if (GetAudioDescriptorSubType(p) == AudioStreaming_General)
            {
                return reinterpret_cast<AudioStreamingGeneral*>(p);
            }
        }

        p = GetNextDescriptor(p);

        // If we parsed to the next interface descriptror then break out
        if (IsUsbInterfaceDescriptor(p))
        {
            break;
        }
    }

    return NULL;
}


//////////////////////////////////////////////////////////////////////////////
AudioStreamingFormatType* Parser::GetAudioStreamingFormatType(InterfaceProfile* pInterfaceProfile, uint8_t bAlternateSetting) NN_NOEXCEPT
{
    uint8_t *p = reinterpret_cast<uint8_t*>(GetInterface(pInterfaceProfile, bAlternateSetting));

    while (p)
    {
        if (GetDescriptorType(p) == DescriptorType::DescriptorType_Interface)
        {
            if (GetAudioDescriptorSubType(p) == AudioStreaming_FormatType)
            {
                return reinterpret_cast<AudioStreamingFormatType*>(p);
            }
        }

        p = GetNextDescriptor(p);

        // If we parsed to the next interface descriptror then break out
        if (IsUsbInterfaceDescriptor(p))
        {
            break;
        }
    }

    return NULL;
}


//////////////////////////////////////////////////////////////////////////////
nn::usb::UsbEndpointDescriptor* Parser::GetEndpoint(InterfaceProfile* pInterfaceProfile, uint8_t bAlternateSetting, uint8_t endpointIndex) NN_NOEXCEPT
{
    uint8_t index = 0;
    uint8_t *p = reinterpret_cast<uint8_t*>(GetInterface(pInterfaceProfile, bAlternateSetting));

    while (p)
    {
        if (GetDescriptorType(p) == nn::usb::UsbDescriptorType_Endpoint)
        {
            if (index == endpointIndex)
            {
                return reinterpret_cast<nn::usb::UsbEndpointDescriptor*>(p);
            }

            index++;
        }

        p = GetNextDescriptor(p);

        // If we parsed to the next interface descriptror then break out
        if (IsUsbInterfaceDescriptor(p))
        {
            break;
        }
    }

    return NULL;
}


//////////////////////////////////////////////////////////////////////////////
AudioStreamingEndpointGeneral* Parser::GetAudioStreamingEndpointGeneral(InterfaceProfile* pInterfaceProfile, uint8_t bAlternateSetting, uint8_t endpointIndex) NN_NOEXCEPT
{
    uint8_t *p = reinterpret_cast<uint8_t*>(GetEndpoint(pInterfaceProfile, bAlternateSetting, endpointIndex));

    while (p)
    {
        if (GetDescriptorType(p) == DescriptorType::DescriptorType_Endpoint)
        {
            if (GetAudioDescriptorSubType(p) == AudioStreaming_Endpoint_General)
            {
                return reinterpret_cast<AudioStreamingEndpointGeneral*>(p);
            }
        }

        p = GetNextDescriptor(p);

        // If we parsed to the next interface descriptror then break out
        if (IsUsbInterfaceDescriptor(p))
        {
            break;
        }
    }

    return NULL;
}


//////////////////////////////////////////////////////////////////////////////
//  private functions
//////////////////////////////////////////////////////////////////////////////
uint8_t Parser::GetAudioControlInterfaceNumber(uint8_t *p) NN_NOEXCEPT
{
    while (p)
    {
        if (
            (GetDescriptorLength(p) == sizeof(nn::usb::UsbInterfaceDescriptor)) &&
            (GetDescriptorType(p) == nn::usb::UsbDescriptorType_Interface)
            )
        {
            nn::usb::UsbInterfaceDescriptor *pUsbInterfaceDescriptor = reinterpret_cast<nn::usb::UsbInterfaceDescriptor*>(p);

            if (
                (pUsbInterfaceDescriptor->bInterfaceClass == nn::usb::UsbClass::UsbClass_Audio) &&
                (pUsbInterfaceDescriptor->bInterfaceSubClass == nn::cduac::AudioSubClassCode::SubClass_AudioControl)
                )
            {
                return pUsbInterfaceDescriptor->bInterfaceNumber;
            }
        }

        p = GetNextDescriptor(p);
    }

    return 0;
}


//////////////////////////////////////////////////////////////////////////////
Result Parser::ParseDescriptors(InterfaceProfile* pOutInterfaceProfile, nn::usb::InterfaceQueryOutput* pInterface) NN_NOEXCEPT
{
    Result result;
    uint8_t bTerminalLink = 0;
    size_t size;
    uint16_t totalBytes = pInterface->deviceProfile.cfgDesc.wTotalLength;
    NN_SDK_ASSERT(totalBytes <= 4096);
    result = m_CtrlSession.ControlRequest(&size, m_CtrlBuffer, 0x80, 0x06, 0x0200, 0, totalBytes);

    if (result.IsSuccess() && (size == totalBytes))
    {
        pOutInterfaceProfile->audioControlInterfaceNumber = GetAudioControlInterfaceNumber(m_CtrlBuffer);

        // Gather descriptors for this interface
        for (uint32_t alternateSetting = 1; alternateSetting <= 0xff; alternateSetting++)
        {
            uint8_t bAlternateSetting =  static_cast<uint8_t>(alternateSetting);
            nn::usb::UsbInterfaceDescriptor *pUsbInterfaceDescriptor = GetInterfaceDescriptor(pInterface->ifProfile.ifDesc.bInterfaceNumber, bAlternateSetting, m_CtrlBuffer);

            if (pUsbInterfaceDescriptor)
            {
                uint8_t *p = reinterpret_cast<uint8_t*>(pUsbInterfaceDescriptor);

                // Lets store all the descriptors up to the next interface descriptor or end
                while (p)
                {
                    StoreDescriptor(pOutInterfaceProfile, p);

                    // AS_GENERAL descriptor carries association to input or output termnal
                    if (GetDescriptorType(p) == DescriptorType_Interface)
                    {
                        if (GetAudioDescriptorSubType(p) == AudioStreaming_General)
                        {
                            bTerminalLink = p[3];
                        }
                    }

                    p = GetNextDescriptor(p);

                    if (IsUsbInterfaceDescriptor(p))
                    {
                        p = nullptr;
                    }
                }
            }
            else
            {
                break;
            }
        }

        // Start by getting the terminal specified by bTermainlLink
        AudioDescriptor *pAudioDescriptor = GetNode(bTerminalLink, m_CtrlBuffer);

        if (pAudioDescriptor)
        {
            // Trace pin
            switch (GetAudioDescriptorSubType(pAudioDescriptor))
            {
            case AudioControl_InputTerminal:

                TraceInputToOutput(pAudioDescriptor, pOutInterfaceProfile);

                break;

            case AudioControl_OutputTerminal:

                TraceOutputToInput(pAudioDescriptor, pOutInterfaceProfile);

                break;

            default:

                // don't care

                break;
            }
        }
    }

    return result;
}


///////////////////////////////////////////////////////////////////////////////
uint8_t Parser::GetNodeId(AudioDescriptor *pAudioDescriptor) NN_NOEXCEPT
{
    uint8_t nodeId = 0;

    if (GetDescriptorType(pAudioDescriptor) == DescriptorType_Interface)
    {
        switch (pAudioDescriptor->bDescriptorSubtype)
        {
            case AudioControl_InputTerminal:
            case AudioControl_OutputTerminal:
            case AudioControl_MixerUnit:
            case AudioControl_SelectorUnit:
            case AudioControl_FeatureUnit:
            case AudioControl_ProcessingUnit:
            case AudioControl_ExtensionUnit:

                {
                    uint8_t *p = reinterpret_cast<uint8_t*>(pAudioDescriptor);

                    nodeId = p[3];
                }

                break;

            default:

                // don't care

                break;
        }
    }

    return nodeId;
}


///////////////////////////////////////////////////////////////////////////////
AudioDescriptor * Parser::GetReferencingNode(uint8_t sourceId, uint8_t *p) NN_NOEXCEPT
{
    AudioDescriptor *pAudioDescriptor = reinterpret_cast<AudioDescriptor*>(GetNextDescriptor(p));

    while (pAudioDescriptor)
    {
        if (pAudioDescriptor->bDescriptorType == DescriptorType_Interface)
        {
            switch (pAudioDescriptor->bDescriptorSubtype)
            {
                case AudioControl_OutputTerminal:

                    {
                        AudioControlOutputTerminal *pAudioControlOutputTerminal = reinterpret_cast<AudioControlOutputTerminal*>(pAudioDescriptor);

                        if (pAudioControlOutputTerminal->bSourceId == sourceId)
                        {
//NN_SDK_LOG("found output id %d\n", sourceId);
                            return pAudioDescriptor;
                        }
                    }

                    break;

                case AudioControl_MixerUnit:
                case AudioControl_SelectorUnit:

                    {
                        uint8_t *pu8 = reinterpret_cast<uint8_t*>(pAudioDescriptor);

                        // n pins associated with n source ids
                        for (uint8_t i = 0; i < pu8[4]; i++)
                        {
                            if (pu8[5 + i] == sourceId)
                            {
//NN_SDK_LOG("found mixer or selector source id %d id %d\n", sourceId, pu8[3]);
                                return pAudioDescriptor;
                            }
                        }
                    }

                    break;

                case AudioControl_FeatureUnit:

                    {
                        AudioControlFeatureUnit *pAudioControlFeatureUnit = reinterpret_cast<AudioControlFeatureUnit*>(pAudioDescriptor);

                        if (pAudioControlFeatureUnit->bSourceId ==sourceId)
                        {
//NN_SDK_LOG("found feature source id %d id %d\n", sourceId, pAudioControlFeatureUnit->bUnitId);
                            return pAudioDescriptor;
                        }
                    }

                    break;

                case AudioControl_ProcessingUnit:
                case AudioControl_ExtensionUnit:

                    {
                        uint8_t *pu8 = reinterpret_cast<uint8_t*>(pAudioDescriptor);

                        // n pins associated with n source ids
                        for (uint8_t i = 0; i < pu8[6]; i++)
                        {
                            if (pu8[7 + i] == sourceId)
                            {
//NN_SDK_LOG("found processing or extension id %d id %d\n", sourceId, pu8[3]);
                                return pAudioDescriptor;
                            }
                        }
                    }

                    break;

                default:

                    // don't care

                    break;
            }
        }

        pAudioDescriptor = reinterpret_cast<AudioDescriptor*>(GetNextDescriptor(pAudioDescriptor));
    }

    return NULL;
}


///////////////////////////////////////////////////////////////////////////////
uint8_t Parser::GetSourceId(AudioDescriptor *pAudioDescriptor) NN_NOEXCEPT
{
    uint8_t sourceId = 0;

    switch (pAudioDescriptor->bDescriptorSubtype)
    {
        case AudioControl_OutputTerminal:

            {
                AudioControlOutputTerminal *pAudioControlOutputTerminal = reinterpret_cast<AudioControlOutputTerminal*>(pAudioDescriptor);
                sourceId = pAudioControlOutputTerminal->bSourceId;
            }

            break;

        case AudioControl_SelectorUnit:

            {
                AudioControlSelectorUnit *pAudioControlSelectorUnit = reinterpret_cast<AudioControlSelectorUnit*>(pAudioDescriptor);
                sourceId = pAudioControlSelectorUnit->baSourceId[0];    // Assume selection 0 for now
            }

            break;

        case AudioControl_FeatureUnit:

            {
                AudioControlFeatureUnit *pAudioControlFeatureUnit = reinterpret_cast<AudioControlFeatureUnit*>(pAudioDescriptor);
                sourceId = pAudioControlFeatureUnit->bSourceId;
            }

            break;
        case AudioControl_MixerUnit:

            {
                AudioControlMixerUnit *pAudioControlMixerUnit = reinterpret_cast<AudioControlMixerUnit*>(pAudioDescriptor);
                sourceId = pAudioControlMixerUnit->baSourceId[0];
            }

            break;
        case AudioControl_ProcessingUnit:
        case AudioControl_ExtensionUnit:
        default:

            // don't care

            break;
    }

    return sourceId;
}


///////////////////////////////////////////////////////////////////////////////
AudioDescriptor * Parser::GetNode(uint8_t id, uint8_t *p) NN_NOEXCEPT
{
    AudioDescriptor *pAudioDescriptor = reinterpret_cast<AudioDescriptor*>(GetNextDescriptor(p));

    while (pAudioDescriptor)
    {
        if (GetNodeId(pAudioDescriptor) == id)
        {
            return pAudioDescriptor;
        }

        pAudioDescriptor = reinterpret_cast<AudioDescriptor*>(GetNextDescriptor(pAudioDescriptor));
    }

    return NULL;
}


///////////////////////////////////////////////////////////////////////////////
void Parser::TraceInputToOutput(AudioDescriptor *pAudioDescriptor, InterfaceProfile *pOutInterfaceProfile) NN_NOEXCEPT
{
    // Trace nodes until output terminal is encountered
    const int MaxIterations = 256;

    int itr = 0;
    while (pAudioDescriptor && itr < MaxIterations)
    {
        ++itr;
        StoreDescriptor(pOutInterfaceProfile, pAudioDescriptor);

        uint8_t nodeId = GetNodeId(pAudioDescriptor);
        if(nodeId == 0)
        {
            break;
        }

        pAudioDescriptor = GetReferencingNode(GetNodeId(pAudioDescriptor), m_CtrlBuffer);

        if (pAudioDescriptor)
        {
            if (pAudioDescriptor->bDescriptorSubtype == AudioControl_OutputTerminal)
            {
                StoreDescriptor(pOutInterfaceProfile, pAudioDescriptor);
                break;
            }
        }
    }
}


///////////////////////////////////////////////////////////////////////////////
void Parser::TraceOutputToInput(AudioDescriptor *pAudioDescriptor, InterfaceProfile *pOutInterfaceProfile) NN_NOEXCEPT
{
    // Trace nodes until input terminal is encountered
    const int MaxIterations = 256;

    int itr = 0;
    while (pAudioDescriptor && itr < MaxIterations)
    {
        ++itr;
        StoreDescriptor(pOutInterfaceProfile, pAudioDescriptor);

        uint8_t sourceId = GetSourceId(pAudioDescriptor);
        if(sourceId == 0)
        {
            break;
        }
        pAudioDescriptor = GetNode(sourceId, m_CtrlBuffer);

        if (pAudioDescriptor)
        {
            if (pAudioDescriptor->bDescriptorSubtype == AudioControl_InputTerminal)
            {
                StoreDescriptor(pOutInterfaceProfile, pAudioDescriptor);
                break;
            }
        }
    }
}


///////////////////////////////////////////////////////////////////////////////
void Parser::StoreDescriptor(InterfaceProfile *pInterfaceProfile, void *p) NN_NOEXCEPT
{
    uint8_t bytes = GetDescriptorLength(p);

    // Make sure we are in bounds
    if ((pInterfaceProfile->descriptorsSize + bytes) < DescriptorCountMax)
    {
        memcpy(&pInterfaceProfile->descriptors[pInterfaceProfile->descriptorsSize], p, GetDescriptorLength(p));
    }

    pInterfaceProfile->descriptorsSize += bytes;
}


///////////////////////////////////////////////////////////////////////////////
uint8_t * Parser::GetNextDescriptor(void *p) NN_NOEXCEPT
{
    uint8_t *p1 = reinterpret_cast<uint8_t*>(p);

    p1 += *p1;

    // Make sure the descriptor chain is 0 terminated or there will be hell to pay :0
    if (*p1)
    {
        return p1;
    }

    return 0;
}


///////////////////////////////////////////////////////////////////////////////
uint8_t Parser::GetDescriptorLength(void *p) NN_NOEXCEPT
{
    AudioDescriptor *pAudioDescriptor = reinterpret_cast<AudioDescriptor*>(p);

    return pAudioDescriptor->bLength;
}


///////////////////////////////////////////////////////////////////////////////
uint8_t Parser::GetDescriptorType(void *p) NN_NOEXCEPT
{
    AudioDescriptor *pAudioDescriptor = reinterpret_cast<AudioDescriptor*>(p);

    return pAudioDescriptor->bDescriptorType;
}


///////////////////////////////////////////////////////////////////////////////
uint8_t Parser::GetAudioDescriptorSubType(void *p) NN_NOEXCEPT
{
    AudioDescriptor *pAudioDescriptor = reinterpret_cast<AudioDescriptor*>(p);

    return pAudioDescriptor->bDescriptorSubtype;
}


///////////////////////////////////////////////////////////////////////////////
nn::usb::UsbInterfaceDescriptor* Parser::GetInterfaceDescriptor(uint8_t bInterfaceNumber, uint8_t bAletrnateSetting, uint8_t *p) NN_NOEXCEPT
{
    while (p)
    {
        if (IsUsbInterfaceDescriptor(p) && (p[2] == bInterfaceNumber))
        {
            if (p[3] == bAletrnateSetting)
            {
                return reinterpret_cast<nn::usb::UsbInterfaceDescriptor*>(p);
            }
        }

        p = GetNextDescriptor(p);
    }

    return NULL;
}


///////////////////////////////////////////////////////////////////////////////
bool Parser::IsUsbInterfaceDescriptor(uint8_t *p) NN_NOEXCEPT
{
    if (p)
    {
        if (GetDescriptorType(p) == nn::usb::UsbDescriptorType_Interface)
        {
            return true;
        }
    }

    return false;
}


} // end of namespace cduac
} // end of namespace nn
