﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   UAC interface implementation
 */

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Assert.h>

#include <nn/cduac/cduac_Spec.h>
#include <nn/cduac/cduac_Api.h>

namespace nn {
namespace cduac {


//////////////////////////////////////////////////////////////////////////////
Result Interface::Initialize(Host* pHost, InterfaceProfile* pInInterfaceProfile) NN_NOEXCEPT
{
    Result result = m_UsbHostInterface.Initialize(pHost->GetUsbHost(), pInInterfaceProfile->usbInterfaceParams.ifProfile.handle);

    if (result.IsSuccess())
    {
        nn::os::InitializeMutex(&m_CtrlBufferMutex, false, 0);

        memcpy(&m_InterfaceProfile, pInInterfaceProfile, sizeof(InterfaceProfile));

        m_UsbHostInterface.SetInterface(0);
        m_AlternateSetting = 0;
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result Interface::Finalize() NN_NOEXCEPT
{
    if (m_AlternateSetting)
    {
        m_UsbHostEndpoint.Finalize();
    }

    m_UsbHostInterface.Finalize();

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
nn::os::SystemEventType* Interface::GetStateChangeEvent() NN_NOEXCEPT
{
    return m_UsbHostInterface.GetStateChangeEvent();
}


//////////////////////////////////////////////////////////////////////////////
InterfaceProfile* Interface::GetInterfaceProfile() NN_NOEXCEPT
{
    return &m_InterfaceProfile;
}


//////////////////////////////////////////////////////////////////////////////
Result Interface::SetAudioControl(uint8_t bRequest, uint16_t wValue, uint8_t id, uint16_t wLength, void *pData) NN_NOEXCEPT
{
    Result result;
    size_t bytesTransferred;

    nn::os::LockMutex(&m_CtrlBufferMutex);

    if (pData && wLength)
    {
        memcpy(m_CtrlBuffer, pData, wLength);
    }

    result = m_UsbHostInterface.ControlRequest(
                                                &bytesTransferred,
                                                m_CtrlBuffer,
                                                RequestType_SetAudioControl,
                                                bRequest,
                                                wValue,
                                                id << 8 | m_InterfaceProfile.audioControlInterfaceNumber,
                                                wLength
                                                );

    nn::os::UnlockMutex(&m_CtrlBufferMutex);

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result Interface::GetAudioControl(uint8_t bRequest, uint16_t wValue, uint8_t id, uint16_t wLength, void *pData) NN_NOEXCEPT
{
    Result result;
    size_t bytesTransferred;

    nn::os::LockMutex(&m_CtrlBufferMutex);

    result = m_UsbHostInterface.ControlRequest(
                                                &bytesTransferred,
                                                m_CtrlBuffer,
                                                RequestType_GetAudioControl,
                                                bRequest,
                                                wValue,
                                                id << 8 | m_InterfaceProfile.audioControlInterfaceNumber,
                                                wLength
                                                );

    if (pData && wLength)
    {
        memcpy(pData, m_CtrlBuffer, wLength);
    }

    nn::os::UnlockMutex(&m_CtrlBufferMutex);

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result Interface::SetEndpointControl(uint8_t bRequest, uint8_t cs, uint8_t endpoint, uint16_t wLength, void *pData) NN_NOEXCEPT
{
    Result result;
    size_t bytesTransferred;

    nn::os::LockMutex(&m_CtrlBufferMutex);

    if (pData && wLength)
    {
        memcpy(m_CtrlBuffer, pData, wLength);
    }

    result = m_UsbHostInterface.ControlRequest(
                                                &bytesTransferred,
                                                m_CtrlBuffer,
                                                RequestType_SetAudioStreaming,
                                                bRequest,
                                                cs << 8,
                                                endpoint,
                                                wLength
                                                );

    nn::os::UnlockMutex(&m_CtrlBufferMutex);

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result Interface::GetEndpointControl(uint8_t bRequest, uint8_t cs, uint8_t endpoint, uint16_t wLength, void *pData) NN_NOEXCEPT
{
    Result result;
    size_t bytesTransferred;

    nn::os::LockMutex(&m_CtrlBufferMutex);

    result = m_UsbHostInterface.ControlRequest(
                                                &bytesTransferred,
                                                m_CtrlBuffer,
                                                RequestType_GetAudioStreaming,
                                                bRequest,
                                                cs << 8,
                                                endpoint,
                                                wLength
                                                );

    if (pData && wLength)
    {
        memcpy(pData, m_CtrlBuffer, wLength);
    }

    nn::os::UnlockMutex(&m_CtrlBufferMutex);

    return result;
}

//////////////////////////////////////////////////////////////////////////////
Result Interface::RegisterBuffer(void *smmuBuffer,   uint32_t smmuBufferSize,
                                 void *reportBuffer, uint32_t reportBufferSize) NN_NOEXCEPT
{
    Result result;

    if (smmuBuffer)
    {
        result = m_UsbHostEndpoint.CreateSmmuSpace(smmuBuffer, smmuBufferSize);
        if (result.IsFailure())
        {
            return result;
        }
    }

    if (reportBuffer)
    {
        result = m_UsbHostEndpoint.ShareReportRing(reportBuffer, reportBufferSize);
        if (result.IsFailure())
        {
            return result;
        }
    }

    return ResultSuccess();
}

//////////////////////////////////////////////////////////////////////////////
Result Interface::SetAltSetting(uint8_t alternateSetting, nn::usb::UsbEndpointDescriptor *pUsbEndpointDescriptor, uint32_t maxUrb, uint16_t maxXferSize) NN_NOEXCEPT
{
    Result result;

    if (alternateSetting)
    {
        result = m_UsbHostInterface.SetInterface(alternateSetting);

        if (result.IsSuccess())
        {
            result = m_UsbHostEndpoint.Initialize(
                                                    &m_UsbHostInterface,
                                                    pUsbEndpointDescriptor,
                                                    maxUrb,
                                                    maxXferSize
                                                    );
        }
    }
    else
    {
        if (m_UsbHostEndpoint.IsInitialized())
        {
            m_UsbHostEndpoint.Finalize();
        }

        // If m_UsbHostEndpoint is not initialized, we still have to set alt setting to 0
        result = m_UsbHostInterface.SetInterface(alternateSetting);
    }

    if (result.IsSuccess())
    {
        m_AlternateSetting = alternateSetting;
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result Interface::PostTransferAsync(void *pBuffer, uint32_t *pBytes, uint32_t usbFrames, uint32_t startOnFrame, uint32_t context) NN_NOEXCEPT
{
    uint32_t id;
    return m_UsbHostEndpoint.BatchBufferAsync(
                                                &id,
                                                pBuffer,
                                                pBytes,
                                                usbFrames,
                                                context,
                                                nn::usb::SchedulePolicy::SchedulePolicy_Absolute,
                                                startOnFrame
                                                );
}


//////////////////////////////////////////////////////////////////////////////
Result Interface::GetXferReport(uint32_t *pOutCount, nn::usb::XferReport *pOutReport, uint32_t count) NN_NOEXCEPT
{
    return m_UsbHostEndpoint.GetXferReport(pOutCount, pOutReport, count);
}


//////////////////////////////////////////////////////////////////////////////
os::SystemEventType* Interface::GetCompletionEvent() NN_NOEXCEPT
{
    return m_UsbHostEndpoint.GetCompletionEvent();
}


//////////////////////////////////////////////////////////////////////////////
Result Interface::GetCurrentFrame(uint32_t *pFrameNumber) NN_NOEXCEPT
{
    return m_UsbHostInterface.GetCurrentFrame(pFrameNumber);
}


} // end of namespace cduac
} // end of namespace nn
