﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief
 *
 * @details
 */
#include "cdmsc_PrivateIncludes.h"

#include "cdmsc_Scsi.h"

namespace nn {
namespace cdmsc {
namespace driver {

// Forward declaration
class Device;

class LogicalUnit
{
public:
    LogicalUnit(Device& device, uint16_t vid, uint16_t pid,
                uint8_t lun, uint32_t capability);
    virtual ~LogicalUnit();

    void* operator new(size_t size) NN_NOEXCEPT;
    void  operator delete(void* p, size_t size) NN_NOEXCEPT;

    Result   Initialize();
    Result   Finalize();

    void       SetHandle(UnitHandle handle) NN_NOEXCEPT;
    UnitHandle GetHandle() const NN_NOEXCEPT;
    uint32_t   GetCapability() const NN_NOEXCEPT;
    void       GetProfile(UnitProfile *pOutProfile) const NN_NOEXCEPT;

    Result   Read(uint64_t lba, uint32_t block, void *pData) NN_NOEXCEPT;
    Result   Write(uint64_t lba, uint32_t block, const void *pData) NN_NOEXCEPT;
    Result   Flush(uint64_t lba, uint32_t block) NN_NOEXCEPT;
    void     KeepAlive();

protected:
    virtual Result TestUnitReady() NN_NOEXCEPT = 0;
    virtual Result RequestSense() NN_NOEXCEPT = 0;
    virtual Result ReadCapacity10() NN_NOEXCEPT = 0;
    virtual Result ReadCapacity16() NN_NOEXCEPT = 0;
    virtual Result ModeSense10() NN_NOEXCEPT = 0;
    virtual Result StartStopUnit(bool start) NN_NOEXCEPT = 0;
    virtual Result SyncCache10(uint32_t lba, uint16_t block) NN_NOEXCEPT = 0;
    virtual Result Read10(uint32_t lba, uint16_t block, void *pData) NN_NOEXCEPT = 0;
    virtual Result Write10(uint32_t lba, uint16_t block, const void *pData) NN_NOEXCEPT = 0;
    virtual Result Read16(uint64_t lba, uint32_t block, void *pData) NN_NOEXCEPT = 0;
    virtual Result Write16(uint64_t lba, uint32_t block, const void *pData) NN_NOEXCEPT = 0;
    virtual Result Inquiry() NN_NOEXCEPT = 0;

    Device&                       m_Device;

    uint8_t                       m_Lun;
    uint32_t                      m_Capability;
    uint32_t                      m_BlockSize;
    uint64_t                      m_BlockCount;

    const char                   *m_Subclass;

    uint8_t                      *m_pDmaBuffer;
    uint8_t                       m_SenseData[ScsiSenseDataMaxLength];
    ScsiModeParameterHeader10     m_ModeParamHeader;
    ScsiInquiryData               m_InquiryData;

protected:
    Result ProtocolRead(void *pCmd, uint8_t cmdLength,
                        void *buffer, uint32_t size, uint32_t *pOutXferred) NN_NOEXCEPT;
    Result ProtocolWrite(void *pCmd, uint8_t cmdLength,
                         const void *buffer, uint32_t size, uint32_t *pOutXferred) NN_NOEXCEPT;

private:
    static const uint32_t         MaxRetry = 3;
    static const uint32_t         RetryIntervalInMs = 1000;
    static const uint32_t         KeepAlivePeriodInMs = 2 * 60 * 1000;
    static const uint32_t         MinBlockSize = 512;
    static const uint32_t         MaxBlockSize = 1024 * 8;
    static const size_t           DmaBufferSize = 1024 * 8;

    NN_STATIC_ASSERT(DmaBufferSize >= MaxBlockSize);

    UnitHandle                    m_Handle;
    uint16_t                      m_Vid;
    uint16_t                      m_Pid;
    bool                          m_IsWriteProtected;
    uint32_t                      m_MaxBlocksPerXfer;
    bool                          m_Is64Bit;

    nn::os::TimerEvent            m_KeepAliveTimer;
    nn::os::MultiWaitHolderType   m_KeepAliveTimerHolder;

private:
    void   ArmKeepAliveTimer();
    void   DisarmKeepAliveTimer();

    bool   IsNotReady();
    bool   IsStartUnitRequired();
    bool   IsMediumNotPresent();
    bool   IsIllegalRequest();
    bool   IsWriteProtected();

    Result WaitReady() NN_NOEXCEPT;
    Result IdentifyCapacity() NN_NOEXCEPT;
    Result IdentifyMode() NN_NOEXCEPT;
    Result TestSyncCommand() NN_NOEXCEPT;
    Result AdminRead() NN_NOEXCEPT;
    Result AttemptRWRecovery(uint64_t lba, uint32_t block, void *pData, bool isRead) NN_NOEXCEPT;
    Result SenseKeyToResult() NN_NOEXCEPT;
};

class ScsiLogicalUnit : public LogicalUnit
{
public:
    ScsiLogicalUnit(Device& device, uint16_t vid, uint16_t pid,
                    uint8_t lun, uint32_t capability);
    virtual ~ScsiLogicalUnit();
protected:
    virtual Result TestUnitReady() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result RequestSense() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result ReadCapacity10() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result ReadCapacity16() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result ModeSense10() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result StartStopUnit(bool start) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result SyncCache10(uint32_t lba, uint16_t block) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Read10(uint32_t lba, uint16_t block, void *pData) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Write10(uint32_t lba, uint16_t block, const void *pData) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Read16(uint64_t lba, uint32_t block, void *pData) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Write16(uint64_t lba, uint32_t block, const void *pData) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Inquiry() NN_NOEXCEPT NN_OVERRIDE;
};

class UfiLogicalUnit : public LogicalUnit
{
public:
    UfiLogicalUnit(Device& device, uint16_t vid, uint16_t pid,
                   uint8_t lun, uint32_t capability);
    virtual ~UfiLogicalUnit();
protected:
    virtual Result TestUnitReady() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result RequestSense() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result ReadCapacity10() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result ReadCapacity16() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result ModeSense10() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result StartStopUnit(bool start) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result SyncCache10(uint32_t lba, uint16_t block) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Read10(uint32_t lba, uint16_t block, void *pData) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Write10(uint32_t lba, uint16_t block, const void *pData) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Read16(uint64_t lba, uint32_t block, void *pData) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Write16(uint64_t lba, uint32_t block, const void *pData) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Inquiry() NN_NOEXCEPT NN_OVERRIDE;
};

} // driver
} // cdmsc
} // nn
