﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "cdmsc_LogicalUnit.h"
#include "cdmsc_Device.h"

namespace nn {
namespace cdmsc {
namespace driver {

LogicalUnit::LogicalUnit(Device& device, uint16_t vid, uint16_t pid,
                         uint8_t lun, uint32_t capability)
    : m_Device(device)
    , m_Lun(lun)
    , m_Capability(capability)
    , m_BlockSize(0)
    , m_BlockCount(0)
    , m_Subclass("")
    , m_pDmaBuffer(nullptr)
    , m_Handle(0)
    , m_Vid(vid)
    , m_Pid(pid)
    , m_IsWriteProtected(false)
    , m_MaxBlocksPerXfer(0xFFFF)
    , m_Is64Bit(false)
    , m_KeepAliveTimer(nn::os::EventClearMode_ManualClear)
{
    std::memset(m_SenseData,        0, sizeof(m_SenseData));
    std::memset(&m_ModeParamHeader, 0, sizeof(m_ModeParamHeader));

    nn::os::InitializeMultiWaitHolder(
        &m_KeepAliveTimerHolder, m_KeepAliveTimer.GetBase()
    );
    nn::os::SetMultiWaitHolderUserData(
        &m_KeepAliveTimerHolder, reinterpret_cast<intptr_t>(this)
    );
}

LogicalUnit::~LogicalUnit()
{
    nn::os::FinalizeMultiWaitHolder(&m_KeepAliveTimerHolder);
}

// We MUST overload operator new because Device is over-aligned.
void*
LogicalUnit::operator new(size_t size) NN_NOEXCEPT
{
    return detail::Allocate(alignof(LogicalUnit), size);
}

void
LogicalUnit::operator delete(void* p, size_t size)NN_NOEXCEPT
{
    detail::Deallocate(p, size);
}

Result
LogicalUnit::Initialize()
{
    Result result = ResultSuccess();

    m_pDmaBuffer = reinterpret_cast<uint8_t*>(
        detail::Allocate(nn::usb::HwLimitDmaBufferAlignmentSize, DmaBufferSize)
    );
    if (m_pDmaBuffer == nullptr)
    {
        return ResultMemoryAllocError();
    }

    m_Device.Lock();

    do {
        NN_CDMSC_BREAK_UPON_ERROR(WaitReady());
        NN_CDMSC_BREAK_UPON_ERROR(IdentifyCapacity());
        NN_CDMSC_BREAK_UPON_ERROR(Inquiry());
        NN_CDMSC_BREAK_UPON_ERROR(IdentifyMode());
        NN_CDMSC_BREAK_UPON_ERROR(TestSyncCommand());
        NN_CDMSC_BREAK_UPON_ERROR(AdminRead());
    } while(0);

    if (result.IsSuccess())
    {
        m_Device.RegisterKeepAliveTimer(m_KeepAliveTimerHolder);
        ArmKeepAliveTimer();
    }
    else
    {
        detail::Deallocate(m_pDmaBuffer, DmaBufferSize);
    }

    m_Device.Unlock();

    return result;
}

Result
LogicalUnit::Finalize()
{
    Result result = ResultSuccess();

    DisarmKeepAliveTimer();

    m_Device.UnregisterKeepAliveTimer(m_KeepAliveTimerHolder);

    detail::Deallocate(m_pDmaBuffer, DmaBufferSize);

    return result;
}

void
LogicalUnit::SetHandle(UnitHandle handle) NN_NOEXCEPT
{
    m_Handle = handle;
}

UnitHandle
LogicalUnit::GetHandle() const NN_NOEXCEPT
{
    return m_Handle;
}

uint32_t
LogicalUnit::GetCapability() const NN_NOEXCEPT
{
    return m_Capability;
}

void
LogicalUnit::GetProfile(UnitProfile *pOutProfile) const NN_NOEXCEPT
{
    pOutProfile->handle           = m_Handle;
    pOutProfile->vid              = m_Vid;
    pOutProfile->pid              = m_Pid;
    pOutProfile->blockSize        = m_BlockSize;
    pOutProfile->blockCount       = m_BlockCount;
    pOutProfile->isWriteProtected = m_IsWriteProtected;
}

Result
LogicalUnit::AttemptRWRecovery(uint64_t lba, uint32_t block,
                               void *pData, bool isRead) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    if (IsWriteProtected())
    {
        m_IsWriteProtected = true;
        return ResultWriteProtected();
    }

    if (IsStartUnitRequired())
    {
        result = StartStopUnit(true);
        if (result.IsFailure())
        {
            return result;
        }

        result = WaitReady();

        if (result.IsFailure())
        {
            return result;
        }

        if (isRead)
        {
            if (m_Is64Bit == false)
            {
                result = Read10(lba, block, pData);
            }
            else
            {
                result = Read16(lba, block, pData);
            }
        }
        else
        {
            if (m_Is64Bit == false)
            {
                result = Write10(lba, block, pData);
            }
            else
            {
                result = Write16(lba, block, pData);
            }
        }
    }
    return result;
}

Result
LogicalUnit::Read(uint64_t lba, uint32_t block, void *pData) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    if (util::is_aligned(reinterpret_cast<uintptr_t>(pData), ::nn::usb::HwLimitDmaBufferAlignmentSize) == false)
    {
        return ResultMisalignedBuffer();
    }

    if (lba >= m_BlockCount)
    {
        return ResultInvalidLba();
    }

    if (lba + block > m_BlockCount)
    {
        return ResultInvalidBlockCount();
    }

    if (block == 0)
    {
        return ResultInvalidBlockCount();
    }

    m_Device.Lock();

    do
    {
        uint32_t blocksToXfer = 0;

        blocksToXfer = block > m_MaxBlocksPerXfer ? m_MaxBlocksPerXfer : block;

        if (m_Is64Bit == false)
        {
            result = Read10(lba, blocksToXfer, pData);
        }
        else
        {
            result = Read16(lba, blocksToXfer, pData);
        }

        if (ResultMscCommandFailed::Includes(result))
        {
            result = AttemptRWRecovery(lba, blocksToXfer, pData, true);
        }

        if (result.IsFailure())
        {
            break;
        }

        block -= blocksToXfer;
        lba += blocksToXfer;
        pData = (void *)((uint8_t *)pData + blocksToXfer * m_BlockSize);
    } while (block > 0);

    ArmKeepAliveTimer();

    m_Device.Unlock();

    if (ResultMscCommandFailed::Includes(result))
    {
        result = SenseKeyToResult();
    }

    return result;
}

Result
LogicalUnit::Write(uint64_t lba, uint32_t block, const void *pData) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    if (util::is_aligned(reinterpret_cast<uintptr_t>(pData), ::nn::usb::HwLimitDmaBufferAlignmentSize) == false)
    {
        return ResultMisalignedBuffer();
    }

    if (m_IsWriteProtected)
    {
        return ResultWriteProtected();
    }

    if (block == 0)
    {
        return ResultInvalidBlockCount();
    }

    if (lba >= m_BlockCount)
    {
        return ResultInvalidLba();
    }

    if (lba + block > m_BlockCount)
    {
        return ResultInvalidBlockCount();
    }

    m_Device.Lock();

    do
    {
        uint32_t blocksToXfer = 0;

        blocksToXfer = block > m_MaxBlocksPerXfer ? m_MaxBlocksPerXfer : block;

        if (m_Is64Bit == false)
        {
            result = Write10(lba, blocksToXfer, pData);
        }
        else
        {
            result = Write16(lba, blocksToXfer, pData);
        }

        if (ResultMscCommandFailed::Includes(result))
        {
            result = AttemptRWRecovery(lba, blocksToXfer, (void*)pData, false);
        }

        if (result.IsFailure())
        {
            break;
        }

        block -= blocksToXfer;
        lba += blocksToXfer;
        pData = (void *)((uint8_t *)pData + blocksToXfer * m_BlockSize);
    } while (block > 0);

    ArmKeepAliveTimer();

    m_Device.Unlock();

    if (ResultMscCommandFailed::Includes(result))
    {
        result = SenseKeyToResult();
    }

    return result;
}

Result
LogicalUnit::Flush(uint64_t lba, uint32_t block) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    if (lba >= m_BlockCount)
    {
        return ResultInvalidLba();
    }

    if (lba + block > m_BlockCount)
    {
        return ResultInvalidBlockCount();
    }

    m_Device.Lock();

    if (m_Capability & DeviceCapability_SyncCache10)
    {
        result = SyncCache10(lba, block);

        // FIXME: Check SenseData

        ArmKeepAliveTimer();
    }
    else
    {
        result = ResultUnsupportedOperation();
    }

    m_Device.Unlock();

    if (ResultMscCommandFailed::Includes(result))
    {
        result = SenseKeyToResult();
    }

    return result;
}

void
LogicalUnit::KeepAlive()
{
    m_Device.Lock();

    AdminRead();

    ArmKeepAliveTimer();

    m_Device.Unlock();
}

Result
LogicalUnit::ProtocolRead(void *pCmd, uint8_t cmdLength,
                          void *buffer, uint32_t size, uint32_t *pOutXferred) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    result = m_Device.BotRead(m_Lun, pCmd, cmdLength, buffer, size, pOutXferred);
    if (ResultCommandProtocolError::Includes(result))
    {
        // Autosense
        result = RequestSense();
        if (result.IsSuccess())
        {
            result = ResultMscCommandFailed();
        }
    }

    return result;
}

Result
LogicalUnit::ProtocolWrite(void *pCmd, uint8_t cmdLength,
                           const void *buffer, uint32_t size, uint32_t *pOutXferred) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    result = m_Device.BotWrite(m_Lun, pCmd, cmdLength, buffer, size, pOutXferred);
    if (ResultCommandProtocolError::Includes(result))
    {
        // Autosense
        result = RequestSense();
        if (result.IsSuccess())
        {
            result = ResultMscCommandFailed();
        }
    }

    return result;
}

void
LogicalUnit::ArmKeepAliveTimer()
{
    m_KeepAliveTimer.Clear();

    m_KeepAliveTimer.StartOneShot(
        nn::TimeSpan::FromMilliSeconds(KeepAlivePeriodInMs)
    );
}

void
LogicalUnit::DisarmKeepAliveTimer()
{
    m_KeepAliveTimer.Stop();
}

bool
LogicalUnit::IsNotReady()
{
    ScsiRequestSenseData *pSense;

    pSense = reinterpret_cast<ScsiRequestSenseData*>(m_SenseData);

    return
        pSense->filemark_eom_ili_senseKey == 2 &&
        pSense->additionalSenseCode == 4;
}


bool
LogicalUnit::IsStartUnitRequired()
{
    ScsiRequestSenseData *pSense;

    pSense = reinterpret_cast<ScsiRequestSenseData*>(m_SenseData);

    return
        pSense->filemark_eom_ili_senseKey    == 2 &&
        pSense->additionalSenseCode          == 4 &&
        pSense->additionalSenseCodeQualifier == 2;
}

bool
LogicalUnit::IsMediumNotPresent()
{
    ScsiRequestSenseData *pSense;

    pSense = reinterpret_cast<ScsiRequestSenseData*>(m_SenseData);

    return
        pSense->filemark_eom_ili_senseKey    == 2 &&
        pSense->additionalSenseCode          == 0x3a;
}

bool
LogicalUnit::IsIllegalRequest()
{
    ScsiRequestSenseData *pSense;

    pSense = reinterpret_cast<ScsiRequestSenseData*>(m_SenseData);

    return pSense->filemark_eom_ili_senseKey == 5;
}

bool
LogicalUnit::IsWriteProtected()
{
    ScsiRequestSenseData *pSense;

    pSense = reinterpret_cast<ScsiRequestSenseData*>(m_SenseData);

    return pSense->filemark_eom_ili_senseKey == 0x07;
}

Result
LogicalUnit::WaitReady() NN_NOEXCEPT
{
    Result result = ResultSuccess();

    do
    {
        result = TestUnitReady();

        if (ResultMscCommandFailed::Includes(result))
        {
            if (IsStartUnitRequired())
            {
                // TODO: revisit - cafe ignores command error
                result = StartStopUnit(true);
                if (result.IsFailure())
                {
                    break;
                }
            }
            else if (IsMediumNotPresent())
            {
                result = ResultNoMedium();
                break;
            }
            else
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(RetryIntervalInMs));
            }
        }
        else
        {
            break;
        }
    } while (IsNotReady());

    return result;
}

Result
LogicalUnit::IdentifyCapacity() NN_NOEXCEPT
{
    Result result = ResultSuccess();

    // TODO: revisit - cafe could do infinite retry
    for (uint32_t i = 0; i < MaxRetry; i++)
    {
        result = ReadCapacity10();

        if (result.IsSuccess())
        {
            if (m_BlockSize >= MinBlockSize && m_BlockSize <= MaxBlockSize)
            {
                break;
            }
        }
        result = ReadCapacity16();
        if (result.IsSuccess())
        {
            if (m_BlockSize < MinBlockSize || m_BlockSize > MaxBlockSize)
            {
                result = ResultUnsupportedDevice();
            }
            else
            {
                m_Is64Bit = true;
                break;
            }
        }
    }

    if (result.IsSuccess())
    {
        m_MaxBlocksPerXfer = nn::usb::HsLimitMaxUrbTransferSize / m_BlockSize;

        if (m_Is64Bit == false && m_MaxBlocksPerXfer >= 0x10000)
        {
            m_MaxBlocksPerXfer = 0xFFFF;
        }
    }

    return result;
}

Result
LogicalUnit::IdentifyMode() NN_NOEXCEPT
{
    Result result = ResultSuccess();

    if (!(m_Capability & DeviceCapability_ModeSense10))
    {
        return result;
    }

    result = ModeSense10();

    if (result.IsSuccess())
    {
        m_IsWriteProtected = m_ModeParamHeader.wp_dpofua & 0x80 ? true : false;
    }
    else if (ResultMscCommandFailed::Includes(result))
    {
        if (IsIllegalRequest())
        {
            m_Capability &= ~DeviceCapability_ModeSense10;
        }

        result = ResultSuccess();
    }
    else
    {
        m_Capability &= ~DeviceCapability_ModeSense10;
    }

    return result;
}

Result
LogicalUnit::TestSyncCommand() NN_NOEXCEPT
{
    Result result = ResultSuccess();

    if (!(m_Capability & DeviceCapability_SyncCache10))
    {
        return result;
    }

    result = SyncCache10(0, 1);

    if (ResultMscCommandFailed::Includes(result))
    {
        if (IsIllegalRequest())
        {
            m_Capability &= ~DeviceCapability_SyncCache10;
        }

        result = ResultSuccess();
    }
    else if (result.IsFailure())
    {
       m_Capability &= ~DeviceCapability_SyncCache10;
    }

    return result;
}

Result
LogicalUnit::AdminRead() NN_NOEXCEPT
{
    Result result = ResultSuccess();

    // TODO: revisit - cafe could do infinite retry
    for (uint32_t i = 0; i < MaxRetry; i++)
    {
        result = Read10(0, 1, m_pDmaBuffer);

        if (ResultMscCommandFailed::Includes(result))
        {
            if (IsStartUnitRequired())
            {
                // TODO: revisit
                //  - cafe ignores command error
                //  - cafe restart from WaitReady
                result = StartStopUnit(true);
                if (result.IsFailure())
                {
                    break;
                }
            }
            else
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(RetryIntervalInMs));
            }
        }
        else
        {
            break;
        }
    }

    return result;
}

Result
LogicalUnit::SenseKeyToResult()
{
    ScsiRequestSenseData *pSense;

    pSense = reinterpret_cast<ScsiRequestSenseData*>(m_SenseData);

    switch (pSense->filemark_eom_ili_senseKey)
    {
    case 0:
    case 1: return ResultSuccess();
    case 2: return ResultNotReady();
    case 3: return ResultMediumError();
    case 4: return ResultHardwareError();
    case 5:
    {
        switch (pSense->additionalSenseCode)
        {
        case 0x21: return ResultInvalidLba();
        case 0x25: return ResultNoMedium();
        default: return ResultIllegalRequest();
        }
    }
    case 7: return ResultWriteProtected();
    default: return ResultMscCommandFailed();
    }
}


ScsiLogicalUnit::ScsiLogicalUnit(Device& device, uint16_t vid, uint16_t pid,
                                 uint8_t lun, uint32_t capability)
    : LogicalUnit(device, vid, pid, lun, capability)
{
    m_Subclass = "SCSI";
}

ScsiLogicalUnit::~ScsiLogicalUnit()
{
}

Result
ScsiLogicalUnit::TestUnitReady() NN_NOEXCEPT
{
    Result   result = ResultSuccess();
    uint32_t xferredSize;

    ScsiTestUnitReady cmd = {
        .opCode  = ScsiCommand_TestUnitReady,
        .control = 0
    };

    result = ProtocolRead(
        &cmd, sizeof(cmd), m_pDmaBuffer, 0, &xferredSize
    );

    return result;
}

Result
ScsiLogicalUnit::RequestSense() NN_NOEXCEPT
{
    Result   result = ResultSuccess();
    uint32_t xferredSize = 0;

    ScsiRequestSense cmd = {
        .opCode           = ScsiCommand_RequestSense,
        .desc             = 0,
        .allocationLength = ScsiSenseDataMaxLength,
        .control          = 0
    };

    result = m_Device.BotRead(
        m_Lun, (uint8_t*)&cmd, sizeof(cmd),
        m_pDmaBuffer, ScsiSenseDataMaxLength, &xferredSize
    );

    if (result.IsSuccess())
    {
        std::memcpy(m_SenseData, m_pDmaBuffer, xferredSize);
    }

    return result;
}

Result
ScsiLogicalUnit::ReadCapacity10() NN_NOEXCEPT
{
    Result   result = ResultSuccess();
    uint32_t xferredSize = 0;

    ScsiReadCapacity10 cmd = {
        .opCode = ScsiCommand_ReadCapacity10,
        .lba = 0,
        .pmi = 0,
        .control = 0
    };
    ScsiReadCapacity10Data *pData = reinterpret_cast<ScsiReadCapacity10Data*>(m_pDmaBuffer);

    result = ProtocolRead(
        &cmd, sizeof(cmd),
        m_pDmaBuffer, sizeof(ScsiReadCapacity10Data), &xferredSize
    );

    if (result.IsSuccess())
    {
        if (pData->lba != 0xFFFFFFFF)
        {
            m_BlockCount = detail::BeToCpu32(pData->lba) + 1;
            m_BlockSize = detail::BeToCpu32(pData->blockLength);
        }
    }

    return result;
}

Result
ScsiLogicalUnit::ReadCapacity16() NN_NOEXCEPT
{
    Result   result = ResultSuccess();
    uint32_t xferredSize = 0;

    ScsiReadCapacity16 cmd = {
        .opCode           = ScsiCommand_ReadCapacity16,
        .serviceAction    = 0x10,
        .lba              = 0,
        .allocationLength = detail::CpuToBe32(sizeof(ScsiReadCapacity16Data)),
        .pmi              = 0,
        .control          = 0
    };
    ScsiReadCapacity16Data *pData = reinterpret_cast<ScsiReadCapacity16Data*>(m_pDmaBuffer);

    NN_UNUSED(pData);

    result = ProtocolRead(
        &cmd, sizeof(cmd),
        m_pDmaBuffer, sizeof(ScsiReadCapacity16Data), &xferredSize
    );

    if (result.IsSuccess())
    {
        if (pData->lba != 0xFFFFFFFFFFFFFFFF)
        {
            m_BlockCount = detail::BeToCpu64(pData->lba) + 1;
            m_BlockSize = detail::BeToCpu32(pData->blockLength);
        }
    }

    return result;
}

Result
ScsiLogicalUnit::ModeSense10() NN_NOEXCEPT
{
    Result   result = ResultSuccess();
    uint32_t xferredSize = 0;

    ScsiModeSense10 cmd = {
        .opCode           = ScsiCommand_ModeSense10,
        .llbaa_dbd        = 0, // FIXME:
        .pc_pageCode      = 0x3f, // current value, all pages
        .subpageCode      = 0,
        .allocationLength = detail::CpuToBe16(sizeof(ScsiModeParameterHeader10)),
        .control          = 0
    };

    result = ProtocolRead(
        &cmd, sizeof(cmd),
        m_pDmaBuffer, sizeof(ScsiModeParameterHeader10), &xferredSize
    );

    if (result.IsSuccess())
    {
        std::memcpy(
            &m_ModeParamHeader, m_pDmaBuffer, sizeof(ScsiModeParameterHeader10)
        );
    }

    return result;
}

Result
ScsiLogicalUnit::StartStopUnit(bool start) NN_NOEXCEPT
{
    Result   result = ResultSuccess();
    uint32_t xferredSize = 0;

    ScsiStartStopUnit cmd = {
        .opCode                            = ScsiCommand_StartStopUnit,
        .immed                             = 0,
        .powerConditionModifier            = 0,
        .powerCondition_noFlush_loej_start = static_cast<uint8_t>(start ? 1 : 0),
        .control                           = 0,
    };

    result = ProtocolRead(
        &cmd, sizeof(cmd), m_pDmaBuffer, 0, &xferredSize
    );

    return result;
}

Result
ScsiLogicalUnit::SyncCache10(uint32_t lba, uint16_t block) NN_NOEXCEPT
{
    Result   result = ResultSuccess();
    uint32_t xferredSize = 0;

    ScsiSyncCache10 cmd = {
        .opCode         = ScsiCommand_SyncCache10,
        .syncNv_immed   = 0,
        .lba            = detail::CpuToBe32(lba),
        .groupNumber    = 0,
        .numberOfBlocks = detail::CpuToBe16(block),
        .control        = 0
    };

    result = ProtocolRead(
        &cmd, sizeof(cmd), m_pDmaBuffer, 0, &xferredSize
    );

    return result;
}

Result
ScsiLogicalUnit::Read10(uint32_t lba, uint16_t block, void *pData) NN_NOEXCEPT
{
    Result   result = ResultSuccess();
    uint32_t xferredSize = 0;

    ScsiRead10 cmd = {
        .opCode                       = ScsiCommand_Read10,
        .rdprotect_dpo_fua_rarc_fuaNv = 0,
        .lba                          = detail::CpuToBe32(lba),
        .groupNumber                  = 0,
        .transferLength               = detail::CpuToBe16(block),
        .control                      = 0
    };

    result = ProtocolRead(
        &cmd, sizeof(cmd), pData, m_BlockSize * block, &xferredSize
    );

    if (result.IsSuccess() && xferredSize != m_BlockSize * block)
    {
        result = ResultIncompleteTransfer();
    }


    return result;
}

Result
ScsiLogicalUnit::Write10(uint32_t lba, uint16_t block, const void *pData) NN_NOEXCEPT
{
    Result   result = ResultSuccess();
    uint32_t xferredSize = 0;

    ScsiWrite10 cmd = {
        .opCode  = ScsiCommand_Write10,
        .wrprotect_dpo_fua_fuaNv = 0,
        .lba                     = detail::CpuToBe32(lba),
        .groupNumber             = 0,
        .transferLength          = detail::CpuToBe16(block),
        .control                 = 0
    };

    result = ProtocolWrite(
        &cmd, sizeof(cmd), pData, m_BlockSize * block, &xferredSize
    );

    if (result.IsSuccess() && xferredSize != m_BlockSize * block)
    {
        result = ResultIncompleteTransfer();
    }

    return result;
}

Result
ScsiLogicalUnit::Read16(uint64_t lba, uint32_t block, void *pData) NN_NOEXCEPT
{
    Result   result = ResultSuccess();
    uint32_t xferredSize = 0;

    ScsiRead16 cmd = {
        .opCode                       = ScsiCommand_Read16,
        .rdprotect_dpo_fua_rarc_fuaNv = 0,
        .lba                          = detail::CpuToBe64(lba),
        .transferLength               = detail::CpuToBe32(block),
        .groupNumber                  = 0,
        .control                      = 0
    };

    result = ProtocolRead(
        &cmd, sizeof(cmd), pData, m_BlockSize * block, &xferredSize
    );

    if (result.IsSuccess() && xferredSize != m_BlockSize * block)
    {
        result = ResultIncompleteTransfer();
    }

    return result;
}

Result
ScsiLogicalUnit::Write16(uint64_t lba, uint32_t block, const void *pData) NN_NOEXCEPT
{
    Result   result = ResultSuccess();
    uint32_t xferredSize = 0;

    ScsiWrite16 cmd = {
        .opCode                  = ScsiCommand_Write16,
        .wrprotect_dpo_fua_fuaNv = 0,
        .lba                     = detail::CpuToBe64(lba),
        .transferLength          = detail::CpuToBe32(block),
        .groupNumber             = 0,
        .control                 = 0
    };

    result = ProtocolWrite(
        &cmd, sizeof(cmd), pData, m_BlockSize * block, &xferredSize
    );

    if (result.IsSuccess() && xferredSize != m_BlockSize * block)
    {
        result = ResultIncompleteTransfer();
    }

    return result;
}

Result
ScsiLogicalUnit::Inquiry() NN_NOEXCEPT
{
    Result   result = ResultSuccess();
    uint32_t xferredSize = 0;
    ScsiInquiry cmd = {
        .opCode            = ScsiCommand_Inquiry,
        .evpd              = 0,
        .pageCode          = 0,
        .transferLength    = detail::CpuToBe16(sizeof(ScsiInquiryData)),
        .control           = 0,
    };

    result = ProtocolRead(
        &cmd, sizeof(cmd), m_pDmaBuffer, sizeof(ScsiInquiryData), &xferredSize
    );

    if (result.IsSuccess())
    {
        uint8_t temp[17] = { 0 };

        std::memcpy(
            &m_InquiryData, m_pDmaBuffer, xferredSize
        );

        NN_CDMSC_INFO("LUN: %d; ", m_Lun);

        std::memcpy(
            temp, m_InquiryData.vendorInformation, 8
        );
        NN_CDMSC_INFO("Vendor information: %s; ", temp);
        std::memset(
            temp, 0, sizeof(temp)
        );

        std::memcpy(
            temp, m_InquiryData.productInformation, 16
        );
        NN_CDMSC_INFO("Product information: %s; ", temp);
        std::memset(
            temp, 0, sizeof(temp)
        );

        std::memcpy(
            temp, m_InquiryData.productRevisionLevel, 4
        );
        NN_CDMSC_INFO("Product revision level: %s\r\n", temp);

        if (m_InquiryData.peripheralQualifer_peripheralDeviceType != 0)
        {
            result = ResultUnsupportedDevice();
        }
    }
    return result;
}



UfiLogicalUnit::UfiLogicalUnit(Device& device, uint16_t vid, uint16_t pid,
                               uint8_t lun, uint32_t capability)
    : LogicalUnit(device, vid, pid, lun, capability)
{
    m_Subclass = "UFI";
}

UfiLogicalUnit::~UfiLogicalUnit()
{
}

Result
UfiLogicalUnit::TestUnitReady() NN_NOEXCEPT
{
    Result   result = ResultSuccess();
    uint32_t xferredSize;

    ScsiTestUnitReady cmd = {
        .opCode  = ScsiCommand_TestUnitReady,
        .control = 0
    };

    result = ProtocolRead(
        &cmd, sizeof(cmd), m_pDmaBuffer, 0, &xferredSize
    );

    return result;
}

Result
UfiLogicalUnit::RequestSense() NN_NOEXCEPT
{
    Result   result = ResultSuccess();
    uint32_t xferredSize = 0;

    ScsiRequestSense cmd = {
        .opCode           = ScsiCommand_RequestSense,
        .desc             = 0,
        .allocationLength = ScsiSenseDataMaxLength,
        .control          = 0
    };

    result = m_Device.BotRead(
        m_Lun, (uint8_t*)&cmd, sizeof(cmd),
        m_pDmaBuffer, ScsiSenseDataMaxLength, &xferredSize
    );

    if (result.IsSuccess())
    {
        std::memcpy(m_SenseData, m_pDmaBuffer, xferredSize);
    }

    return result;
}

Result
UfiLogicalUnit::ReadCapacity10() NN_NOEXCEPT
{
    Result   result = ResultSuccess();
    uint32_t xferredSize = 0;

    ScsiReadCapacity10 cmd = {
        .opCode  = ScsiCommand_ReadCapacity10,
        .lba     = 0,
        .pmi     = 0,
        .control = 0
    };
    ScsiReadCapacity10Data *pData = reinterpret_cast<ScsiReadCapacity10Data*>(m_pDmaBuffer);

    result = ProtocolRead(
        &cmd, sizeof(cmd),
        m_pDmaBuffer, sizeof(ScsiReadCapacity10Data), &xferredSize
    );

    if (result.IsSuccess())
    {
        m_BlockCount = detail::BeToCpu32(pData->lba) + 1; // FIXME: cast to uint64_t?
        m_BlockSize  = detail::BeToCpu32(pData->blockLength);
    }

    return result;
}

Result
UfiLogicalUnit::ReadCapacity16() NN_NOEXCEPT
{
    Result   result = ResultSuccess();
    uint32_t xferredSize = 0;

    ScsiReadCapacity16 cmd = {
        .opCode           = ScsiCommand_ReadCapacity16,
        .serviceAction    = 0x10,
        .lba              = 0,
        .allocationLength = detail::CpuToBe32(sizeof(ScsiReadCapacity16Data)),
        .pmi              = 0,
        .control          = 0
    };
    ScsiReadCapacity16Data *pData = reinterpret_cast<ScsiReadCapacity16Data*>(m_pDmaBuffer);

    NN_UNUSED(pData);

    result = ProtocolRead(
        &cmd, sizeof(cmd),
        m_pDmaBuffer, sizeof(ScsiReadCapacity16Data), &xferredSize
    );

    return result;
}

Result
UfiLogicalUnit::ModeSense10() NN_NOEXCEPT
{
    Result   result = ResultSuccess();
    uint32_t xferredSize = 0;

    ScsiModeSense10 cmd = {
        .opCode           = ScsiCommand_ModeSense10,
        .llbaa_dbd        = 0, // FIXME:
        .pc_pageCode      = 0x3f, // current value, all pages
        .subpageCode      = 0,
        .allocationLength = detail::CpuToBe16(sizeof(ScsiModeParameterHeader10)),
        .control          = 0
    };

    result = ProtocolRead(
        &cmd, sizeof(cmd),
        m_pDmaBuffer, sizeof(ScsiModeParameterHeader10), &xferredSize
    );

    if (result.IsSuccess())
    {
        std::memcpy(
            &m_ModeParamHeader, m_pDmaBuffer, sizeof(ScsiModeParameterHeader10)
        );
    }

    return result;
}

Result
UfiLogicalUnit::StartStopUnit(bool start) NN_NOEXCEPT
{
    Result   result = ResultSuccess();
    uint32_t xferredSize = 0;

    ScsiStartStopUnit cmd = {
        .opCode                            = ScsiCommand_StartStopUnit,
        .immed                             = 0,
        .powerConditionModifier            = 0,
        .powerCondition_noFlush_loej_start = static_cast<uint8_t>(start ? 1 : 0),
        .control                           = 0,
    };

    result = ProtocolRead(
        &cmd, sizeof(cmd), m_pDmaBuffer, 0, &xferredSize
    );

    return result;
}

Result
UfiLogicalUnit::SyncCache10(uint32_t lba, uint16_t block) NN_NOEXCEPT
{
    Result   result = ResultSuccess();
    uint32_t xferredSize = 0;

    ScsiSyncCache10 cmd = {
        .opCode         = ScsiCommand_SyncCache10,
        .syncNv_immed   = 0,
        .lba            = detail::CpuToBe32(lba),
        .groupNumber    = 0,
        .numberOfBlocks = detail::CpuToBe16(block),
        .control        = 0
    };

    result = ProtocolRead(
        &cmd, sizeof(cmd), m_pDmaBuffer, 0, &xferredSize
    );

    return result;
}

Result
UfiLogicalUnit::Read10(uint32_t lba, uint16_t block, void *pData) NN_NOEXCEPT
{
    Result   result = ResultSuccess();
    uint32_t xferredSize = 0;

    ScsiRead10 cmd = {
        .opCode                       = ScsiCommand_Read10,
        .rdprotect_dpo_fua_rarc_fuaNv = 0,
        .lba                          = detail::CpuToBe32(lba),
        .groupNumber                  = 0,
        .transferLength               = detail::CpuToBe16(block),
        .control                      = 0
    };

    result = ProtocolRead(
        &cmd, sizeof(cmd), pData, m_BlockSize * block, &xferredSize
    );

    return result;
}

Result
UfiLogicalUnit::Write10(uint32_t lba, uint16_t block, const void *pData) NN_NOEXCEPT
{
    Result   result = ResultSuccess();
    uint32_t xferredSize = 0;

    ScsiWrite10 cmd = {
        .opCode  = ScsiCommand_Write10,
        .wrprotect_dpo_fua_fuaNv = 0,
        .lba                     = detail::CpuToBe32(lba),
        .groupNumber             = 0,
        .transferLength          = detail::CpuToBe16(block),
        .control                 = 0
    };

    result = ProtocolWrite(
        &cmd, sizeof(cmd), pData, m_BlockSize * block, &xferredSize
    );

    return result;
}

Result
UfiLogicalUnit::Read16(uint64_t lba, uint32_t block, void *pData) NN_NOEXCEPT
{
    Result   result = ResultSuccess();
    uint32_t xferredSize = 0;

    ScsiRead16 cmd = {
        .opCode                       = ScsiCommand_Read16,
        .rdprotect_dpo_fua_rarc_fuaNv = 0,
        .lba                          = detail::CpuToBe64(lba),
        .transferLength               = detail::CpuToBe32(block),
        .groupNumber                  = 0,
        .control                      = 0
    };

    result = ProtocolRead(
        &cmd, sizeof(cmd), pData, m_BlockSize * block, &xferredSize
    );

    return result;
}

Result
UfiLogicalUnit::Write16(uint64_t lba, uint32_t block, const void *pData) NN_NOEXCEPT
{
    Result   result = ResultSuccess();
    uint32_t xferredSize = 0;

    ScsiWrite16 cmd = {
        .opCode                  = ScsiCommand_Write16,
        .wrprotect_dpo_fua_fuaNv = 0,
        .lba                     = detail::CpuToBe64(lba),
        .transferLength          = detail::CpuToBe32(block),
        .groupNumber             = 0,
        .control                 = 0
    };

    result = ProtocolWrite(
        &cmd, sizeof(cmd), pData, m_BlockSize * block, &xferredSize
    );

    return result;
}

Result
UfiLogicalUnit::Inquiry() NN_NOEXCEPT
{
    Result   result = ResultSuccess();
    uint32_t xferredSize = 0;
    ScsiInquiry cmd = {
        .opCode = ScsiCommand_Inquiry,
        .evpd = 0,
        .pageCode = 0,
        .transferLength = detail::CpuToBe16(sizeof(ScsiInquiryData)),
        .control = 0,
    };

    result = ProtocolRead(
        &cmd, sizeof(cmd), m_pDmaBuffer, sizeof(ScsiInquiryData), &xferredSize
    );

    if (result.IsSuccess())
    {
        uint8_t temp[17] = { 0 };

        std::memcpy(
            &m_InquiryData, m_pDmaBuffer, xferredSize
        );

        NN_CDMSC_INFO("LUN: %d; ", m_Lun);

        std::memcpy(
            temp, m_InquiryData.vendorInformation, 8
        );
        NN_CDMSC_INFO("Vendor information: %s; ", temp);
        std::memset(
            temp, 0, sizeof(temp)
        );

        std::memcpy(
            temp, m_InquiryData.productInformation, 16
        );
        NN_CDMSC_INFO("Product information: %s; ", temp);
        std::memset(
            temp, 0, sizeof(temp)
        );

        std::memcpy(
            temp, m_InquiryData.productRevisionLevel, 4
        );
        NN_CDMSC_INFO("Product revision level: %s\r\n", temp);

        if (m_InquiryData.peripheralQualifer_peripheralDeviceType != 0)
        {
            result = ResultUnsupportedDevice();
        }
    }
    return result;
}

} // driver
} // cdmsc
} // nn
