﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief
 *
 * @details
 */

#include "cdmsc_PrivateIncludes.h"

#include "cdmsc_Device.h"
#include "cdmsc_LogicalUnit.h"
#include "cdmsc_SafeHandleManager.h"

namespace nn {
namespace cdmsc {
namespace driver {

class DeviceRegistry
{
public:
    DeviceRegistry();
    ~DeviceRegistry();

    void Update(uint16_t vid, uint16_t pid, uint32_t capability);
    uint32_t Get(uint16_t vid, uint16_t pid);

private:
    static const uint32_t  RegistrySize = 16;
    static const uint32_t  DefaultCapability = 0xffffffff;

    struct {
        uint16_t vid;
        uint16_t pid;
        uint32_t capability;
    } m_Registry[RegistrySize];

    uint32_t m_Index;

private:
    void Add(uint16_t vid, uint16_t pid, uint32_t capability);
};

class Driver
{

public:
    Driver();
    ~Driver();

    Result Initialize(nn::os::EventType *pDeviceAvailableEvent,
                      AllocateFunction   alloc,
                      DeallocateFunction dealloc);
    Result Finalize();

    Result Probe(nn::os::EventType *pDetachEvent, UnitProfile *pOutProfile);

    Result Read(void* pOutBuffer, UnitHandle handle,
                uint64_t lba, uint32_t block);
    Result Write(const void* pInBuffer, UnitHandle handle,
                 uint64_t lba, uint32_t block);
    Result Flush(UnitHandle handle, uint64_t lba, uint32_t block);

    void   GetCapability(uint16_t vid, uint16_t pid, uint32_t *pOutCapability);
    void   UpdateCapability(uint16_t vid, uint16_t pid, uint32_t capability);

    // Detach Event and Xfer Timeout Event
    void   RegisterDeviceEvent(nn::os::MultiWaitHolderType& eventHolder);
    void   UnregisterDeviceEvent(nn::os::MultiWaitHolderType& eventHolder);

    Result RegisterLogicalUnit(LogicalUnit *pLu);
    void   UnregisterLogicalUnit(LogicalUnit *pLu);

private:
    // main thread
    bool                                   m_IsMainThreadRunning;
    nn::os::ThreadType                     m_MainThread;
    nn::os::MultiWaitType                  m_MultiWait;
    nn::os::Event                          m_BreakEvent;
    nn::os::MultiWaitHolderType            m_BreakEventHolder;
    NN_OS_ALIGNAS_THREAD_STACK uint8_t     m_MainThreadStack[1024 * 16];

    // Host Stack
    nn::usb::Host                          m_UsbHost;
    nn::usb::DeviceFilter                  m_UsbDeviceFilter;
    nn::os::SystemEventType                m_UsbIfAvailableEvent;
    nn::os::MultiWaitHolderType            m_UsbIfAvailableEventHolder;

    // Availability event published to user
    nn::os::EventType                     *m_pDeviceAvailableEvent;

    // Device Capability Registry
    DeviceRegistry                         m_Registry;

    uint32_t                               m_DeviceSequence;
    Device                                *m_pDevice[DeviceCountMax];

    // Device handle manager
    SafeHandleManager<LogicalUnit, LogicalUnitCountMax> m_LogicalUnitHandleManager;

private:
    void        MainThread();
    static void MainThreadEntry(void* pThis)
    {
        ((Driver*)pThis)->MainThread();
    }
    void CreateAttached();
    void CreateDevice(nn::usb::InterfaceQueryOutput *pProfile);
    void DestroyDevice(Device *pDevice);
};

} // driver
} // cdmsc
} // nn
