﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "cdmsc_Device.h"
#include "cdmsc_Driver.h"

namespace nn {
namespace cdmsc {
namespace driver {

Device::Device(nn::usb::Host&                  usbHost,
               nn::usb::InterfaceQueryOutput  *pProfile,
               Driver&                         driver,
               uint32_t                        sequence)
    : m_UsbHost(usbHost)
    , m_Profile(*pProfile)
    , m_Driver(driver)
    , m_Sequence(sequence)
    , m_MaxLun(0)
    , m_pStateChangeEvent(nullptr)
    , m_WatchdogTimer(nn::os::EventClearMode_ManualClear)
    , m_Mutex(false)
    , m_IsThreadRunning(false)
    , m_pThreadStack(nullptr)
    , m_BreakEvent(nn::os::EventClearMode_ManualClear)
{
    // NN_CDMSC_INFO won't expand for release build
    NN_UNUSED(m_Sequence);

    std::memset(m_pLogicalUnit, 0, sizeof(m_pLogicalUnit));

    nn::os::InitializeMultiWait(&m_MultiWait);

    nn::os::InitializeMultiWaitHolder(
        &m_BreakEventHolder, m_BreakEvent.GetBase()
    );
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_BreakEventHolder);

    nn::os::InitializeMultiWaitHolder(
        &m_WatchdogTimerHolder, m_WatchdogTimer.GetBase()
    );
    nn::os::SetMultiWaitHolderUserData(
        &m_WatchdogTimerHolder, reinterpret_cast<intptr_t>(this)
    );
}

Device::~Device()
{
    nn::os::FinalizeMultiWaitHolder(&m_WatchdogTimerHolder);

    nn::os::UnlinkMultiWaitHolder(&m_BreakEventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_BreakEventHolder);

    nn::os::FinalizeMultiWait(&m_MultiWait);
}

// We MUST overload operator new because Device is over-aligned.
void*
Device::operator new(size_t size) NN_NOEXCEPT
{
    return detail::Allocate(alignof(Device), size);
}

void
Device::operator delete(void* p, size_t size)NN_NOEXCEPT
{
    detail::Deallocate(p, size);
}

Result
Device::Initialize() NN_NOEXCEPT
{
    Result   result   = ResultSuccess();
    uint16_t vid      = m_Profile.deviceProfile.deviceDesc.idVendor;
    uint16_t pid      = m_Profile.deviceProfile.deviceDesc.idProduct;
    uint8_t  subclass = m_Profile.ifProfile.ifDesc.bInterfaceSubClass;

    // NN_CDMSC_INFO won't expand for release build
    NN_UNUSED(vid);
    NN_UNUSED(pid);

    NN_CDMSC_INFO(
        "UMS-%d (Detected)\n"
        "  VID        0x%04x\n"
        "  PID        0x%04x\n",
        m_Sequence, vid, pid
    );

    switch (subclass)
    {
    case MscSubclass_Ufi:
        NN_CDMSC_INFO("  SubClass   UFI\n");
        break;

    case MscSubclass_Scsi:
        NN_CDMSC_INFO("  SubClass   SCSI\n");
        break;

    default:
        NN_CDMSC_INFO("  SubClass   0x%02x (Unsupported)\n", subclass);
        result = ResultUnsupportedDevice();
        goto bail1;
    }

    // Try to initialize the USB interface.
    // This may rightfully fail if the device has since disconnected
    NN_CDMSC_DO(
        m_UsbInterface.Initialize(&m_UsbHost, m_Profile.ifProfile.handle)
    );
    if (result.IsFailure())
    {
        goto bail1;
    }

    NN_CDMSC_DO(m_Bot.Initialize(&m_UsbInterface));
    if (result.IsFailure())
    {
        goto bail2;
    }

    m_pThreadStack = detail::Allocate(nn::os::ThreadStackAlignment, StackSize);
    if (m_pThreadStack == nullptr)
    {
        result = ResultMemoryAllocError();
        goto bail3;
    }

    m_pStateChangeEvent = m_UsbInterface.GetStateChangeEvent();
    nn::os::InitializeMultiWaitHolder(
        &m_StateChangeEventHolder, m_pStateChangeEvent
    );
    nn::os::SetMultiWaitHolderUserData(
        &m_StateChangeEventHolder, reinterpret_cast<intptr_t>(this)
    );
    m_Driver.RegisterDeviceEvent(m_StateChangeEventHolder);
    m_Driver.RegisterDeviceEvent(m_WatchdogTimerHolder);

    // Create and start the device thread
    NN_CDMSC_ABORT_UPON_ERROR(
        nn::os::CreateThread(
            &m_Thread, ThreadEntry, this, m_pThreadStack, StackSize,
            NN_SYSTEM_THREAD_PRIORITY(cdmsc, DeviceThread)
        )
    );
    nn::os::SetThreadNamePointer(&m_Thread,
                                 NN_SYSTEM_THREAD_NAME(cdmsc, DeviceThread));
    m_IsThreadRunning = true;
    nn::os::StartThread(&m_Thread);

    return result;

bail3:
    NN_CDMSC_ABORT_UNLESS_SUCCESS(m_Bot.Finalize());

bail2:
    NN_CDMSC_ABORT_UNLESS_SUCCESS(m_UsbInterface.Finalize());

bail1:
    NN_CDMSC_INFO("UMS-%d (Initialization Failure)\n", m_Sequence);
    return result;
}

Result
Device::Finalize() NN_NOEXCEPT
{
    Result result = ResultSuccess();

    NN_CDMSC_INFO("UMS-%d (Destroying)\n", m_Sequence);

    // Issue a device reset, cancel any pending transactions
    NN_CDMSC_ABORT_UNLESS_SUCCESS(Detach());

    // Stop and destroy the device thread
    m_IsThreadRunning = false;
    m_BreakEvent.Signal();
    nn::os::WaitThread(&m_Thread);
    nn::os::DestroyThread(&m_Thread);

    detail::Deallocate(m_pThreadStack, StackSize);

    m_Driver.UnregisterDeviceEvent(m_WatchdogTimerHolder);
    m_Driver.UnregisterDeviceEvent(m_StateChangeEventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_StateChangeEventHolder);
    m_pStateChangeEvent = nullptr;

    NN_CDMSC_ABORT_UNLESS_SUCCESS(m_Bot.Finalize());
    NN_CDMSC_ABORT_UNLESS_SUCCESS(m_UsbInterface.Finalize());

    NN_CDMSC_INFO("UMS-%d (Destroyed)\n", m_Sequence);

    return result;
}

Result
Device::Detach() NN_NOEXCEPT
{
    Result result = ResultSuccess();

    result = m_UsbInterface.ResetDevice();

    // The device could have been detached
    if (nn::usb::ResultInterfaceInvalid::Includes(result))
    {
        result = ResultSuccess();
    }

    return result;
}

Result
Device::BotRead(uint8_t lun, void *pCmd, uint8_t cmdLength,
                void *buffer, uint32_t size, uint32_t *pOutXferred) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    NN_CDMSC_ABORT_UNLESS(lun <= m_MaxLun);

    ArmWatchdogTimer(BotTransactionTimeout);
    result = m_Bot.Read(lun, pCmd, cmdLength, buffer, size, pOutXferred);
    DisarmWatchdogTimer();

    if (ResultBotOutOfSync::Includes(result))
    {
        // FIXME: error handling?
        Detach();
    }

    return result;
}

Result
Device::BotWrite(uint8_t lun, void *pCmd, uint8_t cmdLength,
                 const void *buffer, uint32_t size, uint32_t *pOutXferred) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    NN_CDMSC_ABORT_UNLESS(lun <= m_MaxLun);

    ArmWatchdogTimer(BotTransactionTimeout);
    result =  m_Bot.Write(lun, pCmd, cmdLength, buffer, size, pOutXferred);
    DisarmWatchdogTimer();

    if (ResultBotOutOfSync::Includes(result))
    {
        // FIXME: error handling?
        Detach();
    }

    return result;
}

void
Device::RegisterKeepAliveTimer(nn::os::MultiWaitHolderType& timerHolder) NN_NOEXCEPT
{
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &timerHolder);
}

void
Device::UnregisterKeepAliveTimer(nn::os::MultiWaitHolderType& timerHolder) NN_NOEXCEPT
{
    nn::os::UnlinkMultiWaitHolder(&timerHolder);
}

void
Device::Lock() NN_NOEXCEPT
{
    m_Mutex.Lock();
}

void
Device::Unlock() NN_NOEXCEPT
{
    m_Mutex.Unlock();
}

void
Device::Thread()
{
    Result result = ResultSuccess();

    NN_CDMSC_INFO("UMS-%d (Populating)\n", m_Sequence);

    // Query for the max LUN
    ArmWatchdogTimer(5000);
    result = m_Bot.GetMaxLun(&m_MaxLun);
    DisarmWatchdogTimer();

    if (result.IsFailure())
    {
        FireWatchdogTimer();
        goto bail1;
    }

    NN_CDMSC_INFO("  MaxLun     %d\n", m_MaxLun);

    // Now create all the logical units
    result = CreateLogicalUnits();
    if (result.IsFailure())
    {
        FireWatchdogTimer();
        goto bail1;
    }

    while (m_IsThreadRunning) {
        nn::os::MultiWaitHolderType *holder = nn::os::WaitAny(&m_MultiWait);

        if (holder == &m_BreakEventHolder)
        {
            // m_ThreadRun will be checked, breaking out of loop
            m_BreakEvent.Clear();
        }
        else // keep alive timer event
        {
            LogicalUnit *pLu = reinterpret_cast<LogicalUnit*>(
                nn::os::GetMultiWaitHolderUserData(holder)
            );

            pLu->KeepAlive();
        }
    }

    DestroyLogicalUnits();
    return;

bail1:
    NN_CDMSC_INFO("UMS-%d (Populating Failure)\n", m_Sequence);
    return;
}

void
Device::ArmWatchdogTimer(uint32_t timeoutInMs) NN_NOEXCEPT
{
    m_WatchdogTimer.StartOneShot(nn::TimeSpan::FromMilliSeconds(timeoutInMs));
}

void
Device::DisarmWatchdogTimer() NN_NOEXCEPT
{
    m_WatchdogTimer.Stop();
}

void
Device::FireWatchdogTimer() NN_NOEXCEPT
{
    m_WatchdogTimer.Signal();
}

Result
Device::CreateLogicalUnits() NN_NOEXCEPT
{
    Result   result     = ResultSuccess();
    uint32_t capability = 0;
    uint16_t vid        = m_Profile.deviceProfile.deviceDesc.idVendor;
    uint16_t pid        = m_Profile.deviceProfile.deviceDesc.idProduct;
    uint8_t  subclass   = m_Profile.ifProfile.ifDesc.bInterfaceSubClass;

    // Create Logical Units
    for (uint8_t i = 0; i <= m_MaxLun; i++)
    {
        m_Driver.GetCapability(vid, pid, &capability);

        NN_CDMSC_INFO(
            "  LU-%d-%d (Populating)\n"
            "    SyncCache10        %s\n"
            "    ModeSense10        %s\n",
            m_Sequence, i,
            (capability & DeviceCapability_SyncCache10) ? "YES" : "NO",
            (capability & DeviceCapability_ModeSense10) ? "YES" : "NO"
        );

        switch (subclass)
        {
        case MscSubclass_Ufi:
            m_pLogicalUnit[i] = new UfiLogicalUnit(*this, vid, pid, i, capability);
            break;

        case MscSubclass_Scsi:
            m_pLogicalUnit[i] = new ScsiLogicalUnit(*this, vid, pid, i, capability);
            break;

        default:
            NN_UNEXPECTED_DEFAULT;
            break;
        }

        if (m_pLogicalUnit[i] == nullptr)
        {
            NN_CDMSC_INFO("  LU-%d-%d (Populating Failure, OOM)\n", m_Sequence, i);

            result = ResultMemoryAllocError();
            goto bail1;
        }

        result = m_pLogicalUnit[i]->Initialize();

        // Lesson learned
        if (capability != m_pLogicalUnit[i]->GetCapability())
        {
            capability = m_pLogicalUnit[i]->GetCapability();
            m_Driver.UpdateCapability(vid, pid, capability);
            NN_CDMSC_INFO(
                "  LU-%d-%d (Learn Capability)\n"
                "    SyncCache10        %s\n"
                "    ModeSense10        %s\n",
                m_Sequence, i,
                (capability & DeviceCapability_SyncCache10) ? "YES" : "NO",
                (capability & DeviceCapability_ModeSense10) ? "YES" : "NO"
            );
        }

        if (result.IsFailure())
        {
            if (ResultNoMedium::Includes(result))
            {
                NN_CDMSC_INFO("  LU-%d-%d (No Medium)\n", m_Sequence, i);
            }
            NN_CDMSC_INFO("  LU-%d-%d (Populating Failure)\n", m_Sequence, i);

            delete m_pLogicalUnit[i];
            m_pLogicalUnit[i] = nullptr;

            // Ignore the failure, proceed to next LU
            result = ResultSuccess();
            continue;
        }

        // Could have reached our support limit, just ignore the remaining
        // units. You have to remove other devices and then retach this one
        // in order to populate the remaining units.
        result = m_Driver.RegisterLogicalUnit(m_pLogicalUnit[i]);
        if (result.IsFailure())
        {
            NN_CDMSC_INFO("  LU-%d-%d (Ignore, MaxLun limit)\n", m_Sequence, i);

            NN_CDMSC_ABORT_UNLESS_SUCCESS(m_pLogicalUnit[i]->Finalize());
            delete m_pLogicalUnit[i];
            m_pLogicalUnit[i] = nullptr;

            result = ResultSuccess();
            break;
        }

        UnitProfile profile;
        m_pLogicalUnit[i]->GetProfile(&profile);
        NN_CDMSC_INFO(
            "    BlockSize          0x%x\n"
            "    BlockCount         0x%llx\n"
            "    WriteProtection    %s\n"
            "  LU-%d-%d (Populated)\n",
            profile.blockSize, profile.blockCount,
            profile.isWriteProtected ? "YES" : "NO",
            m_Sequence, i
        );
    }

    return result;

bail1:
    DestroyLogicalUnits();

    return result;
}

void
Device::DestroyLogicalUnits() NN_NOEXCEPT
{
    for (uint8_t i = 0; i <= m_MaxLun; i++)
    {
        if (m_pLogicalUnit[i] != nullptr)
        {
            NN_CDMSC_INFO(
                "  LU-%d-%d (Destroying)\n",
                m_Sequence, i
            );

            m_Driver.UnregisterLogicalUnit(m_pLogicalUnit[i]);

            NN_CDMSC_ABORT_UNLESS_SUCCESS(m_pLogicalUnit[i]->Finalize());
            delete m_pLogicalUnit[i];
            m_pLogicalUnit[i] = nullptr;

            NN_CDMSC_INFO(
                "  LU-%d-%d (Destroyed)\n",
                m_Sequence, i
            );
        }
    }
}

} // driver
} // cdmsc
} // nn
