﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

namespace nn {
namespace cdhid {
namespace win32 {

enum
{
    KEYBOARD_THREAD_STACK_SIZE  = 16 * 1024,
    KEYBOARD_DATA_ENTRIES       = 8,
    KEYBOARD_DATA_SIZE          = 8,
    KEYBOARD_KEYS               = 255,
};

class Win32KeyboardInterface
{
    NN_DISALLOW_COPY(Win32KeyboardInterface);
    NN_DISALLOW_MOVE(Win32KeyboardInterface);

public:

    // runtime keyboard params
    bool                            m_KeyDown[KEYBOARD_KEYS];
    uint16_t                        m_KeyState;

    Win32KeyboardInterface() NN_NOEXCEPT{};
    ~Win32KeyboardInterface() NN_NOEXCEPT{};

    Result Initialize() NN_NOEXCEPT;
    Result Finalize() NN_NOEXCEPT;

    Result AcquireDevice() NN_NOEXCEPT;
    Result ReleaseDevice() NN_NOEXCEPT;
    Result SetCtrlSession(nn::sf::SharedPointer<nn::ahid::ICtrlSession> pCtrlSession) NN_NOEXCEPT;

    Result AhidGetCodeBook(uint8_t* pBuffer, uint32_t bufferSize) NN_NOEXCEPT;
    Result AhidRead(uint32_t* pBytesTransferred, uint8_t* pBuffer, uint32_t bytes, nn::TimeSpan timeout) NN_NOEXCEPT;

private:

    bool                            m_IsAcquired;
    nn::ahid::hdr::Hdr              m_Hdr;
    nn::ahid::hdr::DeviceHandle     m_HdrDeviceHandle;
    Win32KeyboardBuffer             m_Buffer;
    Win32KeyboardCodeBook           m_CodeBook;
    uint8_t                         m_KeyboardData[KEYBOARD_DATA_ENTRIES][KEYBOARD_DATA_SIZE];
    nn::sf::SharedPointer<nn::ahid::ICtrlSession>  m_CtrlSession;

    Result AttachDevice() NN_NOEXCEPT;
    Result DetachDevice() NN_NOEXCEPT;

    // Read thread to get Windows Keyboard events
    nn::os::ThreadType              m_ReadThread;
    NN_ALIGNAS(4096) uint8_t        m_ReadThreadStack[KEYBOARD_THREAD_STACK_SIZE];

    void ReadFromKeyboard();

    static void ThreadFunction(void* pThis)
    {
        reinterpret_cast<Win32KeyboardInterface*>(pThis)->ReadFromKeyboard();
    }

    HWND                            m_Hwnd;
};

} // end of namespace win32
} // end of namespace cdhid
} // end of namespace nn
