﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "cdhidWin32Keyboard_Private.h"
#include <nn/nn_SystemThreadDefinition.h>

namespace nn {
namespace cdhid {
namespace win32 {

//////////////////////////////////////////////////////////////////////////////
//  public functions
//////////////////////////////////////////////////////////////////////////////
Result Win32KeyboardInterface::Initialize() NN_NOEXCEPT
{
    m_Hdr.Initialize();
    m_CodeBook.Initialize();
    m_Buffer.Initialize(&m_KeyboardData[0][0], KEYBOARD_DATA_ENTRIES, KEYBOARD_DATA_SIZE);

    AttachDevice();

    // start pollng thread for reads
    nn::os::CreateThread(
        &m_ReadThread,
        ThreadFunction,
        this,
        m_ReadThreadStack,
        KEYBOARD_THREAD_STACK_SIZE,
        NN_SYSTEM_THREAD_PRIORITY(cdhid,Win32KeyboardInterface)
        );

    nn::os::SetThreadNamePointer(&m_ReadThread, NN_SYSTEM_THREAD_NAME(cdhid, Win32KeyboardInterface));
    nn::os::StartThread(&m_ReadThread);

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result Win32KeyboardInterface::Finalize() NN_NOEXCEPT
{
    if (m_Hwnd)
    {
        SendMessage(m_Hwnd, WM_DESTROY, 0, 0);  // hack :0

        nn::os::WaitThread(&m_ReadThread);
        nn::os::DestroyThread(&m_ReadThread);
    }

    DetachDevice();

    m_Buffer.Finalize();
    m_CodeBook.Finalize();
    m_Hdr.Finalize();

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result Win32KeyboardInterface::AcquireDevice() NN_NOEXCEPT
{
    Result result = ResultSuccess();

    if (m_IsAcquired == false)
    {
        m_IsAcquired    = true;
        m_CtrlSession   = nullptr;
    }
    else
    {
        result = nn::ahid::ResultAcquire();
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result Win32KeyboardInterface::ReleaseDevice() NN_NOEXCEPT
{
    Result result = ResultSuccess();

    if (m_IsAcquired == true)
    {
        m_CtrlSession   = nullptr;
        m_IsAcquired    = false;
    }
    else
    {
        result = nn::ahid::ResultAcquire();
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result Win32KeyboardInterface::SetCtrlSession(nn::sf::SharedPointer<nn::ahid::ICtrlSession> pCtrlSession) NN_NOEXCEPT
{
    m_CtrlSession = pCtrlSession;

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result Win32KeyboardInterface::AhidGetCodeBook(uint8_t* pBuffer, uint32_t bufferSize) NN_NOEXCEPT
{
    return m_CodeBook.Get(pBuffer, bufferSize);
}


//////////////////////////////////////////////////////////////////////////////
Result Win32KeyboardInterface::AhidRead(uint32_t* pBytesTransferred, uint8_t* pBuffer, uint32_t bytes, nn::TimeSpan timeout) NN_NOEXCEPT
{
    NN_UNUSED(bytes);
    NN_UNUSED(timeout);

    // TODO check buffer size

    return m_Buffer.Get(pBuffer, pBytesTransferred);
}


//////////////////////////////////////////////////////////////////////////////
//  private functions
//////////////////////////////////////////////////////////////////////////////
Result Win32KeyboardInterface::AttachDevice() NN_NOEXCEPT
{
    m_IsAcquired = false;

    nn::ahid::hdr::DeviceParameters hdrDeviceParameters;

    memset(&hdrDeviceParameters, 0, sizeof(nn::ahid::hdr::DeviceParameters));

    sprintf(reinterpret_cast<char*>(hdrDeviceParameters.servicePath), "Keyboar");
    hdrDeviceParameters.deviceHandle    = 1;
    hdrDeviceParameters.busId           = nn::ahid::hdr::AhidBusIdWin32;
    hdrDeviceParameters.usagePage       = 0x01;     // Generic Desktop Controls
    hdrDeviceParameters.usageId         = 0x06;     // Keyboard

    char manufacturerString[16] = {16 , nn::ahid::hdr::AhidDescriptorTypeString, 'H', 0, 'o', 0, 'r', 0, 'i', 0, 'z', 0, 'o', 0, 'n', 0};
    char productStringKeyboard[28] = {28 , nn::ahid::hdr::AhidDescriptorTypeString, 'A', 0, 'H', 0, 'I', 0, 'D', 0, ' ', 0, 'K', 0, 'e', 0, 'y', 0, 'b', 0, 'o', 0, 'a', 0, 'r', 0, 'd', 0};

    memcpy(hdrDeviceParameters.manufacturer, manufacturerString, sizeof(manufacturerString));
    memcpy(hdrDeviceParameters.product, productStringKeyboard, sizeof(productStringKeyboard));

    return m_Hdr.AttachDevice(&m_HdrDeviceHandle, &hdrDeviceParameters);
}


//////////////////////////////////////////////////////////////////////////////
Result Win32KeyboardInterface::DetachDevice() NN_NOEXCEPT
{
    if (m_CtrlSession)
    {
        m_CtrlSession->SignalStateChangeEvent();
    }

    m_IsAcquired = false;

    return m_Hdr.DetachDevice(m_HdrDeviceHandle);
}


//////////////////////////////////////////////////////////////////////////////
static Win32KeyboardBuffer*    g_pWin32KeyboardBuffer;
static Win32KeyboardInterface* g_pWin32KeyboardInterface;

static LRESULT CALLBACK WndProc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
    switch (uMsg)
    {
    case WM_DESTROY:
        {
            PostQuitMessage(0);
        }

        return 0;

    case WM_INPUT:
        {
            RAWINPUT    rawInput;
            UINT        bufferSize  = sizeof(RAWINPUT);
            UINT        dataSize;

            dataSize = GetRawInputData((HRAWINPUT)lParam, RID_INPUT, &rawInput, &bufferSize, sizeof(RAWINPUTHEADER));

            if (rawInput.header.dwType == RIM_TYPEKEYBOARD)
            {

                bool updateKeys     = false;

                if (rawInput.data.keyboard.Flags & 1)   // key up
                {
                    if (g_pWin32KeyboardInterface->m_KeyDown[rawInput.data.keyboard.MakeCode] == true)
                    {
                        g_pWin32KeyboardInterface->m_KeyDown[rawInput.data.keyboard.MakeCode] = false;
                        updateKeys = true;
                    }
                }
                else // key down
                {
                    if (g_pWin32KeyboardInterface->m_KeyDown[rawInput.data.keyboard.MakeCode] == false)
                    {
                        g_pWin32KeyboardInterface->m_KeyDown[rawInput.data.keyboard.MakeCode] = true;
                        updateKeys = true;
                    }
                }

                if (updateKeys)
                {
                    uint8_t data[KEYBOARD_DATA_SIZE];

                    memset(data, 0, KEYBOARD_DATA_SIZE);

                    if (GetKeyState(VK_LCONTROL) & 0x100)   {data[0] |= 1;}
                    if (GetKeyState(VK_LSHIFT) & 0x100)     {data[0] |= (1 << 1);}
                    if (GetKeyState(VK_RCONTROL) & 0x100)   {data[0] |= (1 << 4);}
                    if (GetKeyState(VK_RSHIFT) & 0x100)     {data[0] |= (1 << 5);}

                    if (GetKeyState(VK_NUMLOCK) & 1)        {data[1] |= 1;}
                    if (GetKeyState(VK_CAPITAL) & 1)        {data[1] |= (1 << 1);}
                    if (GetKeyState(VK_SCROLL) & 1)         {data[1] |= (1 << 2);}
                    if (GetKeyState(VK_KANA) & 1)           {data[1] |= (1 << 4);}

                    uint32_t count = 2;

                    for (uint32_t i = 0; i < KEYBOARD_KEYS; i++)
                    {
                        if (g_pWin32KeyboardInterface->m_KeyDown[i])
                        {
                            data[count++] = static_cast<uint8_t>(i);

                            if (count == KEYBOARD_DATA_SIZE)
                            {
                                /*

                                    actually Windows RawInput keyboard will only
                                    account for up to 4 keys pressed at once, so
                                    we'll never get here :0

                                 */
                                break;
                            }
                        }
                    }

                    g_pWin32KeyboardBuffer->Put(data);
                }
            }
        }

        return 0;

        default:

            break;
    }

    return DefWindowProc(hwnd, uMsg, wParam, lParam);
}

void Win32KeyboardInterface::ReadFromKeyboard()
{
    g_pWin32KeyboardInterface   = this;
    g_pWin32KeyboardBuffer      = &this->m_Buffer;
    const wchar_t CLASS_NAME[]  = L"AHID Keyboard";

    WNDCLASS wndclass = { };

    wndclass.lpfnWndProc    = WndProc;
    wndclass.hInstance      = 0;
    wndclass.lpszClassName  = CLASS_NAME;

    RegisterClass(&wndclass);

    m_Hwnd = CreateWindow(
                            CLASS_NAME,
                            CLASS_NAME,
                            0,
                            0,
                            0,
                            0,
                            0,
                            HWND_MESSAGE,
                            NULL,
                            0,
                            this
                            );

    RAWINPUTDEVICE rid;

    rid.usUsagePage = 0x01;
    rid.usUsage     = 0x06; // keyboard
    rid.dwFlags     = RIDEV_INPUTSINK;  // we need to process legacy messages to get LED state
    rid.hwndTarget  = m_Hwnd;

    RegisterRawInputDevices(&rid, 1, sizeof(rid));

    ShowWindow(m_Hwnd, SW_MAXIMIZE);
//    EnableWindow(m_Hwnd, true);
//    SetFocus(m_Hwnd);

    MSG msg = { };

    while (GetMessage(&msg, m_Hwnd, 0, 0))
    {
        TranslateMessage(&msg);
        DispatchMessage(&msg);

        if (msg.message == 0)   // hack :0
        {
            break;
        }
    }
}


} // end of namespace win32
} // end of namespace cdhid
} // end of namespace nn
