﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "cdhidUsb_Private.h"

#include <nn/nn_Common.h>
#include <nn/sf/sf_HipcServer.h>
#include <type_traits>
#include <new>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SystemThreadDefinition.h>

namespace nn {
namespace cdhid {
namespace usb {

//////////////////////////////////////////////////////////////////////////////
//  public functions
//////////////////////////////////////////////////////////////////////////////
Result UsbHidServer::Initialize() NN_NOEXCEPT
{
    m_Enum.Initialize();

    Result result = InitializePort(0, USBHID_SERVER_MAX_SESSIONS, "ahid:cd");

    if (result.IsSuccess())
    {
        m_HeapHandle = nn::lmem::CreateExpHeap(
                                                &m_SfHeapBuffer,
                                                sizeof(m_SfHeapBuffer),
                                                nn::lmem::CreationOption_NoOption
                                                );
        m_SfAllocator.Attach(m_HeapHandle);

        for (int i = 0; i < USBHID_SERVER_MAX_THREADS; i++)
        {
            nn::os::CreateThread(
                &m_Thread[i],
                SessionThread,
                this,
                m_ThreadStack[i],
                USBHID_SERVER_THREAD_STACK_SIZE,
                NN_SYSTEM_THREAD_PRIORITY(cdhid,IpcServer)
                );

            nn::os::SetThreadNamePointer(&m_Thread[i], NN_SYSTEM_THREAD_NAME(cdhid, IpcServer));
            nn::os::StartThread(&m_Thread[i]);
        }
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidServer::Finalize() NN_NOEXCEPT
{
    RequestStop();

    for (int i = 0; i < USBHID_SERVER_MAX_THREADS; i++)
    {
        nn::os::WaitThread(&m_Thread[i]);
        nn::os::DestroyThread(&m_Thread[i]);
    }

    m_Enum.Finalize();

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
nn::sf::SharedPointer<nn::ahid::IServerSession> UsbHidServer::GetDfcSession() NN_NOEXCEPT
{
    return CreateSession();
}


//////////////////////////////////////////////////////////////////////////////
//  private functions
//////////////////////////////////////////////////////////////////////////////
nn::sf::SharedPointer<nn::ahid::IServerSession> UsbHidServer::CreateSession() NN_NOEXCEPT
{
    return Factory::CreateSharedEmplaced<nn::ahid::IServerSession, ServerSession>(&m_SfAllocator, &m_Enum, &m_SfAllocator);
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidServer::OnNeedsToAccept(int portIndex, PortObjectImpl* pPort) NN_NOEXCEPT
{
    NN_UNUSED(portIndex);
    return AcceptImpl(pPort, CreateSession());
}


//////////////////////////////////////////////////////////////////////////////
void UsbHidServer::SessionThread(void *arg) NN_NOEXCEPT
{
    UsbHidServer *pThis = reinterpret_cast<UsbHidServer*>(arg);

    pThis->Start();
    pThis->LoopAuto();
}



} // end of namespace usb
} // end of namespace cdhid
} // end of namespace nn
