﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "cdhidUsb_Private.h"

namespace nn {
namespace cdhid {
namespace usb {

//////////////////////////////////////////////////////////////////////////////
//  public functions
//////////////////////////////////////////////////////////////////////////////
Result UsbHidInterface::Initialize(int index) NN_NOEXCEPT
{
    m_HandleIndex   = index;
    m_IsOccupied    = false;
    m_IsAcquired    = false;

    ChangeHandle();

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidInterface::Finalize() NN_NOEXCEPT
{
    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidInterface::IncHandleSemaphore(uint32_t handle) NN_NOEXCEPT
{
    if (((handle >> 16) == m_HandleIndex) && ((handle & 0xffff) == m_HandleInstance))
    {
        // TODO inc transaction semaphore

        return ResultSuccess();
    }

    return nn::ahid::ResultInvalidHandle();
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidInterface::DecHandleSemaphore() NN_NOEXCEPT
{
    // TODO dec transaction semaphore

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidInterface::ClientAcquireDevice(uint32_t handle) NN_NOEXCEPT
{
    Result result = IncHandleSemaphore(handle);

    if (result.IsSuccess())
    {
        if (m_IsAcquired == false)
        {
            m_IsAcquired    = true;
            m_CtrlSession   = nullptr;
        }
        else
        {
            result = nn::ahid::ResultAcquire();
        }

        DecHandleSemaphore();
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidInterface::ClientReleaseDevice(uint32_t handle) NN_NOEXCEPT
{
    Result result = IncHandleSemaphore(handle);

    if (result.IsSuccess())
    {
        if (m_IsAcquired == true)
        {
            m_CtrlSession   = nullptr;
            m_IsAcquired    = false;
        }
        else
        {
            result = nn::ahid::ResultAcquire();
        }

        DecHandleSemaphore();
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidInterface::SetCtrlSession(nn::sf::SharedPointer<nn::ahid::ICtrlSession> pCtrlSession) NN_NOEXCEPT
{
    m_CtrlSession = pCtrlSession;

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidInterface::HostAttachDevice(nn::usb::Host* pClient, nn::usb::InterfaceQueryOutput* pProfile, nn::ahid::hdr::Hdr* pHdr, int interfaceIndex) NN_NOEXCEPT
{
    if (m_IsOccupied == false)
    {
        ChangeHandle();
        m_IsOccupied = true;
        m_IsAcquired = false;

        if (StartHsSession(pClient, pProfile).IsSuccess())
        {
            AttachToHdr(pHdr, interfaceIndex);
            return ResultSuccess();
        }
    }

    return nn::ahid::ResultInvalidHandle();
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidInterface::HostDetachDevice(nn::ahid::hdr::Hdr* pHdr) NN_NOEXCEPT
{
    if (m_IsOccupied == true)
    {
        if (m_CtrlSession)
        {
            m_CtrlSession->SignalStateChangeEvent();
        }

        ChangeHandle();
        DetachFromHdr(pHdr);
        StopHsSession();

        m_IsOccupied = false;
        m_IsAcquired = false;

        return ResultSuccess();
    }

    return nn::ahid::ResultInvalidHandle();
}


//////////////////////////////////////////////////////////////////////////////
nn::usb::InterfaceHandle UsbHidInterface::GetInterfaceHandle() NN_NOEXCEPT
{
    return m_HsProfile.ifProfile.handle;
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidInterface::AhidGetCodeBook(uint8_t* pBuffer, uint32_t bufferSize, int32_t deviceHandle) NN_NOEXCEPT
{
    return m_CodeBook.GetCodeBook(pBuffer, bufferSize);
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidInterface::AhidGetString(uint8_t* pBuffer, uint32_t bytes, uint8_t stringIndex, int32_t deviceHandle) NN_NOEXCEPT
{
    Result result = IncHandleSemaphore(deviceHandle);

    if (result.IsSuccess())
    {
        result = GetDeviceString(pBuffer, bytes, stringIndex);

        DecHandleSemaphore();
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidInterface::AhidGetReport(uint32_t reportType, uint32_t reportId, uint8_t* pBuffer, uint32_t bufferSize, int32_t deviceHandle) NN_NOEXCEPT
{
    Result result = IncHandleSemaphore(deviceHandle);

    if (result.IsSuccess())
    {
        size_t bytesTransferred;

        result = Ctrl(
                    &bytesTransferred,
                    pBuffer,
                    0xa1,
                    0x01,
                    (reportType << 8) | reportId,
                    m_HsProfile.ifProfile.ifDesc.bInterfaceNumber,
                    bufferSize
                    );

        if (result.IsSuccess() && (bytesTransferred != bufferSize))
        {
            result = nn::ahid::ResultBufferSize();
        }

        DecHandleSemaphore();
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidInterface::AhidSetReport(uint32_t reportType, uint32_t reportId, uint8_t* pBuffer, uint32_t bufferSize, int32_t deviceHandle) NN_NOEXCEPT
{
    Result result = IncHandleSemaphore(deviceHandle);

    if (result.IsSuccess())
    {
        size_t bytesTransferred;

        result = Ctrl(
                    &bytesTransferred,
                    pBuffer,
                    0x21,
                    0x09,
                    (reportType << 8) | reportId,
                    m_HsProfile.ifProfile.ifDesc.bInterfaceNumber,
                    bufferSize
                    );

        if (result.IsSuccess() && (bytesTransferred != bufferSize))
        {
            result = nn::ahid::ResultBufferSize();
        }

        DecHandleSemaphore();
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidInterface::AhidGetIdle(uint8_t* pIdle, uint8_t reportId, int32_t deviceHandle) NN_NOEXCEPT
{
    Result result = IncHandleSemaphore(deviceHandle);

    if (result.IsSuccess())
    {
        size_t bytesTransferred;

        result = Ctrl(
                    &bytesTransferred,
                    pIdle,
                    0xa1,
                    0x02,
                    reportId,
                    m_HsProfile.ifProfile.ifDesc.bInterfaceNumber,
                    1
                    );

        if (result.IsSuccess() && (bytesTransferred != 1))
        {
            result = nn::ahid::ResultBufferSize();
        }

        DecHandleSemaphore();
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidInterface::AhidSetIdle(uint8_t idle, uint8_t reportId, int32_t deviceHandle) NN_NOEXCEPT
{
    Result result = IncHandleSemaphore(deviceHandle);

    if (result.IsSuccess())
    {
        size_t bytesTransferred;

        result = Ctrl(
                    &bytesTransferred,
                    0,
                    0x21,
                    0x0a,
                    (idle << 8) | reportId,
                    m_HsProfile.ifProfile.ifDesc.bInterfaceNumber,
                    0
                    );

        if (result.IsSuccess() && (bytesTransferred != 0))
        {
            result = nn::ahid::ResultBufferSize();
        }

        DecHandleSemaphore();
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidInterface::AhidGetProtocol(uint8_t* pProtocol, int32_t deviceHandle) NN_NOEXCEPT
{
    Result result = IncHandleSemaphore(deviceHandle);

    if (result.IsSuccess())
    {
        size_t bytesTransferred;

        result = Ctrl(
                    &bytesTransferred,
                    pProtocol,
                    0xa1,
                    0x03,
                    0,
                    m_HsProfile.ifProfile.ifDesc.bInterfaceNumber,
                    1
                    );

        if (result.IsSuccess() && (bytesTransferred != 1))
        {
            result = nn::ahid::ResultBufferSize();
        }

        DecHandleSemaphore();
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidInterface::AhidSetProtocol(uint8_t protocol, int32_t deviceHandle) NN_NOEXCEPT
{
    Result result = IncHandleSemaphore(deviceHandle);

    if (result.IsSuccess())
    {
        size_t bytesTransferred;

        result = Ctrl(
                    &bytesTransferred,
                    0,
                    0x21,
                    0x0b,
                    protocol,
                    m_HsProfile.ifProfile.ifDesc.bInterfaceNumber,
                    0
                    );

        if (result.IsSuccess() && (bytesTransferred != 0))
        {
            result = nn::ahid::ResultBufferSize();
        }

        DecHandleSemaphore();
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidInterface::AhidGetDescriptor(uint8_t* pBuffer, uint32_t bufferSize, uint8_t bmRequestType, uint16_t wValue, uint16_t wIndex, int32_t deviceHandle) NN_NOEXCEPT
{
    Result result = IncHandleSemaphore(deviceHandle);

    if (result.IsSuccess())
    {
        size_t bytesTransferred;

        result = Ctrl(
                    &bytesTransferred,
                    pBuffer,
                    bmRequestType,
                    0x06,
                    wValue,
                    wIndex,
                    bufferSize
                    );

        if (result.IsSuccess())
        {
            result = nn::ahid::ResultBufferSize();
        }

        DecHandleSemaphore();
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidInterface::AhidSetDescriptor(uint8_t* pBuffer, uint32_t bufferSize, uint8_t bmRequestType, uint16_t wValue, uint16_t wIndex, int32_t deviceHandle) NN_NOEXCEPT
{
    Result result = IncHandleSemaphore(deviceHandle);

    if (result.IsSuccess())
    {
        size_t bytesTransferred;

        result = Ctrl(
                    &bytesTransferred,
                    pBuffer,
                    bmRequestType,
                    0x07,
                    wValue,
                    wIndex,
                    bufferSize
                    );

        if (result.IsSuccess())
        {
            result = nn::ahid::ResultBufferSize();
        }

        DecHandleSemaphore();
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidInterface::AhidRead(uint32_t* pBytesTransferred, uint8_t* pBuffer, uint32_t bytes, int32_t deviceHandle, nn::TimeSpan timeout) NN_NOEXCEPT
{
    Result result = IncHandleSemaphore(deviceHandle);

    if (result.IsSuccess())
    {
        if (bytes <= sizeof(m_ReadBuffer))
        {
            size_t bytesTransferred = 0;

            // transfer to aligned buffer
            result = Read(&bytesTransferred, m_ReadBuffer, bytes, timeout);

            if (result.IsSuccess())
            {
                memcpy(pBuffer, m_ReadBuffer, bytesTransferred);
                *pBytesTransferred = bytesTransferred;
            }
        }
        else
        {
            *pBytesTransferred = 0;
            result = nn::ahid::ResultBufferSize();
        }

        DecHandleSemaphore();
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidInterface::AhidWrite(uint32_t* pBytesTransferred, uint8_t* pBuffer, uint32_t bytes, int32_t deviceHandle, nn::TimeSpan timeout) NN_NOEXCEPT
{
    Result result = IncHandleSemaphore(deviceHandle);

    if (result.IsSuccess())
    {
        if (bytes <= sizeof(m_WriteBuffer))
        {
            size_t bytesTransferred = 0;

            // transfer to aligned buffer
            memcpy(m_WriteBuffer, pBuffer, bytes);

            result = Write(&bytesTransferred, m_WriteBuffer, bytes, timeout);

            if (result.IsSuccess())
            {
                *pBytesTransferred = bytesTransferred;
            }
        }
        else
        {
            *pBytesTransferred = 0;
            result = nn::ahid::ResultBufferSize();
        }

        DecHandleSemaphore();
    }

    return result;
}



//////////////////////////////////////////////////////////////////////////////
//  private functions
//////////////////////////////////////////////////////////////////////////////
void UsbHidInterface::ChangeHandle() NN_NOEXCEPT
{
    m_HandleInstance++;

    if (m_HandleInstance == 0)
    {
        m_HandleInstance = 1;
    }
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidInterface::StartHsSession(nn::usb::Host* pClient, nn::usb::InterfaceQueryOutput* pProfile) NN_NOEXCEPT
{
    m_pHsClient = pClient;

    memcpy(&m_HsProfile, pProfile, sizeof(nn::usb::InterfaceQueryOutput));

    return m_HsSession.Initialize(pClient, m_HsProfile.ifProfile.handle);
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidInterface::StopHsSession() NN_NOEXCEPT
{
    Result result;

    if(m_HsSessionRead.IsInitialized())
    {
        if((result=m_HsSessionRead.Finalize()).IsFailure()) return result;
    }

    if(m_HsSessionWrite.IsInitialized())
    {
        if((result=m_HsSessionWrite.Finalize()).IsFailure()) return result;
    }

    return m_HsSession.Finalize();
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidInterface::AttachToHdr(nn::ahid::hdr::Hdr* pHdr, int i) NN_NOEXCEPT
{
    Result result;
    nn::ahid::hdr::DeviceParameters hdrDeviceParameters;

    memset(&hdrDeviceParameters, 0, sizeof(nn::ahid::hdr::DeviceParameters));

    // usagePage and usageId are encoded in HID descriptor
    GetHidDescriptor(); // HID report descriptor is in m_pTransactionBuffer after this returns

    // temp... use parsor later
    hdrDeviceParameters.usagePage       = m_CtrlBuffer[1];
    hdrDeviceParameters.usageId         = m_CtrlBuffer[3];

    // create service name, device handle ...
    sprintf(reinterpret_cast<char*>(hdrDeviceParameters.servicePath), "ahid:cd");
    hdrDeviceParameters.deviceHandle    = (m_HandleIndex << 16) | m_HandleInstance;
    hdrDeviceParameters.busId           = nn::ahid::hdr::AhidBusIdUsb;

    // items in the UsbDeviceDescriptor
    hdrDeviceParameters.vendorId        = m_HsProfile.deviceProfile.deviceDesc.idVendor;
    hdrDeviceParameters.productId       = m_HsProfile.deviceProfile.deviceDesc.idProduct;
    hdrDeviceParameters.versionNumber   = m_HsProfile.deviceProfile.deviceDesc.bcdDevice;

    // string descriptors
    size_t size;
    m_LanguageId = 0;

    // get language id
    result = Ctrl(&size, m_CtrlBuffer, 0x80, 0x06, (0x03 << 8) | 0, 0, 0xff);

    if (result.IsSuccess() && (size >= 4))
    {
        if (
            (m_CtrlBuffer[1] == nn::usb::UsbDescriptorType::UsbDescriptorType_String) &&    // is string descriptor
            (m_CtrlBuffer[0] >= 4) &&                                                       // bLength >= than header + 1 entry
            (m_CtrlBuffer[0] <= size)                                                       // bLength <= size read
            )
        {
            uint8_t entriesCount = (m_CtrlBuffer[0] - 2) / 2;

            if (entriesCount)
            {
                uint16_t *pEntry = reinterpret_cast<uint16_t*>(&m_CtrlBuffer[2]);

                for (uint8_t i = 0; i < entriesCount; i++)
                {
                    // default to the first entry then look for English, use if supported
                    if ((*pEntry == 0x0409) || (i == 0))
                    {
                        memcpy(&m_LanguageId, pEntry, 2);
                    }

                    pEntry++;
                }

                GetDeviceString(hdrDeviceParameters.manufacturer,   64, m_HsProfile.deviceProfile.deviceDesc.iManufacturer);
                GetDeviceString(hdrDeviceParameters.product,        64, m_HsProfile.deviceProfile.deviceDesc.iProduct);
                GetDeviceString(hdrDeviceParameters.serialNumber,   64, m_HsProfile.deviceProfile.deviceDesc.iSerialNumber);
            }
        }
    }

    result = pHdr->AttachDevice(&m_HdrDeviceHandle, &hdrDeviceParameters);

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidInterface::DetachFromHdr(nn::ahid::hdr::Hdr* pHdr) NN_NOEXCEPT
{
    return pHdr->DetachDevice(m_HdrDeviceHandle);
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidInterface::GetDeviceString(uint8_t* pBuffer, uint32_t bufferSize, uint8_t stringIndex) NN_NOEXCEPT
{
    Result result = nn::ahid::ResultBufferSize();

    if (stringIndex && (bufferSize >= 2))
    {
        size_t size;

        result = Ctrl(&size, m_CtrlBuffer, 0x80, 0x06, (0x03 << 8) | stringIndex, m_LanguageId, 64);

        // m_CtrlBuffer is 4K so we can always copy 64
        memcpy(pBuffer, m_CtrlBuffer, 64);

        // adjust string descriptor size if > 64
        if (pBuffer[0] > 64)
        {
            pBuffer[0] = 64;
        }
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidInterface::GetHidDescriptor() NN_NOEXCEPT
{
    Result result;
    size_t size;

    uint16_t totalBytes = m_HsProfile.deviceProfile.cfgDesc.wTotalLength;

    result = Ctrl(&size, m_CtrlBuffer, 0x80, 0x06, 0x0200, 0, totalBytes);

    if (result.IsSuccess() && (size == totalBytes))
    {
        // - traverse the descriptor until we find the HID descriptor for this interface
        // - parse the HID descriptor to create code book

        int32_t bytesRemaining = totalBytes;
        uint8_t* p = m_CtrlBuffer;
        uint8_t interfaceNumber = 0xff;
        UsbHidDescriptorHeader* pUsbHidDescriptorHeader = 0;

        // skip the configuration descriptor
        p += sizeof(nn::usb::UsbConfigDescriptor);
        bytesRemaining -= sizeof(nn::usb::UsbConfigDescriptor);

        while (bytesRemaining > 0)
        {
            nn::usb::UsbDescriptorHeader* pUsbDescriptorHeader = reinterpret_cast<nn::usb::UsbDescriptorHeader*>(p);

            if (pUsbDescriptorHeader->bDescriptorType == 0x04)  // interface desc
            {
                nn::usb::UsbInterfaceDescriptor* pUsbInterfaceDescriptor = reinterpret_cast<nn::usb::UsbInterfaceDescriptor*>(p);

                interfaceNumber = pUsbInterfaceDescriptor->bInterfaceNumber;
            }
            else if (interfaceNumber == m_HsProfile.ifProfile.ifDesc.bInterfaceNumber)
            {
                // endpoint
                if (pUsbDescriptorHeader->bDescriptorType == 0x05)
                {
                    nn::usb::UsbEndpointDescriptor* pUsbEndpointDescriptor = reinterpret_cast<nn::usb::UsbEndpointDescriptor*>(p);

                    if (pUsbEndpointDescriptor->bEndpointAddress & 0x80)    // device to host
                    {
                        m_HsSessionRead.Initialize(&m_HsSession, pUsbEndpointDescriptor, 1, pUsbEndpointDescriptor->wMaxPacketSize);
                    }
                    else
                    {
                        m_HsSessionWrite.Initialize(&m_HsSession, pUsbEndpointDescriptor, 1, pUsbEndpointDescriptor->wMaxPacketSize);
                    }

//                    NN_SDK_LOG("bEndpointAddress    %02x\n", pUsbEndpointDescriptor->bEndpointAddress);
//                    NN_SDK_LOG("wMaxPacketSize      %d\n", pUsbEndpointDescriptor->wMaxPacketSize);
                }

                // HID
                if (pUsbDescriptorHeader->bDescriptorType == 0x21)
                {
                    // HID descriptor comes before endpoint descriptors, lets store it and fetch the HID descriptor later
                    pUsbHidDescriptorHeader = reinterpret_cast<UsbHidDescriptorHeader*>(p);
                }
            }

            p += pUsbDescriptorHeader->bLength;
            bytesRemaining -= pUsbDescriptorHeader->bLength;
        }

        // Get HID descriptor
        if (pUsbHidDescriptorHeader)
        {
            if (pUsbHidDescriptorHeader->bNumDescriptors)
            {
                for (int i = 0; i < pUsbHidDescriptorHeader->bNumDescriptors; i++)
                {
                    // we only care about report descriptor
                    if (pUsbHidDescriptorHeader->descriptorParam[i].bDescriptorType == 0x22)
                    {
                        // we will be writing over the descriptor so lets save the read size for compare
                        uint32_t readSize = pUsbHidDescriptorHeader->descriptorParam[i].wDescriptorLength;

                        result = Ctrl(
                                        &size,
                                        m_CtrlBuffer,
                                        0x81,
                                        0x06,
                                        0x2200,
                                        m_HsProfile.ifProfile.ifDesc.bInterfaceNumber,
                                        readSize
                                        );

                        if (result.IsSuccess() && (size == readSize))
                        {
                            m_CodeBook.Initialize(m_CtrlBuffer, readSize);
                        }

                        break;
                    }
                }
            }
        }
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidInterface::Ctrl(size_t* pBytesTransferred, uint8_t* pBuffer, uint8_t bmRequestType, uint8_t bRequest, uint16_t wValue, uint16_t wIndex, uint16_t wLength) NN_NOEXCEPT
{
    Result result;

    if (wLength > 4096)
    {
        return nn::ahid::ResultBufferSize();
    }

    if (!(bmRequestType & 0x80))   // out: host to device
    {
        memcpy(m_CtrlBuffer, pBuffer, wLength);
    }

    result = m_HsSession.ControlRequest(
                                        pBytesTransferred,
                                        m_CtrlBuffer,
                                        bmRequestType,
                                        bRequest,
                                        wValue,
                                        wIndex,
                                        wLength
                                        );

    if (bmRequestType & 0x80)      // in: device to host
    {
        if (result.IsSuccess())
        {
            memcpy(pBuffer, m_CtrlBuffer, *pBytesTransferred);
        }
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidInterface::Read(size_t* pBytesTransferred, uint8_t* pBuffer, uint32_t bytes, nn::TimeSpan timeout) NN_NOEXCEPT
{
    Result result;

    if (timeout == 0)
    {
        result = m_HsSessionRead.PostBuffer(
                                      pBytesTransferred,
                                      pBuffer,
                                      bytes
                                      );
    }
    else
    {
        uint32_t xferId = 0;
        *pBytesTransferred = 0;
        result = m_HsSessionRead.PostBufferAsync(
                                          &xferId,
                                          pBuffer,
                                          bytes,
                                          0
                                          );
        if(result.IsSuccess())
        {
            uint32_t count;
            nn::usb::XferReport report;
            if (m_HsSessionRead.TimedWaitForCompletion(timeout)) // transaction completed
            {
                result = m_HsSessionRead.GetXferReport(&count, &report, 1);

                if (result.IsSuccess())
                {
                    NN_ABORT_UNLESS(count == 1, "GetXferReport count != 1");
                    *pBytesTransferred = report.transferredSize;
                    result = report.result;
                }
            }
            else // transaction not completed before timeout
            {
                m_HsSessionRead.ClearEndpointHalt();
                m_HsSessionRead.GetXferReport(&count, &report, 1);
                result = nn::ahid::ResultTimeout();
            }
        }
    }

    if(nn::usb::ResultStalled::Includes(result) || nn::usb::ResultTransactionError::Includes(result))
    {
        m_HsSessionRead.ClearEndpointHalt();
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidInterface::Write(size_t* pBytesTransferred, uint8_t* pBuffer, uint32_t bytes, nn::TimeSpan timeout) NN_NOEXCEPT
{
    Result result;

    if (timeout == 0)
    {
        result = m_HsSessionWrite.PostBuffer(
                                          pBytesTransferred,
                                          pBuffer,
                                          bytes
                                          );
    }
    else
    {
        uint32_t xferId = 0;
        *pBytesTransferred = 0;
        result = m_HsSessionWrite.PostBufferAsync(
                                          &xferId,
                                          pBuffer,
                                          bytes,
                                          0
                                          );
        if(result.IsSuccess())
        {
            uint32_t count;
            nn::usb::XferReport report;
            if (m_HsSessionWrite.TimedWaitForCompletion(timeout)) // transaction completed
            {
                result = m_HsSessionWrite.GetXferReport(&count, &report, 1);

                if (result.IsSuccess())
                {
                    NN_ABORT_UNLESS(count == 1, "GetXferReport count != 1");
                    *pBytesTransferred = report.transferredSize;
                    result = report.result;
                }
            }
            else // transaction not completed before timeout
            {
                m_HsSessionWrite.ClearEndpointHalt();
                m_HsSessionWrite.GetXferReport(&count, &report, 1);
                result = nn::ahid::ResultTimeout();
            }
        }
    }

    if(nn::usb::ResultStalled::Includes(result) || nn::usb::ResultTransactionError::Includes(result))
    {
        m_HsSessionWrite.ClearEndpointHalt();
    }

    return result;
}

} // end of namespace usb
} // end of namespace cdhid
} // end of namespace nn
