﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "cdhidUsb_Private.h"
#include "../../ahid/hdr/libnn_hdrServer/ahid_Hdr.h"
#include <nn/nn_SystemThreadDefinition.h>

namespace nn {
namespace cdhid {
namespace usb {

//////////////////////////////////////////////////////////////////////////////
//  public functions
//////////////////////////////////////////////////////////////////////////////
Result UsbHidEnum::Initialize() NN_NOEXCEPT
{
    for (int i = 0; i < USBHID_SERVER_MAX_NUM_DEVICES; i++)
    {
        m_UsbHidInterface[i].Initialize(i);
    }

    nn::os::InitializeEvent(&m_TeardownEvent, false, nn::os::EventClearMode_ManualClear);

    m_Continue = true;

    nn::os::CreateThread(
        &m_Thread,
        EnumThread,
        this,
        m_ThreadStack,
        ENUM_THREAD_STACK_SIZE,
        NN_SYSTEM_THREAD_PRIORITY(cdhid,EnumThread)
        );

    nn::os::SetThreadNamePointer(&m_Thread, NN_SYSTEM_THREAD_NAME(cdhid, EnumThread));
    nn::os::StartThread(&m_Thread);

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidEnum::Finalize() NN_NOEXCEPT
{
    m_Continue = false;

    nn::os::SignalEvent(&m_TeardownEvent);
    nn::os::WaitThread(&m_Thread);
    nn::os::DestroyThread(&m_Thread);
    nn::os::FinalizeEvent(&m_TeardownEvent);

    for (int i = 0; i < USBHID_SERVER_MAX_NUM_DEVICES; i++)
    {
        m_UsbHidInterface[i].Finalize();
    }

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidEnum::AcquireDevice(uint32_t deviceHandle, UsbHidInterface **ppUsbHidInterface) NN_NOEXCEPT
{
    Result result;
    int index;

    result = GetInterfaceIndex(deviceHandle, &index);

    if (result.IsSuccess())
    {
        result = m_UsbHidInterface[index].ClientAcquireDevice(deviceHandle);

        if (result.IsSuccess())
        {
            *ppUsbHidInterface = &m_UsbHidInterface[index];
        }
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidEnum::ReleaseDevice(uint32_t deviceHandle) NN_NOEXCEPT
{
    Result result;
    int index;

    result = GetInterfaceIndex(deviceHandle, &index);

    if (result.IsSuccess())
    {
        result = m_UsbHidInterface[index].ClientReleaseDevice(deviceHandle);
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
//  private functions
//////////////////////////////////////////////////////////////////////////////
void UsbHidEnum::EnumThread(void *arg) NN_NOEXCEPT
{
    UsbHidEnum *pThis = reinterpret_cast<UsbHidEnum*>(arg);

    pThis->m_Hdr.InitializeWith(nn::ahid::hdr::HdrGetDfcSession());
    pThis->m_UsbHs.Initialize();

    nn::usb::DeviceFilter devFilter;

    // Filter for HID class interface
    memset(&devFilter, 0, sizeof(devFilter));
    devFilter.matchFlags        = nn::usb::DeviceFilterMatchFlags::DeviceFilterMatchFlags_InterfaceClass;
    devFilter.bInterfaceClass   = nn::usb::UsbClass::UsbClass_Hid;

    pThis->m_UsbHs.CreateInterfaceAvailableEvent(pThis->m_AttachEvent.GetBase(), nn::os::EventClearMode_ManualClear, 0, &devFilter);
    pThis->m_pDetachEvent = pThis->m_UsbHs.GetInterfaceStateChangeEvent();

    nn::os::InitializeMultiWait(&pThis->m_MultiWait);
    nn::os::InitializeMultiWaitHolder(&pThis->m_MultiWaitHolderTeardown, &pThis->m_TeardownEvent);
    nn::os::InitializeMultiWaitHolder(&pThis->m_MultiWaitHolderAttach, pThis->m_AttachEvent.GetBase());
    nn::os::InitializeMultiWaitHolder(&pThis->m_MultiWaitHolderDetach, pThis->m_pDetachEvent);
    nn::os::LinkMultiWaitHolder(&pThis->m_MultiWait, &pThis->m_MultiWaitHolderTeardown);
    nn::os::LinkMultiWaitHolder(&pThis->m_MultiWait, &pThis->m_MultiWaitHolderAttach);
    nn::os::LinkMultiWaitHolder(&pThis->m_MultiWait, &pThis->m_MultiWaitHolderDetach);

    while (pThis->m_Continue)
    {
        Result result;
        int32_t ifCount;
        nn::usb::InterfaceQueryOutput ifList[USBHID_SERVER_MAX_NUM_DEVICES];
        nn::os::MultiWaitHolderType* holder = nn::os::WaitAny(&pThis->m_MultiWait);

        if (holder == &pThis->m_MultiWaitHolderDetach)
        {
            nn::os::ClearSystemEvent(pThis->m_pDetachEvent);

            result = pThis->m_UsbHs.QueryAcquiredInterfaces(&ifCount, ifList, sizeof(ifList));

            if (result.IsSuccess())
            {
                pThis->ProcessDetach(ifList, ifCount);
            }
        }

        if (holder == &pThis->m_MultiWaitHolderAttach)
        {
            pThis->m_AttachEvent.Clear();

            result = pThis->m_UsbHs.QueryAvailableInterfaces(&ifCount, ifList, sizeof(ifList), &devFilter);

            if (result.IsSuccess())
            {
                pThis->ProcessAttach(ifList, ifCount);
            }
        }
    }

    nn::os::ClearEvent(&pThis->m_TeardownEvent);

    nn::os::UnlinkMultiWaitHolder(&pThis->m_MultiWaitHolderDetach);
    nn::os::UnlinkMultiWaitHolder(&pThis->m_MultiWaitHolderAttach);
    nn::os::UnlinkMultiWaitHolder(&pThis->m_MultiWaitHolderTeardown);
    nn::os::FinalizeMultiWaitHolder(&pThis->m_MultiWaitHolderDetach);
    nn::os::FinalizeMultiWaitHolder(&pThis->m_MultiWaitHolderAttach);
    nn::os::FinalizeMultiWaitHolder(&pThis->m_MultiWaitHolderTeardown);
    nn::os::FinalizeMultiWait(&pThis->m_MultiWait);

    pThis->m_UsbHs.DestroyInterfaceAvailableEvent(pThis->m_AttachEvent.GetBase(), 0);

    pThis->m_UsbHs.Finalize();
    pThis->m_Hdr.Finalize();
}


//////////////////////////////////////////////////////////////////////////////
void UsbHidEnum::ProcessAttach(nn::usb::InterfaceQueryOutput* pProfile, int32_t ifCount) NN_NOEXCEPT
{
    for (int32_t i = 0; i < ifCount; i++)
    {
        for (int j = 0; j < USBHID_SERVER_MAX_NUM_DEVICES; j++)
        {
            Result result = m_UsbHidInterface[j].HostAttachDevice(&m_UsbHs, pProfile, &m_Hdr, i);

            if (result.IsSuccess())
            {
                break;
            }
        }

        pProfile++;
    }
}


//////////////////////////////////////////////////////////////////////////////
void UsbHidEnum::ProcessDetach(nn::usb::InterfaceQueryOutput* pProfile, int32_t ifCount) NN_NOEXCEPT
{
    // Run through attached devices
    for (int i = 0; i < USBHID_SERVER_MAX_NUM_DEVICES; i++)
    {
        nn::usb::InterfaceQueryOutput* pIterateProfile = pProfile;
        bool isAttached = false;

        // For each interface if the handle is not in the attached list then it is detached
        nn::usb::InterfaceHandle interfaceHandle = m_UsbHidInterface[i].GetInterfaceHandle();

        for (int j = 0; j < ifCount; j++)
        {
            if (pIterateProfile->ifProfile.handle == interfaceHandle)
            {
                isAttached = true;
            }

            pIterateProfile++;
        }

        if (isAttached == false)
        {
            m_UsbHidInterface[i].HostDetachDevice(&m_Hdr);
        }
    }
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidEnum::GetInterfaceIndex(int deviceHandle, int *pIndex) NN_NOEXCEPT
{
    int index = deviceHandle >> 16;

    if (index < USBHID_SERVER_MAX_NUM_DEVICES)
    {
        *pIndex = index;

        return ResultSuccess();
    }

    return nn::ahid::ResultInvalidHandle();
}


} // end of namespace usb
} // end of namespace cdhid
} // end of namespace nn
