﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstring>
#include <limits>
#include <memory>
#include <string>
#include <type_traits>

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/settings/factory/settings_Bluetooth.h>
#include <nn/settings/factory/settings_ConfigurationId.h>
#include <nn/settings/factory/settings_MotionSensor.h>
#include <nn/settings/factory/settings_WirelessLan.h>
#include <nn/settings/system/settings_Region.h>
#include <nn/TargetConfigs/build_Base.h>

#include <nn/cal/cal.h>

#include "cal_Crc16.h"
#include "cal_FileSystem.h"
#include "cal_KeyEncryptor.h"
#include "cal_Settings.h"

//!< コマンドを実行します。
#define COMMAND_DO(command)\
    do\
    {\
        const nn::Result& result = command;\
        if (result.IsFailure()) { return result; }\
    } while (NN_STATIC_CONDITION(0))

#define VERIFY_DO(command)\
    do\
    {\
        isSuccess = command;\
        if (!isSuccess) { NN_RESULT_SUCCESS; }\
    } while (NN_STATIC_CONDITION(0))

namespace nn { namespace cal {

namespace {

//!< 未初期化のヘッダか否かを返します。
bool IsUninitializedHeader(
        const CalibrationInfoHeader& header) NN_NOEXCEPT
{
    return header.updateCount == 0 ? true : false;
}

//!< ベリファイ可能なフォーマットか否かを返します。
bool IsVerifiableFormat(
        const CalibrationInfoHeader& header, uint32_t supportedVersion) NN_NOEXCEPT
{
    if (IsUninitializedHeader(header))
    {
        return true;
    }

    return header.version >= supportedVersion ? true : false;
}

//!< 生産時較正情報のヘッダーのベリファイを行います。
bool VerifyCalibrationInfoHeader(const CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& header = value.header;
    sub = (header.magicNumber == CalibrationInfoMagicNumber);
    all &= sub;
    sub = (header.version != 0);
    all &= sub;
    sub = (header.bodySize == sizeof(value.body));
    all &= sub;
    sub = (header.model == CalibrationInfoModel);
    all &= sub;
    sub = true;
    all &= sub;
    sub = (GetCrc16(
               &header,
               sizeof(header) - sizeof(header.crc16) - sizeof(header.bodyHash)
               ) == header.crc16);
    all &= sub;
    ::std::unique_ptr<char[]> bodyHash(new char[sizeof(header.bodyHash)]);
    ::nn::crypto::Sha256Generator generator;
    generator.Initialize();
    generator.Update(&value.body, sizeof(value.body));
    generator.GetHash(bodyHash.get(), sizeof(header.bodyHash));
    sub = (::std::memcmp(&header.bodyHash,
                         bodyHash.get(),
                         sizeof(header.bodyHash)) == 0);
    all &= sub;
    return all;
}

//!< 無線 LAN に関する生産時較正情報のベリファイを行います。
bool VerifyCalibrationInfoWirelessLan(const CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& body = value.body;
    sub = (GetCrc16(&body.wirelessLanCountryCodesBlock,
                    sizeof(body.wirelessLanCountryCodesBlock) -
                    sizeof(body.wirelessLanCountryCodesBlock.crc16)) ==
           body.wirelessLanCountryCodesBlock.crc16);
    all &= sub;
    sub = (GetCrc16(&body.wirelessLanMacAddressBlock.macAddress,
                    sizeof(body.wirelessLanMacAddressBlock.macAddress)) ==
           body.wirelessLanMacAddressBlock.crc16);
    all &= sub;
    return all;
}

//!< 加速度センサに関する生産時較正情報のベリファイを行います。
bool VerifyCalibrationInfoAccelerometer(const CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& body = value.body;
    sub = (GetCrc16(&body.accelerometerOffsetBlock.accelerometerOffset,
                    sizeof(body.accelerometerOffsetBlock.accelerometerOffset)
                    ) == body.accelerometerOffsetBlock.crc16);
    all &= sub;
    sub = (GetCrc16(&body.accelerometerScaleBlock.accelerometerScale,
                    sizeof(body.accelerometerScaleBlock.accelerometerScale)
                    ) == body.accelerometerScaleBlock.crc16);
    all &= sub;
    return all;
}

//!< ジャイロスコープに関する生産時較正情報のベリファイを行います。
bool VerifyCalibrationInfoGyroscope(const CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& body = value.body;
    sub = (GetCrc16(&body.gyroscopeOffsetBlock.gyroscopeOffset,
                    sizeof(body.gyroscopeOffsetBlock.gyroscopeOffset)
                    ) == body.gyroscopeOffsetBlock.crc16);
    all &= sub;
    sub = (GetCrc16(&body.gyroscopeScaleBlock.gyroscopeScale,
                    sizeof(body.gyroscopeScaleBlock.gyroscopeScale)
                    ) == body.gyroscopeScaleBlock.crc16);
    all &= sub;
    return all;
}

//!< デバイス登録に関する生産時較正情報のベリファイを行います。
bool VerifyCalibrationInfoDeviceRegister(const CalibrationInfo& value
                                        ) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& body = value.body;
    sub = (GetCrc16(&body.eccP256DeviceKeyBlock,
                    sizeof(body.eccP256DeviceKeyBlock) -
                    sizeof(body.eccP256DeviceKeyBlock.crc16)) ==
           body.eccP256DeviceKeyBlock.crc16);
    all &= sub;
    sub = (GetCrc16(&body.eccP256DeviceCertificateBlock,
                    sizeof(body.eccP256DeviceCertificateBlock) -
                    sizeof(body.eccP256DeviceCertificateBlock.crc16)) ==
           body.eccP256DeviceCertificateBlock.crc16);
    all &= sub;
    sub = (GetCrc16(&body.eccB233DeviceKeyBlock,
                    sizeof(body.eccB233DeviceKeyBlock) -
                    sizeof(body.eccB233DeviceKeyBlock.crc16)) ==
           body.eccB233DeviceKeyBlock.crc16);
    all &= sub;
    sub = (GetCrc16(&body.eccB233DeviceCertificateBlock,
                    sizeof(body.eccB233DeviceCertificateBlock) -
                    sizeof(body.eccB233DeviceCertificateBlock.crc16)) ==
           body.eccB233DeviceCertificateBlock.crc16);
    all &= sub;
    if (IsVerifiableFormat(value.header, 8))
    {
        sub = (GetCrc16(&body.rsa2048DeviceCertificateBlock,
                        sizeof(body.rsa2048DeviceCertificateBlock) -
                        sizeof(body.rsa2048DeviceCertificateBlock.crc16)) ==
               body.rsa2048DeviceCertificateBlock.crc16);
        all &= sub;
    }
    return all;
}

//!< eTicket 発行に関する生産時較正情報のベリファイを行います。
bool VerifyCalibrationInfoETicket(const CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& body = value.body;
    sub = (GetCrc16(&body.eccP256ETicketKeyBlock,
                    sizeof(body.eccP256ETicketKeyBlock) -
                    sizeof(body.eccP256ETicketKeyBlock.crc16)) ==
           body.eccP256ETicketKeyBlock.crc16);
    all &= sub;
    sub = (GetCrc16(&body.eccP256ETicketCertificateBlock,
                    sizeof(body.eccP256ETicketCertificateBlock) -
                    sizeof(body.eccP256ETicketCertificateBlock.crc16)) ==
           body.eccP256ETicketCertificateBlock.crc16);
    all &= sub;
    sub = (GetCrc16(&body.eccB233ETicketKeyBlock,
                    sizeof(body.eccB233ETicketKeyBlock) -
                    sizeof(body.eccB233ETicketKeyBlock.crc16)) ==
           body.eccB233ETicketKeyBlock.crc16);
    all &= sub;
    sub = (GetCrc16(&body.eccB233ETicketCertificateBlock,
                    sizeof(body.eccB233ETicketCertificateBlock) -
                    sizeof(body.eccB233ETicketCertificateBlock.crc16)) ==
           body.eccB233ETicketCertificateBlock.crc16);
    all &= sub;
    sub = (GetCrc16(&body.rsa2048ETicketKeyBlock,
        sizeof(body.rsa2048ETicketKeyBlock) -
        sizeof(body.rsa2048ETicketKeyBlock.crc16)) ==
        body.rsa2048ETicketKeyBlock.crc16);
    all &= sub;
    sub = (GetCrc16(&body.rsa2048ETicketCertificateBlock,
        sizeof(body.rsa2048ETicketCertificateBlock) -
        sizeof(body.rsa2048ETicketCertificateBlock.crc16)) ==
        body.rsa2048ETicketCertificateBlock.crc16);
    all &= sub;
    return all;
}

//!< SSL クライアント認証に関する生産時較正情報のベリファイを行います。
bool VerifyCalibrationInfoSsl(const CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& body = value.body;
    sub = (GetCrc16(&body.sslKeyBlock,
                    sizeof(body.sslKeyBlock) - sizeof(body.sslKeyBlock.crc16)
                    ) == body.sslKeyBlock.crc16);
    all &= sub;
    sub = (GetCrc16(&body.sslCertificateSizeBlock,
                    sizeof(body.sslCertificateSizeBlock) -
                    sizeof(body.sslCertificateSizeBlock.crc16)) ==
           body.sslCertificateSizeBlock.crc16);
    all &= sub;
    auto size = static_cast<size_t>(
        body.sslCertificateSizeBlock.sslCertificateSize);
    const auto maxSize = sizeof(body.sslCertificateBlock.sslCertificate);
    size = size < maxSize ? size : maxSize;
    ::std::unique_ptr<char[]> sslCertificateHash(
        new char[sizeof(body.sslCertificateBlock.sslCertificateHash)]);
    ::nn::crypto::Sha256Generator generator;
    generator.Initialize();
    generator.Update(&body.sslCertificateBlock.sslCertificate, size);
    generator.GetHash(sslCertificateHash.get(),
                      sizeof(body.sslCertificateBlock.sslCertificateHash));
    sub = (::std::memcmp(
               &body.sslCertificateBlock.sslCertificateHash,
               sslCertificateHash.get(),
               sizeof(body.sslCertificateBlock.sslCertificateHash)) == 0);
    all &= sub;
    return all;
}

//!< 乱数に関する生産時較正情報のベリファイを行います。
bool VerifyCalibrationInfoRandomNumber(const CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& body = value.body;
    ::std::unique_ptr<char[]> randomNumberHash(
        new char[sizeof(body.randomNumberBlock.randomNumberHash)]);
    ::nn::crypto::Sha256Generator generator;
    generator.Initialize();
    generator.Update(&body.randomNumberBlock.randomNumber,
                     sizeof(body.randomNumberBlock.randomNumber));
    generator.GetHash(randomNumberHash.get(),
                      sizeof(body.randomNumberBlock.randomNumberHash));
    sub = (::std::memcmp(
               &body.randomNumberBlock.randomNumberHash,
               randomNumberHash.get(),
               sizeof(body.randomNumberBlock.randomNumberHash)) == 0);
    all &= sub;
    return all;
}

//!< ゲームカード相互認証に関する生産時較正情報のベリファイを行います。
bool VerifyCalibrationInfoGameCard(const CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& body = value.body;
    sub = (GetCrc16(&body.gameCardKeyBlock,
                    sizeof(body.gameCardKeyBlock) -
                    sizeof(body.gameCardKeyBlock.crc16)
                    ) == body.gameCardKeyBlock.crc16);
    all &= sub;
    ::std::unique_ptr<char[]> gameCardCertificateHash(
        new char[
            sizeof(body.gameCardCertificateBlock.gameCardCertificateHash)]);
    ::nn::crypto::Sha256Generator generator;
    generator.Initialize();
    generator.Update(&body.gameCardCertificateBlock.gameCardCertificate,
                     sizeof(body.gameCardCertificateBlock.gameCardCertificate));
    generator.GetHash(gameCardCertificateHash.get(),
                      sizeof(body.gameCardCertificateBlock
                                 .gameCardCertificateHash));
    sub = (::std::memcmp(
               &body.gameCardCertificateBlock.gameCardCertificateHash,
               gameCardCertificateHash.get(),
               sizeof(body.gameCardCertificateBlock.gameCardCertificateHash)
               ) == 0);
    all &= sub;
    return all;
}

//!< 電池 LOT に関する生産時較正情報のベリファイを行います。
bool VerifyCalibrationInfoBatteryLot(const CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& body = value.body;
    sub = (GetCrc16(&body.batteryLotBlock,
        sizeof(body.batteryLotBlock) -
        sizeof(body.batteryLotBlock.crc16)
        ) == body.batteryLotBlock.crc16);
    all &= sub;

    return all;
}

//!< スピーカー補正に関する生産時較正情報のベリファイを行います。
bool VerifyCalibrationInfoSpeaker(const CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& body = value.body;
    sub = (GetCrc16(&body.speakerCalibrationValueBlock,
        sizeof(body.speakerCalibrationValueBlock) -
        sizeof(body.speakerCalibrationValueBlock.crc16)
        ) == body.speakerCalibrationValueBlock.crc16);
    all &= sub;

    return all;
}

//!< リージョンコードに関する生産時較正情報のベリファイを行います
bool VerifyCalibrationInfoRegionCode(const CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& body = value.body;
    sub = (GetCrc16(&body.regionCodeBlock,
        sizeof(body.regionCodeBlock) -
        sizeof(body.regionCodeBlock.crc16)
        ) == body.regionCodeBlock.crc16);
    all &= sub;

    return all;
}

//!< amiibo に関する生産時較正情報のベリファイを行います
bool VerifyCalibrationInfoAmiibo(const CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& body = value.body;
    sub = (GetCrc16(&body.amiiboKeyBlock,
        sizeof(body.amiiboKeyBlock) - sizeof(body.amiiboKeyBlock.crc16)
        ) == body.amiiboKeyBlock.crc16);
    all &= sub;
    sub = (GetCrc16(&body.amiiboEcqvCertificateBlock,
        sizeof(body.amiiboEcqvCertificateBlock) - sizeof(body.amiiboEcqvCertificateBlock.crc16)
        ) == body.amiiboEcqvCertificateBlock.crc16);
    all &= sub;
    sub = (GetCrc16(&body.amiiboEcdsaCertificateBlock,
        sizeof(body.amiiboEcdsaCertificateBlock) - sizeof(body.amiiboEcdsaCertificateBlock.crc16)
        ) == body.amiiboEcdsaCertificateBlock.crc16);
    all &= sub;
    sub = (GetCrc16(&body.amiiboEcqvBlsKeyBlock,
        sizeof(body.amiiboEcqvBlsKeyBlock) - sizeof(body.amiiboEcqvBlsKeyBlock.crc16)
        ) == body.amiiboEcqvBlsKeyBlock.crc16);
    all &= sub;
    sub = (GetCrc16(&body.amiiboEcqvBlsCertificateBlock,
        sizeof(body.amiiboEcqvBlsCertificateBlock) - sizeof(body.amiiboEcqvBlsCertificateBlock.crc16)
        ) == body.amiiboEcqvBlsCertificateBlock.crc16);
    all &= sub;
    sub = (GetCrc16(&body.amiiboEcqvBlsRootCertificateBlock,
        sizeof(body.amiiboEcqvBlsRootCertificateBlock) - sizeof(body.amiiboEcqvBlsRootCertificateBlock.crc16)
        ) == body.amiiboEcqvBlsRootCertificateBlock.crc16);
    all &= sub;

    return all;
}

//!< 本体カラーバリエーション に関する生産時較正情報のベリファイを行います。
bool VerifyCalibrationInfoColorVariation(const CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& body = value.body;
    sub = (GetCrc16(&body.colorVariationBlock,
        sizeof(body.colorVariationBlock) -
        sizeof(body.colorVariationBlock.crc16)
        ) == body.colorVariationBlock.crc16);
    all &= sub;

    return all;
}

//!< 本体モデルタイプに関する生産時較正情報のベリファイを行います。
bool VerifyCalibrationInfoProductModel(const CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& body = value.body;
    sub = (GetCrc16(&body.productModelBlock,
        sizeof(body.productModelBlock) -
        sizeof(body.productModelBlock.crc16)
        ) == body.productModelBlock.crc16);
    all &= sub;

    return all;
}

//!< LCDバックライト出力の写像の係数に関する生産時較正情報のベリファイを行います。
bool VerifyCalibrationInfoLcdBacklightBrightnessMapping(const CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& body = value.body;
    sub = (GetCrc16(&body.lcdBacklightBrightnessMappingBlock,
        sizeof(body.lcdBacklightBrightnessMappingBlock) -
        sizeof(body.lcdBacklightBrightnessMappingBlock.crc16)
        ) == body.lcdBacklightBrightnessMappingBlock.crc16);
    all &= sub;

    return all;
}

//!< 拡張された秘密鍵に関する生産時較正情報のベリファイを行います。
bool VerifyCalibrationInfoExtendedKey(const CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& body = value.body;
    sub = (GetCrc16(&body.extendedEccB233DeviceKeyBlock,
        sizeof(body.extendedEccB233DeviceKeyBlock) -
        sizeof(body.extendedEccB233DeviceKeyBlock.crc16)
        ) == body.extendedEccB233DeviceKeyBlock.crc16);
    all &= sub;
    sub = (GetCrc16(&body.extendedEccB233ETicketKeyBlock,
        sizeof(body.extendedEccB233ETicketKeyBlock) -
        sizeof(body.extendedEccB233ETicketKeyBlock.crc16)
        ) == body.extendedEccB233ETicketKeyBlock.crc16);
    all &= sub;
    sub = (GetCrc16(&body.extendedEccP256ETicketKeyBlock,
        sizeof(body.extendedEccP256ETicketKeyBlock) -
        sizeof(body.extendedEccP256ETicketKeyBlock.crc16)
        ) == body.extendedEccP256ETicketKeyBlock.crc16);
    all &= sub;
    sub = (GetCrc16(&body.extendedGameCardKeyBlock,
        sizeof(body.extendedGameCardKeyBlock) -
        sizeof(body.extendedGameCardKeyBlock.crc16)
        ) == body.extendedGameCardKeyBlock.crc16);
    all &= sub;
    sub = (GetCrc16(&body.extendedRsa2048ETicketKeyBlock,
        sizeof(body.extendedRsa2048ETicketKeyBlock) -
        sizeof(body.extendedRsa2048ETicketKeyBlock.crc16)
        ) == body.extendedRsa2048ETicketKeyBlock.crc16);
    all &= sub;
    sub = (GetCrc16(&body.extendedSslKeyBlock,
        sizeof(body.extendedSslKeyBlock) -
        sizeof(body.extendedSslKeyBlock.crc16)
        ) == body.extendedSslKeyBlock.crc16);
    all &= sub;
    if (IsVerifiableFormat(value.header, 8))
    {
        sub = (GetCrc16(&body.extendedRsa2048DeviceKeyBlock,
            sizeof(body.extendedRsa2048DeviceKeyBlock) -
            sizeof(body.extendedRsa2048DeviceKeyBlock.crc16)
            ) == body.extendedRsa2048DeviceKeyBlock.crc16);
        all &= sub;
    }
    return all;
}

//!< LCD ベンダ ID に関する生産時較正情報のベリファイを行います。
bool VerifyCalibrationInfoLcdVendorId(const CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& body = value.body;
    sub = (GetCrc16(&body.lcdVendorIdBlock,
        sizeof(body.lcdVendorIdBlock) -
        sizeof(body.lcdVendorIdBlock.crc16)
        ) == body.lcdVendorIdBlock.crc16);
    all &= sub;

    return all;
}

//!< USB Type-C Power Source 回路バージョンに関する生産時較正情報のベリファイを行います。
bool VerifyCalibrationInfoUsbTypeCPowerSourceCircuit(const CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    if (IsVerifiableFormat(value.header, 10))
    {
        bool sub = true;
        const auto& body = value.body;
        sub = (GetCrc16(&body.usbTypeCPowerSourceCircuitBlock,
            sizeof(body.usbTypeCPowerSourceCircuitBlock) -
            sizeof(body.usbTypeCPowerSourceCircuitBlock.crc16)
            ) == body.usbTypeCPowerSourceCircuitBlock.crc16);
        all &= sub;
    }
    return all;
}

//!< 生産時較正情報に含まれる秘密鍵を暗号化します。
::nn::Result EncryptPrivateKeys(CalibrationInfo* pOutValue) NN_NOEXCEPT
{
    COMMAND_DO(EncryptExtendedEccB233DeviceKey(pOutValue));
    COMMAND_DO(EncryptExtendedRsa2048DeviceKey(pOutValue));
    COMMAND_DO(EncryptExtendedRsa2048ETicketKey(pOutValue));
    COMMAND_DO(EncryptExtendedSslKey(pOutValue));
    COMMAND_DO(EncryptExtendedGameCardKey(pOutValue));
    COMMAND_DO(EncryptAmiiboKey(pOutValue));
    COMMAND_DO(EncryptAmiiboEcqvBlsKey(pOutValue));

    NN_RESULT_SUCCESS;
}

} // namespace

//!< ZeroInitialize コマンドを実行します。
nn::Result ZeroInitialize() NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());

    return SetCalibrationInfo(*calibrationInfo);
}

//!< FinishUp コマンドを実行します。
nn::Result FinishUp() NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& header = calibrationInfo->header;
    header.magicNumber = CalibrationInfoMagicNumber;
    if (IsUninitializedHeader(header))
    {
        header.version = CalibrationInfoFormatVersion;
    }
    header.bodySize = static_cast<uint32_t>(sizeof(CalibrationInfoBody));
    header.model = CalibrationInfoModel;
    ++(header.updateCount);
    header.crc16 = GetCrc16(
        &header,
        sizeof(header) - sizeof(header.crc16) - sizeof(header.bodyHash));
    ::nn::crypto::Sha256Generator generator;
    generator.Initialize();
    generator.Update(&calibrationInfo->body, sizeof(calibrationInfo->body));
    generator.GetHash(&header.bodyHash, sizeof(header.bodyHash));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));

    NN_RESULT_SUCCESS;
}

//!< Verify コマンドを実行します。
nn::Result Verify(bool &isSuccess) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    VERIFY_DO(VerifyCalibrationInfoHeader(*calibrationInfo));
    const auto& body = calibrationInfo->body;
    VERIFY_DO((GetCrc16(&body.configurationId1Block.configurationId1,
        sizeof(body.configurationId1Block.configurationId1)) ==
        body.configurationId1Block.crc16));
    VERIFY_DO(VerifyCalibrationInfoWirelessLan(*calibrationInfo));
    VERIFY_DO((GetCrc16(&body.bluetoothBdAddressBlock.bdAddress,
        sizeof(body.bluetoothBdAddressBlock.bdAddress)) ==
        body.bluetoothBdAddressBlock.crc16));
    VERIFY_DO(VerifyCalibrationInfoAccelerometer(*calibrationInfo));
    VERIFY_DO(VerifyCalibrationInfoGyroscope(*calibrationInfo));
    VERIFY_DO((GetCrc16(&body.serialNumberBlock,
        sizeof(body.serialNumberBlock) -
        sizeof(body.serialNumberBlock.crc16)) ==
        body.serialNumberBlock.crc16));
    VERIFY_DO(VerifyCalibrationInfoDeviceRegister(*calibrationInfo));
    VERIFY_DO(VerifyCalibrationInfoETicket(*calibrationInfo));
    VERIFY_DO(VerifyCalibrationInfoSsl(*calibrationInfo));
    VERIFY_DO(VerifyCalibrationInfoRandomNumber(*calibrationInfo));
    VERIFY_DO(VerifyCalibrationInfoGameCard(*calibrationInfo));
    VERIFY_DO(VerifyCalibrationInfoBatteryLot(*calibrationInfo));
    VERIFY_DO(VerifyCalibrationInfoSpeaker(*calibrationInfo));
    VERIFY_DO(VerifyCalibrationInfoRegionCode(*calibrationInfo));
    VERIFY_DO(VerifyCalibrationInfoAmiibo(*calibrationInfo));
    VERIFY_DO(VerifyCalibrationInfoColorVariation(*calibrationInfo));
    VERIFY_DO(VerifyCalibrationInfoProductModel(*calibrationInfo));
    VERIFY_DO(VerifyCalibrationInfoLcdBacklightBrightnessMapping(*calibrationInfo));
    VERIFY_DO(VerifyCalibrationInfoExtendedKey(*calibrationInfo));
    VERIFY_DO(VerifyCalibrationInfoLcdVendorId(*calibrationInfo));
    VERIFY_DO(VerifyCalibrationInfoUsbTypeCPowerSourceCircuit(*calibrationInfo));
    isSuccess = true;
    NN_RESULT_SUCCESS;
}

//!< WriteConfigurationId1 コマンドを実行します。
nn::Result WriteConfigurationId1(
    const ::nn::settings::factory::ConfigurationId1& configurationId1) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.configurationId1Block;
    block.configurationId1 = ::nn::settings::factory::ConfigurationId1();
    ::std::memset(&block.configurationId1, 0, sizeof(block.configurationId1));
    ::std::strncpy( reinterpret_cast<char*>(&block.configurationId1),
                    reinterpret_cast<const char*>(&configurationId1),
                    sizeof(block.configurationId1) - 1);
    block.crc16 = GetCrc16(&block.configurationId1,
                           sizeof(block.configurationId1));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteWlanCountryCodes コマンドを実行します。
nn::Result WriteWlanCountryCodes(
    const ::nn::settings::factory::CountryCode *countryCodeArray,
    const size_t arrayLength) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(arrayLength <= 128);
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.wirelessLanCountryCodesBlock;
    ::std::memset(&block, 0, sizeof(block));
    block.count = static_cast<const uint32_t>(arrayLength);
    for (size_t i = 0; i < block.count; ++i)
    {
        const ::std::string& string = countryCodeArray[i].string;
        for (size_t j = 0; j < 2 && j < string.size(); ++j)
        {
            block.countryCodes[i].string[j] = string[j];
        }
    }
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteWlanMacAddress コマンドを実行します。
nn::Result WriteWlanMacAddress(
    const ::nn::settings::factory::MacAddress& macAddress) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.wirelessLanMacAddressBlock;
    block.macAddress = ::nn::settings::factory::MacAddress();
    for (size_t i = 0; i < 6; ++i)
    {
        block.macAddress.octets[i] = macAddress.octets[i];
    }
    block.crc16 = GetCrc16(&block.macAddress, sizeof(block.macAddress));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteBdAddress コマンドを実行します。
nn::Result WriteBdAddress(
    const ::nn::settings::factory::BdAddress& bdAddress) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.bluetoothBdAddressBlock;
    block.bdAddress = ::nn::settings::factory::BdAddress();
    for (size_t i = 0; i < 6; ++i)
    {
        block.bdAddress.octets[i] = bdAddress.octets[i];
    }
    block.crc16 = GetCrc16(&block.bdAddress, sizeof(block.bdAddress));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteAccelerometerOffset コマンドを実行します。
nn::Result WriteAccelerometerOffset(
    const int16_t x, const int16_t y, const int16_t z) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.accelerometerOffsetBlock;
    block.accelerometerOffset.x = x;
    block.accelerometerOffset.y = y;
    block.accelerometerOffset.z = z;
    block.crc16 = GetCrc16(&block.accelerometerOffset,
                           sizeof(block.accelerometerOffset));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteAccelerometerScale コマンドを実行します。
nn::Result WriteAccelerometerScale(
    const int16_t x, const int16_t y, const int16_t z) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.accelerometerScaleBlock;
    block.accelerometerScale.x = x;
    block.accelerometerScale.y = y;
    block.accelerometerScale.z = z;
    block.crc16 = GetCrc16(&block.accelerometerScale,
                           sizeof(block.accelerometerScale));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteGyroscopeOffset コマンドを実行します。
nn::Result WriteGyroscopeOffset(
    const int16_t x, const int16_t y, const int16_t z) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.gyroscopeOffsetBlock;
    block.gyroscopeOffset.x = x;
    block.gyroscopeOffset.y = y;
    block.gyroscopeOffset.z = z;
    block.crc16 = GetCrc16(&block.gyroscopeOffset,
                           sizeof(block.gyroscopeOffset));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteGyroscopeScale コマンドを実行します。
nn::Result WriteGyroscopeScale(
    const int16_t x, const int16_t y, const int16_t z) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.gyroscopeScaleBlock;
    block.gyroscopeScale.x = x;
    block.gyroscopeScale.y = y;
    block.gyroscopeScale.z = z;
    block.crc16 = GetCrc16(&block.gyroscopeScale,
                           sizeof(block.gyroscopeScale));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteSerialNumber コマンドを実行します。
nn::Result WriteSerialNumber(
    const ::nn::settings::factory::SerialNumber& serialNumber) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.serialNumberBlock;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::strncpy( reinterpret_cast<char*>(&block.serialNumber),
                    reinterpret_cast<const char*>(&serialNumber),
                    sizeof(block.serialNumber) - 1);
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteEccP256DeviceKey コマンドを実行します。
nn::Result WriteEccP256DeviceKey(const EccP256DeviceKey& deviceKey) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.eccP256DeviceKeyBlock;
    ::std::memcpy(&block, &deviceKey, sizeof(deviceKey));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteEccP256DeviceCertificate コマンドを実行します。
nn::Result WriteEccP256DeviceCertificate(
    const DeviceCertificate& deviceCertificate) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.eccP256DeviceCertificateBlock;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::memcpy(&block, &deviceCertificate, sizeof(deviceCertificate));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteEccB233DeviceKey コマンドを実行します。
nn::Result WriteEccB233DeviceKey(const EccB233DeviceKey& deviceKey) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.eccB233DeviceKeyBlock;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::memcpy(&block, &deviceKey, sizeof(deviceKey));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteEccB233DeviceCertificate コマンドを実行します。
nn::Result WriteEccB233DeviceCertificate(const DeviceCertificate& deviceCertificate) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.eccB233DeviceCertificateBlock;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::memcpy(&block, &deviceCertificate, sizeof(deviceCertificate));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteEccP256ETicketKey コマンドを実行します。
nn::Result WriteEccP256ETicketKey(const EccP256DeviceKey& eTicketKey) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.eccP256ETicketKeyBlock;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::memcpy(&block, &eTicketKey, sizeof(eTicketKey));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteEccP256ETicketCertificate コマンドを実行します。
nn::Result WriteEccP256ETicketCertificate(
    const ETicketCertificate& eTicketCertificate) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.eccP256ETicketCertificateBlock;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::memcpy(&block, &eTicketCertificate, sizeof(eTicketCertificate));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteEccB233ETicketKey コマンドを実行します。
nn::Result WriteEccB233ETicketKey(const EccB233DeviceKey& eTicketKey) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.eccB233ETicketKeyBlock;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::memcpy(&block, &eTicketKey, sizeof(eTicketKey));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteEccB233ETicketCertificate コマンドを実行します。
nn::Result WriteEccB233ETicketCertificate(const ETicketCertificate& eTicketCertificate) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.eccB233ETicketCertificateBlock;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::memcpy(&block, &eTicketCertificate, sizeof(eTicketCertificate));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteSslKey コマンドを実行します。
nn::Result WriteSslKey(const SslKey& sslKey) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.sslKeyBlock;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::memcpy(&block, &sslKey, sizeof(sslKey));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteSslCertificate コマンドを実行します。
nn::Result WriteSslCertificate(
    const void* pSslCertificate, const size_t certificateSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(certificateSize <= sizeof(SslCertificate));
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& sizeBlock = calibrationInfo->body.sslCertificateSizeBlock;
    sizeBlock = ::std::remove_reference<decltype(sizeBlock)>::type();
    auto& block = calibrationInfo->body.sslCertificateBlock;
    ::std::memset(&block, 0, sizeof(block));
    ::std::memcpy(&block, pSslCertificate, certificateSize);
    sizeBlock.sslCertificateSize = certificateSize;
    sizeBlock.crc16 = GetCrc16(&sizeBlock,
                               sizeof(sizeBlock) - sizeof(sizeBlock.crc16));
    ::nn::crypto::Sha256Generator generator;
    generator.Initialize();
    generator.Update(&block.sslCertificate,
                     static_cast<size_t>(sizeBlock.sslCertificateSize));
    generator.GetHash(&block.sslCertificateHash,
                      sizeof(block.sslCertificateHash));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteRandomNumber コマンドを実行します。
nn::Result WriteRandomNumber(const RandomNumber& randomNumber) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.randomNumberBlock;
    ::std::memset(&block, 0, sizeof(block));
    ::std::memcpy(&block, &randomNumber, sizeof(randomNumber));
    ::nn::crypto::Sha256Generator generator;
    generator.Initialize();
    generator.Update(&block.randomNumber, sizeof(block.randomNumber));
    generator.GetHash(&block.randomNumberHash, sizeof(block.randomNumberHash));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteGameCardKey コマンドを実行します。
nn::Result WriteGameCardKey(const GameCardKey& gamecardKey) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.gameCardKeyBlock;
    ::std::memset(&block, 0, sizeof(block));
    ::std::memcpy(&block, &gamecardKey, sizeof(gamecardKey));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteGameCardCertificate コマンドを実行します。
nn::Result WriteGameCardCertificate(
    const GameCardCertificate& gameCardCertificate) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.gameCardCertificateBlock;
    ::std::memset(&block, 0, sizeof(block));
    ::std::memcpy(&block, &gameCardCertificate, sizeof(gameCardCertificate));
    ::nn::crypto::Sha256Generator generator;
    generator.Initialize();
    generator.Update(&block.gameCardCertificate,
                     sizeof(block.gameCardCertificate));
    generator.GetHash(&block.gameCardCertificateHash,
                      sizeof(block.gameCardCertificateHash));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteRsa2048ETicketKey コマンドを実行します。
nn::Result WriteRsa2048ETicketKey(
    const Rsa2048ETicketKey& rsa2048ETicketKey) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.rsa2048ETicketKeyBlock;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::memcpy(&block, &rsa2048ETicketKey, sizeof(rsa2048ETicketKey));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteRsa2048ETicketCertificate コマンドを実行します。
nn::Result WriteRsa2048ETicketCertificate(
    const Rsa2048ETicketCertificate& rsa2048ETicketCertificate) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.rsa2048ETicketCertificateBlock;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::memcpy(&block, &rsa2048ETicketCertificate, sizeof(rsa2048ETicketCertificate));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteBatteryLot コマンドを実行します。
nn::Result WriteBatteryLot(
    const ::nn::settings::factory::BatteryLot& batteryLot) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.batteryLotBlock;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::strncpy( reinterpret_cast<char*>(&block.batteryLot),
                    batteryLot.string,
                    sizeof(block.batteryLot) - 1);
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteSpeakerCalibrationValue コマンドを実行します。
nn::Result WriteSpeakerCalibrationValue(
    const SpeakerCalibrationValue& speakerCalibrationValue) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.speakerCalibrationValueBlock;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::memcpy(&block, &speakerCalibrationValue, sizeof(speakerCalibrationValue));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteRegionCode コマンドを実行します。
nn::Result WriteRegionCode(const ::nn::settings::system::RegionCode regionCode) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.regionCodeBlock;
    block.regionCode = regionCode;
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteAmiiboKey コマンドを実行します。
nn::Result WriteAmiiboKey(const AmiiboKey& amiiboKey) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.amiiboKeyBlock;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::memcpy(&block, &amiiboKey, sizeof(amiiboKey));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(EncryptAmiiboKey(calibrationInfo.get()));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteAmiiboEcqvCertificate コマンドを実行します。
nn::Result WriteAmiiboEcqvCertificate(
    const AmiiboEcqvCertificate& amiiboEcqvCertificate) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.amiiboEcqvCertificateBlock;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::memcpy(&block, &amiiboEcqvCertificate, sizeof(amiiboEcqvCertificate));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteAmiiboEcdsaCertificate コマンドを実行します。
nn::Result WriteAmiiboEcdsaCertificate(
    const AmiiboEcdsaCertificate& amiiboEcdsaCertificate) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.amiiboEcdsaCertificateBlock;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::memcpy(&block, &amiiboEcdsaCertificate, sizeof(amiiboEcdsaCertificate));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!<  WriteAmiiboEcqvBlsKey コマンドを実行します。
nn::Result WriteAmiiboEcqvBlsKey(
    const AmiiboEcqvBlsKey& amiiboEcqvBlsKey) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.amiiboEcqvBlsKeyBlock;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::memcpy(&block, &amiiboEcqvBlsKey, sizeof(AmiiboEcqvBlsKey));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(EncryptAmiiboEcqvBlsKey(calibrationInfo.get()));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!<  WriteAmiiboEcqvBlsCertificate コマンドを実行します。
nn::Result WriteAmiiboEcqvBlsCertificate(
    const AmiiboEcqvBlsCertificate& amiiboEcqvBlsCertificate) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.amiiboEcqvBlsCertificateBlock;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::memcpy(&block, &amiiboEcqvBlsCertificate, sizeof(AmiiboEcqvBlsCertificate));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!<  WriteAmiiboEcqvBlsRootCertificate コマンドを実行します。
nn::Result WriteAmiiboEcqvBlsRootCertificate(
    const AmiiboEcqvBlsRootCertificate& amiiboEcqvBlsRootCertificate) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.amiiboEcqvBlsRootCertificateBlock;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::memcpy(&block, &amiiboEcqvBlsRootCertificate, sizeof(AmiiboEcqvBlsRootCertificate));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!<  WriteProductModel コマンドを実行します。
nn::Result WriteProductModel(
    const ProductModel& productModel) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.productModelBlock;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::memcpy(&block, &productModel, sizeof(ProductModel));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!<  WriteColorVariation コマンドを実行します。
nn::Result WriteColorVariation(
    const ColorVariation& colorVariation) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.colorVariationBlock;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::memcpy(&block, &colorVariation, sizeof(ColorVariation));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!<  WriteLcdBacklightBrightnessMapping コマンドを実行します。
nn::Result WriteLcdBacklightBrightnessMapping(
    const LcdBacklightBrightnessMapping& lcdBacklightBrightnessMapping) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.lcdBacklightBrightnessMappingBlock;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::memcpy(&block, &lcdBacklightBrightnessMapping, sizeof(LcdBacklightBrightnessMapping));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteExtendedEccB233DeviceKey コマンドを実行します。
nn::Result WriteExtendedEccB233DeviceKey(const ExtendedEccB233DeviceKey& deviceKey) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.extendedEccB233DeviceKeyBlock;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::memcpy(&block, &deviceKey, sizeof(deviceKey));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(EncryptExtendedEccB233DeviceKey(calibrationInfo.get()));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteExtendedEccP256ETicketKey コマンドを実行します。
nn::Result WriteExtendedEccP256ETicketKey(const ExtendedEccP256DeviceKey& eTicketKey) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.extendedEccP256ETicketKeyBlock;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::memcpy(&block, &eTicketKey, sizeof(eTicketKey));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteExtendedEccB233ETicketKey コマンドを実行します。
nn::Result WriteExtendedEccB233ETicketKey(const ExtendedEccB233DeviceKey& eTicketKey) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.extendedEccB233ETicketKeyBlock;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::memcpy(&block, &eTicketKey, sizeof(eTicketKey));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteExtendedRsa2048ETicketKey コマンドを実行します。
nn::Result WriteExtendedRsa2048ETicketKey(
    const ExtendedRsa2048ETicketKey& rsa2048ETicketKey) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.extendedRsa2048ETicketKeyBlock;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::memcpy(&block, &rsa2048ETicketKey, sizeof(rsa2048ETicketKey));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(EncryptExtendedRsa2048ETicketKey(calibrationInfo.get()));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteExtendedSslKey コマンドを実行します。
nn::Result WriteExtendedSslKey(const ExtendedSslKey& sslKey) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.extendedSslKeyBlock;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::memcpy(&block, &sslKey, sizeof(sslKey));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(EncryptExtendedSslKey(calibrationInfo.get()));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteExtendedGameCardKey コマンドを実行します。
nn::Result WriteExtendedGameCardKey(const ExtendedGameCardKey& gamecardKey) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.extendedGameCardKeyBlock;
    ::std::memset(&block, 0, sizeof(block));
    ::std::memcpy(&block, &gamecardKey, sizeof(gamecardKey));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(EncryptExtendedGameCardKey(calibrationInfo.get()));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!<  WriteLcdVendorId コマンドを実行します。
nn::Result WriteLcdVendorId(
    const LcdVendorId& lcdVendorId) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.lcdVendorIdBlock;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::memcpy(&block, &lcdVendorId, sizeof(LcdVendorId));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteExtendedRsa2048DeviceKey コマンドを実行します。
nn::Result WriteExtendedRsa2048DeviceKey(const ExtendedRsa2048DeviceKey& deviceKey) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.extendedRsa2048DeviceKeyBlock;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::memcpy(&block, &deviceKey, sizeof(deviceKey));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(EncryptExtendedRsa2048DeviceKey(calibrationInfo.get()));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteRsa2048DeviceCertificate コマンドを実行します。
nn::Result WriteRsa2048DeviceCertificate(
    const Rsa2048DeviceCertificate& rsa2048DeviceCertificate) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.rsa2048DeviceCertificateBlock;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::memcpy(&block, &rsa2048DeviceCertificate, sizeof(rsa2048DeviceCertificate));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteUsbTypeCPowerSourceCircuitVersion コマンドを実行します。
nn::Result WriteUsbTypeCPowerSourceCircuitVersion(
    const UsbTypeCPowerSourceCircuit& usbTypeCPowerSourceCircuit) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.usbTypeCPowerSourceCircuitBlock;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::memcpy(&block, &usbTypeCPowerSourceCircuit, sizeof(usbTypeCPowerSourceCircuit));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteHousingSubColor コマンドを実行します。
nn::Result WriteHousingSubColor(const HousingColor& housingColor) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.housingSubColorBlock;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::memcpy(&block, &housingColor, sizeof(HousingColor));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteHousingBezelColor コマンドを実行します。
nn::Result WriteHousingBezelColor(const HousingColor& housingColor) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.housingBezelColorBlock;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::memcpy(&block, &housingColor, sizeof(HousingColor));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteHousingMainColor1 コマンドを実行します。
nn::Result WriteHousingMainColor1(const HousingColor& housingColor) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.housingMainColor1Block;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::memcpy(&block, &housingColor, sizeof(HousingColor));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteHousingMainColor2 コマンドを実行します。
nn::Result WriteHousingMainColor2(const HousingColor& housingColor) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.housingMainColor2Block;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::memcpy(&block, &housingColor, sizeof(HousingColor));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< WriteHousingMainColor3 コマンドを実行します。
nn::Result WriteHousingMainColor3(const HousingColor& housingColor) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    auto& block = calibrationInfo->body.housingMainColor3Block;
    block = ::std::remove_reference<decltype(block)>::type();
    ::std::memcpy(&block, &housingColor, sizeof(HousingColor));
    block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< PutCalibrationFile コマンドを実行します。
nn::Result PutCalibrationData(const RawCalibrationInfo& rawCalibrationInfo) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    ::std::memset(calibrationInfo.get(), 0, sizeof(CalibrationInfo));
    ::std::memcpy(calibrationInfo.get(), rawCalibrationInfo.data, sizeof(CalibrationInfo));
    COMMAND_DO(EncryptPrivateKeys(calibrationInfo.get()));
    COMMAND_DO(SetCalibrationInfo(*calibrationInfo));
    NN_RESULT_SUCCESS;
}

//!< ReadCalibrationInfoVersion コマンドを実行します。
nn::Result ReadCalibrationInfoVersion(uint32_t* pOutValue) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    ::std::memset(calibrationInfo.get(), 0, sizeof(CalibrationInfo));
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    const auto& header = calibrationInfo->header;
    *pOutValue = header.version;
    NN_RESULT_SUCCESS;
}

//!< ReadColorVariation コマンドを実行します。
nn::Result ReadColorVariation(ColorVariation* pOutValue) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    ::std::memset(calibrationInfo.get(), 0, sizeof(CalibrationInfo));
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    const auto& body = calibrationInfo->body;
    const auto& block = body.colorVariationBlock;
    *pOutValue = block.colorVariation;
    NN_RESULT_SUCCESS;
}

//!< ReadHousingSubColor コマンドを実行します。
nn::Result ReadHousingSubColor(HousingColor* pOutValue) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    ::std::memset(calibrationInfo.get(), 0, sizeof(CalibrationInfo));
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    const auto& body = calibrationInfo->body;
    const auto& block = body.housingSubColorBlock;
    *pOutValue = block.housingColor;
    NN_RESULT_SUCCESS;
}

//!< ReadHousingBezelColor コマンドを実行します。
nn::Result ReadHousingBezelColor(HousingColor* pOutValue) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    ::std::memset(calibrationInfo.get(), 0, sizeof(CalibrationInfo));
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    const auto& body = calibrationInfo->body;
    const auto& block = body.housingBezelColorBlock;
    *pOutValue = block.housingColor;
    NN_RESULT_SUCCESS;
}

//!< ReadHousingMainColor1 コマンドを実行します。
nn::Result ReadHousingMainColor1(HousingColor* pOutValue) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    ::std::memset(calibrationInfo.get(), 0, sizeof(CalibrationInfo));
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    const auto& body = calibrationInfo->body;
    const auto& block = body.housingMainColor1Block;
    *pOutValue = block.housingColor;
    NN_RESULT_SUCCESS;
}

//!< ReadHousingMainColor2 コマンドを実行します。
nn::Result ReadHousingMainColor2(HousingColor* pOutValue) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    ::std::memset(calibrationInfo.get(), 0, sizeof(CalibrationInfo));
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    const auto& body = calibrationInfo->body;
    const auto& block = body.housingMainColor2Block;
    *pOutValue = block.housingColor;
    NN_RESULT_SUCCESS;
}

//!< ReadHousingMainColor3 コマンドを実行します。
nn::Result ReadHousingMainColor3(HousingColor* pOutValue) NN_NOEXCEPT
{
    ::std::unique_ptr<CalibrationInfo> calibrationInfo(new CalibrationInfo());
    ::std::memset(calibrationInfo.get(), 0, sizeof(CalibrationInfo));
    COMMAND_DO(GetCalibrationInfo(calibrationInfo.get()));
    const auto& body = calibrationInfo->body;
    const auto& block = body.housingMainColor3Block;
    *pOutValue = block.housingColor;
    NN_RESULT_SUCCESS;
}

//!< WriteFile コマンドを実行します。
nn::Result WriteFile(
    void* pWriteData, const size_t writeSize, const ::std::string& filePath) NN_NOEXCEPT
{
    FatPartition fatPartition;
    COMMAND_DO(fatPartition.Mount());
    FatPartitionFile dstFile;
    auto dstHandle = ::nn::fs::FileHandle();
    COMMAND_DO(dstFile.Open(&dstHandle, filePath.c_str(), ::nn::fs::OpenMode_Write));
    COMMAND_DO(CreateFatPartitionFile(dstHandle, pWriteData, writeSize));

    NN_RESULT_SUCCESS;
}

//!< ReadFile コマンドを実行します。
nn::Result ReadFile(void* pOutValue, const size_t readSize, const ::std::string& filePath) NN_NOEXCEPT
{
    FatPartition fatPartition;
    COMMAND_DO(fatPartition.Mount());
    FatPartitionFile srcFile;
    auto srcHandle = ::nn::fs::FileHandle();
    COMMAND_DO(srcFile.Open(&srcHandle, filePath.c_str(), ::nn::fs::OpenMode_Read));
    int64_t fileSize = 0;
    COMMAND_DO(::nn::fs::GetFileSize(&fileSize, srcHandle));
    NN_ABORT_UNLESS(static_cast<const int64_t>(readSize) <=fileSize);
    COMMAND_DO(ReadFatPartitionFile(srcHandle, pOutValue, readSize));

    NN_RESULT_SUCCESS;
}

nn::Result GetFileSize(int64_t* pOutValue, const ::std::string& filePath) NN_NOEXCEPT
{
    FatPartition fatPartition;
    COMMAND_DO(fatPartition.Mount());
    FatPartitionFile srcFile;
    auto srcHandle = ::nn::fs::FileHandle();
    COMMAND_DO(srcFile.Open(&srcHandle, filePath.c_str(), ::nn::fs::OpenMode_Read));
    COMMAND_DO(::nn::fs::GetFileSize(pOutValue, srcHandle));
    NN_RESULT_SUCCESS;
}

//!< DeleteFile コマンドを実行します。
nn::Result DeleteFile(const ::std::string& filePath) NN_NOEXCEPT
{
    FatPartition fatPartition;
    COMMAND_DO(fatPartition.Mount());
    COMMAND_DO(DeleteFatPartitionFile(filePath));
    NN_RESULT_SUCCESS;
}

//!< MakeDirectory コマンドを実行します。
nn::Result CreateDirectory(const ::std::string& directoryPath) NN_NOEXCEPT
{
    FatPartition fatPartition;
    COMMAND_DO(fatPartition.Mount());
    COMMAND_DO(CreateFatPartitionDirectory(directoryPath));
    NN_RESULT_SUCCESS;
}

//!< DeleteDirectory コマンドを実行します。
nn::Result DeleteDirectoryRecursively(
    const ::std::string& directoryPath) NN_NOEXCEPT
{
    FatPartition fatPartition;
    COMMAND_DO(fatPartition.Mount());
    COMMAND_DO(DeleteFatPartitionDirectory(directoryPath));
    NN_RESULT_SUCCESS;
}


nn::Result IsFileExist(bool& outIsExist, const ::std::string& filePath) NN_NOEXCEPT
{
    FatPartition fatPartition;
    COMMAND_DO(fatPartition.Mount());
    COMMAND_DO(IsFatPartitionFileExist(outIsExist, filePath));
    NN_RESULT_SUCCESS;
}

nn::Result IsDirectoryExist(bool& outIsExist, const ::std::string& filePath) NN_NOEXCEPT
{
    FatPartition fatPartition;
    COMMAND_DO(fatPartition.Mount());
    COMMAND_DO(IsFatPartitionDirectoryExist(outIsExist, filePath));
    NN_RESULT_SUCCESS;
}

}}
