﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/sf/sf_Types.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/sf/sf_HipcClientProxyByName.h> // for nn::sf::CreateHipcProxyByName
#include <nn/sf/sf_ExpHeapAllocator.h>      // for nn::sf::ExpHeapStaticAllocator


#include <nn/btm/btm_Types.h>
#include <nn/btm/debug/btm_DebugApi.h>
#include <nn/btm/btm_Result.h>
#include <nn/btm/btm_IBtm.sfdl.h>
#include <nn/btm/btm_ServiceName.h>

namespace nn { namespace btm {

namespace {
nn::sf::SharedPointer<IBtmDebug> g_BtmDebug;
struct  BtmDebugByHipcTag;
typedef nn::sf::ExpHeapStaticAllocator<1024 * 16, BtmDebugByHipcTag> BtmDebugAllocator;
nn::os::MutexType g_Mutex = NN_OS_MUTEX_INITIALIZER(false);
uint8_t g_InitCount = 0;
}   // namespace

// BtmDebugAllocator を静的コンストラクタで初期化するためのヘルパー
class BtmDebugAllocatorInitializer
{
public:

    BtmDebugAllocatorInitializer() NN_NOEXCEPT
    {
        BtmDebugAllocator::Initialize(nn::lmem::CreationOption_NoOption);
    }

} g_BtmDebugAllocatorInitializer;


}}// namespace nn::btm


namespace nn { namespace btm { namespace debug {


void InitializeBtmDebugInterface() NN_NOEXCEPT
{
    LockMutex(&g_Mutex);
    if(g_InitCount == 0)
    {
        NN_SDK_ASSERT(!g_BtmDebug);

        nn::sf::SharedPointer<IBtmDebug> ret;
        auto result = nn::sf::CreateHipcProxyByName<IBtmDebug, BtmDebugAllocator::Policy>(&ret, BtmDebugServiceName);
        NN_ABORT_UNLESS(result.IsSuccess());

        //[Todo]ポインタを直に渡してしまっても良さそう
        g_BtmDebug = ret;
        g_InitCount++;
        NN_UNUSED(result);
    }
    else if(g_InitCount == 255)
    {
        NN_ABORT("[btm]Initialized count over 255");
    }
    else
    {
        //初期化済みの場合は、初期化カウントを増やすのみ
        g_InitCount++;
    }
    UnlockMutex(&g_Mutex);
}

void FinalizeBtmDebugInterface() NN_NOEXCEPT
{
    LockMutex(&g_Mutex);
    if(g_InitCount == 0)
    {
        NN_ABORT("[btm]Initialized count over 0");
    }
    else if(g_InitCount == 1)
    {
        //初期化カウントが残り1の時は末期化する
        NN_SDK_ASSERT(g_BtmDebug);
        g_BtmDebug = nullptr;
        g_InitCount--;
    }
    else
    {
        //初期化カウントが残っている場合はカウントを減らすのみ
        g_InitCount--;
    }
    UnlockMutex(&g_Mutex);
}


//-----------------------------------------
//Discovery
//-----------------------------------------
void RegisterSystemEventForDiscovery(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);
    nn::sf::NativeHandle handle;
    bool isSuccess = g_BtmDebug->AcquireDiscoveryEventImpl(&handle);
    if(isSuccess)
    {
        nn::os::AttachReadableHandleToSystemEvent(pSystemEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_AutoClear);
        handle.Detach();
    }
    else
    {
        NN_ABORT("[btm]There are no system events.\n");
    }
}

nn::Result StartDiscovery() NN_NOEXCEPT
{
    auto result = g_BtmDebug->StartDiscoveryImpl();
    return result;
}

nn::Result CancelDiscovery() NN_NOEXCEPT
{
    auto result = g_BtmDebug->CancelDiscoveryImpl();
    return result;
}

void GetDiscoveredDevices(DevicePropertyList* pList) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pList);
    g_BtmDebug->GetDevicePropertyImpl(pList);
}

//-----------------------------------------
//Pairing
//-----------------------------------------
nn::Result CreateBond(const BdAddress *pAddress) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAddress);
    auto result = g_BtmDebug->CreateBondImpl(*pAddress);
    return result;
}

nn::Result CancelBond(const BdAddress *pAddress) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAddress);
    auto result = g_BtmDebug->CancelBondImpl(*pAddress);
    return result;
}



//-----------------------------------------
//Test
//-----------------------------------------
nn::Result ChangeTsiMode(const BdAddress* pAddress, TsiMode tsiMode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAddress);
    if(tsiMode > TsiMode_10Fd1Td1Si15 && tsiMode != TsiMode_Active)
    {
        return nn::btm::ResultInvalidArgument();
    }
    auto result = g_BtmDebug->SetTsiModeImpl(*pAddress, tsiMode);
    return result;
}

void GeneralTest(int mode) NN_NOEXCEPT
{
    g_BtmDebug->GeneralTestImpl(mode);
}

void GeneralGet(int mode, GeneralInfoList* pGeneralInfoList) NN_NOEXCEPT
{
    g_BtmDebug->GeneralGetImpl(mode, pGeneralInfoList);
}

//-----------------------------------------
//Hid
//-----------------------------------------
void HidConnect(const BdAddress* pAddress) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAddress);
    g_BtmDebug->HidConnectImpl(*pAddress);
}

//-----------------------------------------
// BLE
//-----------------------------------------
bool GetGattClientDisconnectionReason(BleDisconnectionReason* pOutReason, uint32_t connectionHandle, const BdAddress& address) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutReason);

    uint16_t reason;

    bool IsReasonExsisting = g_BtmDebug->GetGattClientDisconnectionReasonImpl(&reason, connectionHandle, address);

    *pOutReason = static_cast<BleDisconnectionReason>(reason);

    return IsReasonExsisting;
}

bool GetBleConnectionParameter(uint16_t* pOutInterval, uint16_t* pOutLatency, uint16_t* pOutTimeout,
                               uint32_t connectionHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutInterval);
    NN_SDK_REQUIRES_NOT_NULL(pOutLatency);
    NN_SDK_REQUIRES_NOT_NULL(pOutTimeout);

    return g_BtmDebug->GetBleConnectionParameterImpl(pOutInterval, pOutLatency, pOutTimeout, connectionHandle);
}

bool GetBleConnectionParameterRequest(uint16_t* pOutIntervalMin, uint16_t* pOutIntervalMax, uint16_t* pOutLatency, uint16_t* pOutTimeout,
                                      uint32_t connectionHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutIntervalMin);
    NN_SDK_REQUIRES_NOT_NULL(pOutIntervalMax);
    NN_SDK_REQUIRES_NOT_NULL(pOutLatency);
    NN_SDK_REQUIRES_NOT_NULL(pOutTimeout);

    return g_BtmDebug->GetBleConnectionParameterRequestImpl(pOutIntervalMin, pOutIntervalMax, pOutLatency, pOutTimeout, connectionHandle);
}

}}} // namespace nn::btm::debug

