﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/bpc/detail/bpc_Log.h>
#include <nn/bpc/driver/bpc.h>
#include <nn/i2c/i2c.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>

#include "bpc_SleepButtonCounter.h"
#include "bpc_ExternalRtc.h"
#include "bpc_PmicAccessor-hardware.nx.h"

namespace nn { namespace bpc { namespace driver { namespace detail {

namespace {

// 本体起動直後から押しっぱなしだったときには、押下時間の計測開始が実際より遅れるので、少し早めにセットしておいたほうが良い
const int           SetQuasiOffTimerCount = 5;

// NOTE: NX では使われていない
const int           ShutdownCount = 7;

nn::os::EventType   g_InitializedEvent;
nn::os::ThreadType  g_SleepButtonCounterThread;
SleepButtonState    g_SleepButtonState = SleepButtonState_Released;

nn::os::SystemEventType g_SleepButtonShutdownEvent;
nn::os::SystemEventType g_SleepButtonManualResetWarningEvent;
nn::os::SystemEventType g_SleepButtonStateChangedEvent;

const size_t ThreadStackSize = 8192;
NN_ALIGNAS(4096) uint8_t g_ThreadStack[ThreadStackSize];

nn::i2c::I2cSession* pg_I2cSession = nullptr;

nn::os::MessageQueueType g_OnOffMessageQueue;
const int OnOffMessageQueueBufferCount = 0x20;
uintptr_t g_OnOffMessageQueueBuffer[OnOffMessageQueueBufferCount];

bool ReadEn0State() NN_NOEXCEPT
{
    const uint8_t maskEn0 = 0x04;
    const uint8_t offsetOnOffStat = 0x15;
    uint8_t regOnOffStat = 0x00;

    // initialize state
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::ReadSingleRegister(&regOnOffStat, *pg_I2cSession, &offsetOnOffStat));

    // current state
    return (regOnOffStat & maskEn0) ? true : false;
}

nn::Result SetQuasiOffTimer(bool enable) NN_NOEXCEPT
{
    static bool s_IsEnabled = false;

    if ( !s_IsEnabled && enable )
    {
        int64_t currentTime;
        NN_RESULT_DO(GetRtcTime(&currentTime));
        int64_t targetTime = currentTime + 15; // 現在から 15 秒後

        NN_RESULT_DO(EnableRtcAlarm(targetTime, RtcAlarmType::QuasiOff));
        s_IsEnabled = true;
    }
    else if ( s_IsEnabled && !enable )
    {
        NN_RESULT_DO(DisableRtcAlarm(RtcAlarmType::QuasiOff));
        s_IsEnabled = false;
    }

    NN_RESULT_SUCCESS;
}

void SleepButtonCounterThread(void *arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    nn::os::SignalEvent(&g_InitializedEvent);

    bool en0State = ReadEn0State();

    int count = 0;
    TimeSpan span = nn::TimeSpan::FromMilliSeconds(1000LL);
    nn::os::Tick spanTick = nn::os::ConvertToTick(span);
    nn::os::Tick currentTick;
    nn::os::Tick offsetTick;

    //nn::os::Tick baseTick = nn::os::ConvertToTick(nn::TimeSpan::FromMilliSeconds(0LL));
    nn::os::Tick baseTick = nn::os::GetSystemTick();

    while ( 1 )
    {
        if ( en0State )
        {
            uintptr_t data = 0;
            bool received = false;
            currentTick = nn::os::GetSystemTick();

            // assuming Tick operator manage overflow issues
            offsetTick = currentTick - baseTick;

            if ( offsetTick < spanTick )
            {
                received = nn::os::TimedReceiveMessageQueue(&data, &g_OnOffMessageQueue,
                                                            nn::os::ConvertToTimeSpan(spanTick - offsetTick));
                // NN_DETAIL_BPC_TRACE("1: received %d. data 0x%x\n", received, data);
            }

            if ( !received )
            {
                count++;
                baseTick += spanTick;
                // NN_DETAIL_BPC_TRACE("1: Count %d. baseTick %lld\n", count, baseTick);
                if ( count == SetQuasiOffTimerCount )
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(SetQuasiOffTimer(true)); // 強制電源断に備えて、シャットダウン完了後に擬似オフに行くようにしておく
                }
                if ( count == ShutdownCount )
                {
                    nn::os::SignalSystemEvent(&g_SleepButtonShutdownEvent);
                }
            }
            else if ( data == MessageOnOff_Finalize )
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(SetQuasiOffTimer(false));
                break;
            }
            else if ( (data == MessageOnOff_Rising) || (data == MessageOnOff_Falling) || (data == MessageOnOff_Update) )
            {
                en0State = ReadEn0State();
                // NN_DETAIL_BPC_TRACE("1: en0State %d.\n", en0State);

                SleepButtonState oldState = g_SleepButtonState;
                g_SleepButtonState = (en0State) ? SleepButtonState_Pushed : SleepButtonState_Released;
                if ( g_SleepButtonState != oldState )
                {
                    // NN_DETAIL_BPC_TRACE("1: Signaling\n");
                    nn::os::SignalSystemEvent(&g_SleepButtonStateChangedEvent);
                }

                NN_ABORT_UNLESS_RESULT_SUCCESS(SetQuasiOffTimer(false));
            }
            // SIGLO-43468 : 仕様変更により 1sec はレギュレータドライバ上でマスクされて無効になっています。
            else if ( data == MessageOnOff_1sec )
            {
                //nn::os::SignalSystemEvent(&g_SleepButton1secEvent);
                //NN_DETAIL_BPC_TRACE("1sec Interrupt.\n");
            }
            // SIGLO-43468 : 仕様変更により ManualResetWarning はレギュレータドライバ上でマスクされて無効になっています。
            else if ( data == MessageOnOff_ManualResetWarning )
            {
                nn::os::SignalSystemEvent(&g_SleepButtonManualResetWarningEvent);
                NN_DETAIL_BPC_TRACE("ManualReset Interrupt.\n");
            }
        }
        else
        {
            count = 0;

            uintptr_t data = 0;
            nn::os::ReceiveMessageQueue(&data, &g_OnOffMessageQueue);
            // NN_DETAIL_BPC_TRACE("2: received. data 0x%x\n", data);

            if ( data == MessageOnOff_Finalize )
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(SetQuasiOffTimer(false));
                break;
            }
            else if ( (data == MessageOnOff_Rising) || (data == MessageOnOff_Falling) || (data == MessageOnOff_Update) )
            {
                en0State = ReadEn0State();
                // NN_DETAIL_BPC_TRACE("2: en0State %d.\n", en0State);

                SleepButtonState oldState = g_SleepButtonState;
                g_SleepButtonState = (en0State) ? SleepButtonState_Pushed : SleepButtonState_Released;
                if ( g_SleepButtonState != oldState )
                {
                    // NN_DETAIL_BPC_TRACE("2: Signaling\n");
                    nn::os::SignalSystemEvent(&g_SleepButtonStateChangedEvent);
                }

                baseTick = nn::os::GetSystemTick();
                // NN_DETAIL_BPC_TRACE("2: baseTick %lld\n", count, baseTick);

                NN_ABORT_UNLESS_RESULT_SUCCESS(SetQuasiOffTimer(false));
            }
            else if ( data == MessageOnOff_1sec )
            {
                //nn::os::SignalSystemEvent(&g_SleepButton1secEvent);
                //NN_DETAIL_BPC_TRACE("1sec Interrupt.\n");
            }
            // SIGLO-43468 : 仕様変更により ManualResetWarning はマスクされて無効になっています。
            else if ( data == MessageOnOff_ManualResetWarning )
            {
                nn::os::SignalSystemEvent(&g_SleepButtonManualResetWarningEvent);
                NN_DETAIL_BPC_TRACE("ManualReset Interrupt.\n");
            }
        }
    }
    NN_DETAIL_BPC_TRACE("Sleep Button Counter Thread is terminating.\n");
}

void InitializeSleepButtonCounterThread() NN_NOEXCEPT
{
    auto result = nn::os::CreateThread(&g_SleepButtonCounterThread, SleepButtonCounterThread,
                                  nullptr, g_ThreadStack, ThreadStackSize,
                                  NN_SYSTEM_THREAD_PRIORITY(bpc, SleepButtonCounter));

    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::os::SetThreadNamePointer(&g_SleepButtonCounterThread, NN_SYSTEM_THREAD_NAME(bpc, SleepButtonCounter));

    nn::os::StartThread(&g_SleepButtonCounterThread);

    NN_DETAIL_BPC_TRACE("Waiting for initialization of Sleep Button Counter Thread.\n");

    NN_ABORT_UNLESS(nn::os::TimedWaitEvent(&g_InitializedEvent, nn::TimeSpan::FromMilliSeconds(5000LL)),
                                           "Failed to initialize Sleep Button Counter Thread.");

    NN_DETAIL_BPC_TRACE("Sleep Button Counter Thread has been initialized.\n");
}

void FinalizeSleepButtonCounterThread() NN_NOEXCEPT
{
    nn::os::JamMessageQueue(&g_OnOffMessageQueue, MessageOnOff_Finalize);

    nn::os::WaitThread(&g_SleepButtonCounterThread);
    nn::os::DestroyThread(&g_SleepButtonCounterThread);

    NN_DETAIL_BPC_TRACE("Sleep Button Counter Thread has been terminated.\n");
}

} // namespace

void InitializeSleepButtonCounter() NN_NOEXCEPT
{
    pg_I2cSession = GetPmicI2cSession();

    nn::os::InitializeMessageQueue(&g_OnOffMessageQueue, g_OnOffMessageQueueBuffer, OnOffMessageQueueBufferCount);

    nn::os::CreateSystemEvent(&g_SleepButtonShutdownEvent, nn::os::EventClearMode_ManualClear, true);
    nn::os::CreateSystemEvent(&g_SleepButtonManualResetWarningEvent, nn::os::EventClearMode_ManualClear, true);
    nn::os::CreateSystemEvent(&g_SleepButtonStateChangedEvent, nn::os::EventClearMode_ManualClear, true);

    nn::os::InitializeEvent(&g_InitializedEvent, false, nn::os::EventClearMode_AutoClear);

    // start thread
    InitializeSleepButtonCounterThread();
}

void FinalizeSleepButtonCounter() NN_NOEXCEPT
{
    // stop thread
    FinalizeSleepButtonCounterThread();

    nn::os::FinalizeEvent(&g_InitializedEvent);

    nn::os::DestroySystemEvent(&g_SleepButtonStateChangedEvent);
    nn::os::DestroySystemEvent(&g_SleepButtonManualResetWarningEvent);
    nn::os::DestroySystemEvent(&g_SleepButtonShutdownEvent);

    nn::os::FinalizeMessageQueue(&g_OnOffMessageQueue);

    pg_I2cSession = nullptr;
}

nn::os::MessageQueueType* GetSleepButtonCounterMessageQueue() NN_NOEXCEPT
{
    return &g_OnOffMessageQueue;
}

void GetSleepButtonState(SleepButtonState* pOutState) NN_NOEXCEPT
{
    *pOutState = g_SleepButtonState;
}

void StartSleepButtonStateUpdate() NN_NOEXCEPT
{
    nn::os::JamMessageQueue(&g_OnOffMessageQueue, MessageOnOff_Update);
}

void GetPowerButtonEventPtr(nn::os::SystemEventType** pOutEventPtr, EventTarget target) NN_NOEXCEPT
{
    switch ( target )
    {
    case EventTarget_SleepButtonShutdown:
        *pOutEventPtr = &g_SleepButtonShutdownEvent;
        break;
    case EventTarget_SleepButtonManualResetWarning:
        *pOutEventPtr = &g_SleepButtonManualResetWarningEvent;
        break;
    case EventTarget_SleepButtonStateChanged:
        *pOutEventPtr = &g_SleepButtonStateChangedEvent;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

}}}} // namespace nn::bpc::driver::detail
