﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/os.h>
#include <nn/os/os_Mutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/bpc/bpc_ShimInternal.h>
#include <nn/bpc/bpc_BoardPowerControl.h>
#include <nn/bpc/bpc_WakeupConfig.h>
#include <nn/bpc/bpc_WakeupConfigServer.h>
#include <nn/bpc/bpc_RtcTypes.h>

#include "bpc_GetManagerByHipc.h"

namespace nn { namespace bpc {

void InitializeWakeupConfig() NN_NOEXCEPT
{
    InitializeBoardPowerControl();
}

void FinalizeWakeupConfig() NN_NOEXCEPT
{
    FinalizeBoardPowerControl();
}

NN_DEPRECATED nn::Result CreateWakeupTimer(int *pOutTimerHandle, nn::TimeSpan interval) NN_NOEXCEPT
{
    return CreateWakeupTimer(pOutTimerHandle, interval, WakeupTimerType_BackgroundTask);
}

nn::Result CreateWakeupTimer(int *pOutTimerHandle, nn::TimeSpan interval, WakeupTimerType type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutTimerHandle);
    NN_SDK_REQUIRES_RANGE(interval.GetSeconds(), MinWakeupIntervalInSeconds, MaxRtcTimeInSeconds);
    auto pManager = GetInternalBoardPowerControlManager();
    NN_SDK_ASSERT_NOT_NULL(pManager);
    NN_RESULT_DO(pManager->CreateWakeupTimerEx(pOutTimerHandle, interval, type));
    NN_RESULT_SUCCESS;
}

nn::Result CancelWakeupTimer(int timerHandle) NN_NOEXCEPT
{
    auto pManager = GetInternalBoardPowerControlManager();
    NN_SDK_ASSERT_NOT_NULL(pManager);
    NN_RESULT_DO(pManager->CancelWakeupTimer(timerHandle));
    NN_RESULT_SUCCESS;
}

nn::Result EnableWakeupTimerOnDevice(bool *pOutIsEnabled, int* pOutTimerHandle, WakeupTimerType* pOutTimerType) NN_NOEXCEPT
{
    auto pManager = GetInternalBoardPowerControlManager();
    NN_SDK_ASSERT_NOT_NULL(pManager);

    bool isEnabled;
    int handle;
    WakeupTimerType type;
    NN_RESULT_DO(pManager->EnableWakeupTimerOnDevice(&isEnabled, &handle, &type));
    if ( pOutIsEnabled )
    {
        *pOutIsEnabled = isEnabled;
    }
    if ( pOutTimerHandle )
    {
        *pOutTimerHandle = handle;
    }
    if ( pOutTimerType )
    {
        *pOutTimerType = type;
    }
    NN_RESULT_SUCCESS;
}

WakeupTimerType GetLastEnabledWakeupTimerType() NN_NOEXCEPT
{
    auto pManager = GetInternalBoardPowerControlManager();
    NN_SDK_ASSERT_NOT_NULL(pManager);

    WakeupTimerType type;
    NN_ABORT_UNLESS_RESULT_SUCCESS(pManager->GetLastEnabledWakeupTimerType(&type));
    return type;
}

nn::Result CleanAllWakeupTimers(WakeupTimerType type) NN_NOEXCEPT
{
    auto pManager = GetInternalBoardPowerControlManager();
    NN_SDK_ASSERT_NOT_NULL(pManager);
    NN_RESULT_DO(pManager->CleanAllWakeupTimers(type));
    NN_RESULT_SUCCESS;
}

}} // namepsace nn::bpc
