﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/oe.h>

#include <nn/bluetooth/bluetooth_LeApi.h>
#include <nn/bluetooth/bluetooth_ProprietaryLeApi.h>

#include <nn/bluetooth/bluetooth_Types.h>
#include <nn/bluetooth/bluetooth_UserApi.h>

#include <nn/btm/user/btm_UserTypes.h>
#include <nn/btm/user/btm_UserApi.h>

#include "bluetooth_Utility.h"

namespace nn { namespace bluetooth {
    static BleGattOperationStatus ConvertBluetoothLeStatus(BluetoothLeStatus status)
    {
        switch (status)
        {
        case BluetoothLeStatus::BT_OK:
            return BleGattOperationStatus::BleGattOperationStatus_Success;

        case BluetoothLeStatus::BT_ERR_CLIENT:
            return BleGattOperationStatus::BleGattOperationStatus_ClientError;

        case BluetoothLeStatus::BT_ERR_BUSY:
        case BluetoothLeStatus::BT_GATT_ERR_BUSY:
            return BleGattOperationStatus::BleGattOperationStatus_ClientBusy;

        case BluetoothLeStatus::BT_BLE_ERR_INVALID_PARAM:
        case BluetoothLeStatus::BT_GATT_ERR_INVALID_PARAM:
        case BluetoothLeStatus::BT_ERR_BAD_PARAM:
        case BluetoothLeStatus::BT_GATT_ERR_INVALID_ATTR_LEN:
            return BleGattOperationStatus::BleGattOperationStatus_InvalidParameter;

        case BluetoothLeStatus::BT_GATT_ERR_INVALID_HANDLE:
            return BleGattOperationStatus::BleGattOperationStatus_InvalidHandle;

        case BluetoothLeStatus::BT_GATT_ERR_READ_PERMISSION:
            return BleGattOperationStatus::BleGattOperationStatus_ReadNotPermitted;

        case BluetoothLeStatus::BT_GATT_ERR_WRITE_PERMISSION:
            return BleGattOperationStatus::BleGattOperationStatus_WriteNotPermitted;

        case BluetoothLeStatus::BT_GATT_ERR_INVALID_AUTH:
            return BleGattOperationStatus::BleGattOperationStatus_AuthenticationNotEnough;

        case BluetoothLeStatus::BT_GATT_ERR_NOT_FOUND:
            return BleGattOperationStatus::BleGattOperationStatus_AttributeNotFound;
        case BluetoothLeStatus::BT_GATT_ERR_NO_RESULT:
            return BleGattOperationStatus::BleGattOperationStatus_OperationResultNotFound;
        default:
            return BleGattOperationStatus::BleGattOperationStatus_UnknownError;
        }
    }

    void InitializeBle() NN_NOEXCEPT
    {
        // Create IPC session with bluetooth driver
        user::InitializeBluetoothUserInterface();

        // Create IPC sesstion with BTM
        nn::btm::user::InitializeBtmUserInterface();
    }

    void AcquireBleScanEvent(nn::os::SystemEventType* pOutSystemEvent) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pOutSystemEvent);

        nn::btm::user::AcquireBleScanEvent(pOutSystemEvent);
    }

    void GetBleScanParameter(BleAdvertisePacketParameter* pOutParameter, int parameterId) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pOutParameter);
        NN_ABORT_UNLESS_RANGE(parameterId, 0x0000, 0x10000);

        nn::btm::user::BleAdvFilterForGeneral filter;

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::user::GetBleScanFilterParameter(&filter, static_cast<uint16_t>(parameterId)));

        memcpy(pOutParameter->manufacturerId, filter.manufacturerId, BleNnAdvertiseManufacturerIdSize);
        memcpy(pOutParameter->clientId, filter.clientId, BleNnAdvertiseManufactureClientIdSize);
        memcpy(pOutParameter->serverId, filter.serverId, BleNnAdvertiseManufactureServerIdSize);
    }

    void GetBleScanParameter(GattAttributeUuid* pOutUuid, int parameterId) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pOutUuid);
        NN_ABORT_UNLESS_RANGE(parameterId, 0x0000, 0x10000);

        nn::btm::user::BleAdvFilterForSmartDevice filter;

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::user::GetBleScanFilterParameter(&filter, static_cast<uint16_t>(parameterId)));

        memcpy(pOutUuid->uu.uuid128, filter.uu.uuid128, GattAttributeUuidLength_128);
    }

    nn::Result StartBleScanGeneral(const BleAdvertisePacketParameter& scanParameter) NN_NOEXCEPT
    {
        nn::btm::user::BleAdvFilterForGeneral filter;

        memcpy(filter.manufacturerId, scanParameter.manufacturerId, BleNnAdvertiseManufacturerIdSize);
        memcpy(filter.clientId, scanParameter.clientId, BleNnAdvertiseManufactureClientIdSize);
        memcpy(filter.serverId, scanParameter.serverId, BleNnAdvertiseManufactureServerIdSize);

        auto result = btm::user::StartBleScanForGeneral(filter);

        return ConvertResult(result);
    }

    nn::Result StopBleScanGeneral() NN_NOEXCEPT
    {
        auto result = btm::user::StopBleScanForGeneral();

        return ConvertResult(result);
    }

    int GetBleScanResult(BleScanResult* pOutScanResult, int count) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pOutScanResult);
        NN_ABORT_UNLESS_GREATER_EQUAL(count, 0);

        BleScanResult resultGeneral[BleScanResultCountMax / 2];
        BleScanResult resultSmartDevice[BleScanResultCountMax / 2];

        uint8_t countGeneral =  nn::btm::user::GetBleScanResultsForGeneral(resultGeneral, static_cast<uint8_t>(NN_ARRAY_SIZE(resultGeneral)));
        uint8_t countSmartDevice = nn::btm::user::GetBleScanResultsForSmartDevice(resultSmartDevice, static_cast<uint8_t>(NN_ARRAY_SIZE(resultSmartDevice)));

        int indexGeneral = 0;
        int indexSmartDevice = 0;

        for (int i = 0; i < count; ++i)
        {
            if (indexGeneral < countGeneral && indexSmartDevice < countSmartDevice)
            {
                if (resultGeneral[indexGeneral].rssi >= resultSmartDevice[indexSmartDevice].rssi)
                {
                    pOutScanResult[i] = resultGeneral[indexGeneral];
                    indexGeneral++;
                }
                else
                {
                    pOutScanResult[i] = resultSmartDevice[indexSmartDevice];
                    indexSmartDevice++;
                }
            }
            else if (indexGeneral < countGeneral && countSmartDevice <= indexSmartDevice)
            {
                pOutScanResult[i] = resultGeneral[indexGeneral];
                indexGeneral++;
            }
            else if (countGeneral <= indexGeneral && indexSmartDevice < countSmartDevice)
            {
                pOutScanResult[i] = resultSmartDevice[indexSmartDevice];
                indexSmartDevice++;
            }
            else
            {
                break;
            }
        }

        for (int i = indexGeneral + indexSmartDevice; i < count; ++i)
        {
            pOutScanResult[i] = BleScanResult();
        }

        return indexGeneral + indexSmartDevice;
    }

    nn::Result EnableBleAutoConnection(const BleAdvertisePacketParameter& scanParameter) NN_NOEXCEPT
    {
        nn::btm::user::BleAdvFilterForGeneral filter;

        memcpy(filter.manufacturerId, scanParameter.manufacturerId, BleNnAdvertiseManufacturerIdSize);
        memcpy(filter.clientId, scanParameter.clientId, BleNnAdvertiseManufactureClientIdSize);
        memcpy(filter.serverId, scanParameter.serverId, BleNnAdvertiseManufactureServerIdSize);

        auto result = btm::user::StartBleScanForPaired(filter);

        return ConvertResult(result);
    }

    nn::Result DisableBleAutoConnection() NN_NOEXCEPT
    {
        auto result = btm::user::StopBleScanForPaired();

        return ConvertResult(result);
    }

    nn::Result StartBleScanSmartDevice(const GattAttributeUuid& uuid) NN_NOEXCEPT
    {
        nn::btm::user::BleAdvFilterForSmartDevice filter;

        filter = uuid;

        auto result = btm::user::StartBleScanForSmartDevice(filter);

        return ConvertResult(result);
    }

    nn::Result StopBleScanSmartDevice() NN_NOEXCEPT
    {
        auto result = btm::user::StopBleScanForSmartDevice();

        return ConvertResult(result);
    }

    void AcquireBleConnectionStateChangedEvent(nn::os::SystemEventType* pOutSystemEvent) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pOutSystemEvent);

        btm::user::AcquireBleConnectionEvent(pOutSystemEvent);
    }

    nn::Result ConnectToGattServer(const Address& bdAddress) NN_NOEXCEPT
    {
        auto result = btm::user::BleConnect(bdAddress);

        return ConvertResult(result);
    }

    nn::Result DisconnectFromGattServer(uint32_t connectionHandle) NN_NOEXCEPT
    {
        auto result = btm::user::BleDisconnect(connectionHandle);

        return ConvertResult(result);
    }

    int GetBleConnectionInfoList(BleConnectionInfo* pOutConnectionInfo, int count) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pOutConnectionInfo);
        NN_ABORT_UNLESS_GREATER_EQUAL(count, 0);

        return btm::user::BleGetConnectionState(pOutConnectionInfo, static_cast<uint8_t>(count));
    }

    void AcquireBleServiceDiscoveryEvent(nn::os::SystemEventType* pOutSystemEvent) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pOutSystemEvent);

         btm::user::AcquireBleServiceDiscoveryEvent(pOutSystemEvent);
    }

    int GetGattServices(GattService* pOutGatttServices, int count, uint32_t connectionHandle) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pOutGatttServices);
        NN_ABORT_UNLESS_GREATER_EQUAL(count, 0);

        nn::btm::user::GattService btmServices[GattAttributeCountMaxClient];

        uint8_t inNum = static_cast<uint8_t>(count);
        uint8_t serviceNum = nn::btm::user::GetGattServices(btmServices, NN_ARRAY_SIZE(btmServices), connectionHandle);

        serviceNum = (inNum < serviceNum) ? inNum : serviceNum;

        for (int i = 0; i < serviceNum; ++i)
        {
            pOutGatttServices[i] = GattService(btmServices[i].uuid, btmServices[i].handle, connectionHandle, btmServices[i].instanceId, btmServices[i].endGroupHandle, btmServices[i].isPrimaryService);
        }

        return serviceNum;
    }

    bool GetGattService(GattService* pOutGatttService, const GattAttributeUuid& uuid, uint32_t connectionHandle) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pOutGatttService);

        nn::btm::user::GattService btmService;

        bool isServiceExist = nn::btm::user::GetGattService(&btmService, connectionHandle, uuid);

        if(isServiceExist)
        {
            *pOutGatttService = GattService(uuid, btmService.handle, connectionHandle, btmService.instanceId, btmService.endGroupHandle, btmService.isPrimaryService);

            return true;
        }

        return isServiceExist;
    }

    void AcquireBlePairingEvent(nn::os::SystemEventType* pOutSystemEvent) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pOutSystemEvent);

        btm::user::AcquireBlePairingEvent(pOutSystemEvent);
    }

    nn::Result PairGattServer(uint32_t connectionHandle, const BleAdvertisePacketParameter& scanParameter) NN_NOEXCEPT
    {
        nn::btm::user::BleAdvFilterForGeneral filter;

        memcpy(filter.manufacturerId, scanParameter.manufacturerId, BleNnAdvertiseManufacturerIdSize);
        memcpy(filter.clientId, scanParameter.clientId, BleNnAdvertiseManufactureClientIdSize);
        memcpy(filter.serverId, scanParameter.serverId, BleNnAdvertiseManufactureServerIdSize);

        auto result = nn::btm::user::BlePairDevice(connectionHandle, filter);

        return ConvertResult(result);
    }

    nn::Result UnpairGattServer(uint32_t connectionHandle, const BleAdvertisePacketParameter& scanParameter) NN_NOEXCEPT
    {
        nn::btm::user::BleAdvFilterForGeneral filter;

        memcpy(filter.manufacturerId, scanParameter.manufacturerId, BleNnAdvertiseManufacturerIdSize);
        memcpy(filter.clientId, scanParameter.clientId, BleNnAdvertiseManufactureClientIdSize);
        memcpy(filter.serverId, scanParameter.serverId, BleNnAdvertiseManufactureServerIdSize);

        auto result = nn::btm::user::BleUnPairDevice(connectionHandle, filter);

        return ConvertResult(result);
    }

    nn::Result UnpairGattServer(const Address& bluetoothAddress, const BleAdvertisePacketParameter& scanParameter) NN_NOEXCEPT
    {
        nn::btm::user::BleAdvFilterForGeneral filter;

        memcpy(filter.manufacturerId, scanParameter.manufacturerId, BleNnAdvertiseManufacturerIdSize);
        memcpy(filter.clientId, scanParameter.clientId, BleNnAdvertiseManufactureClientIdSize);
        memcpy(filter.serverId, scanParameter.serverId, BleNnAdvertiseManufactureServerIdSize);

        auto result = nn::btm::user::BleUnPairDevice(bluetoothAddress, filter);

        return ConvertResult(result);
    }

    int GetPairedGattServerAddress(Address* pOutAddresses, int count, const BleAdvertisePacketParameter& scanParameter) NN_NOEXCEPT
    {
        nn::btm::user::BleAdvFilterForGeneral filter;

        memcpy(filter.manufacturerId, scanParameter.manufacturerId, BleNnAdvertiseManufacturerIdSize);
        memcpy(filter.clientId, scanParameter.clientId, BleNnAdvertiseManufactureClientIdSize);
        memcpy(filter.serverId, scanParameter.serverId, BleNnAdvertiseManufactureServerIdSize);

        return nn::btm::user::BleGetPairedDevices(pOutAddresses, static_cast<uint8_t>(count), filter);
    }

    void AcquireBleMtuConfigEvent(nn::os::SystemEventType* pOutSystemEvent) NN_NOEXCEPT
    {
        btm::user::AcquireBleMtuConfigEvent(pOutSystemEvent);
    }

    nn::Result ConfigureBleMtu(uint32_t connectionHandle, uint16_t mtu)
    {
        NN_ABORT_UNLESS_RANGE(mtu, BleMtuDefault, BleMtuMax + 1);

        auto result = btm::user::ConfigureBleMtu(connectionHandle, mtu);

        return ConvertResult(result);
    }

    uint16_t GetBleMtu(uint32_t connectionHandle) NN_NOEXCEPT
    {
        return btm::user::GetBleMtu(connectionHandle);
    }

    void AcquireBleGattOperationEvent(nn::os::SystemEventType* pOutSystemEvent) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pOutSystemEvent);

        auto result = user::RegisterBleEvent(pOutSystemEvent);

        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        NN_UNUSED(result);
    }

    nn::Result RegisterGattOperationNotification(const GattAttributeUuid& uuid) NN_NOEXCEPT
    {
        nn::btm::user::BleDataPath path;
        path.uuid = uuid;
        path.path = nn::btm::user::BleDataPathType::BLE_DATA_PATH_GENERAL;

        auto result = nn::btm::user::RegisterBleGattDataPath(path);

        return ConvertResult(result);
    }

    nn::Result UnregisterGattOperationNotification(const GattAttributeUuid& uuid) NN_NOEXCEPT
    {
        nn::btm::user::BleDataPath path;
        path.uuid = uuid;
        path.path = nn::btm::user::BleDataPathType::BLE_DATA_PATH_GENERAL;

        auto result = nn::btm::user::UnregisterBleGattDataPath(path);

        return ConvertResult(result);
    }

    void GetGattOperationResult(BleClientGattOperationInfo* pOutResult) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pOutResult);

        BleEventType btEventType;
        uint8_t btEventBuffer[BUFFER_SIZE_OF_BLE_OUT];

        user::GetLeEventInfo(&btEventType, btEventBuffer, NN_ARRAY_SIZE(btEventBuffer));

        switch (btEventType)
        {
        case EventFromLeClientGattOpCallback:
        {
            InfoFromLeGattOperationCallback* pInfo = reinterpret_cast<InfoFromLeGattOperationCallback*>(btEventBuffer);

            pOutResult->status              = ConvertBluetoothLeStatus(pInfo->status);
            pOutResult->connectionHandle    = pInfo->connId;
            pOutResult->operation           = pInfo->operation;
            pOutResult->serviceUuid         = pInfo->serviceUuid;
            pOutResult->charcteristicUuid   = pInfo->charcteristicUuid;
            pOutResult->descriptorUuid      = pInfo->descriptorUuid;
            pOutResult->length              = static_cast<size_t>(pInfo->length);
            memcpy(pOutResult->data, pInfo->value, pInfo->length);
        }
        break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    nn::Result ReadGattCharacteristic(const GattCharacteristic& characteristic) NN_NOEXCEPT
    {
        GattService service;
        GattId      serviceId;
        GattId      characteristicId;
        uint16_t    connectionHandle;
        bool        isPrimaryService;

        if (!(characteristic.GetProperties() & GattAttributeProperty_Read))
        {
            return ResultUnsupportedGattProperty();
        }

        connectionHandle = characteristic.GetConnectionHandle();

        characteristic.GetService(&service);

        serviceId.instanceId    = service.GetInstanceId();
        serviceId.uuid          = service.GetUuid();
        isPrimaryService        = service.IsPrimaryService();

        characteristicId.instanceId = characteristic.GetInstanceId();
        characteristicId.uuid       = characteristic.GetUuid();

        auto result = user::LeClientReadCharacteristic(connectionHandle,
                                                       serviceId, isPrimaryService,
                                                       characteristicId,
                                                       0);

        return ConvertResult(result);

    }

    nn::Result WriteGattCharacteristic(const GattCharacteristic& characteristic) NN_NOEXCEPT
    {
        GattService service;
        GattId      serviceId;
        GattId      characteristicId;
        uint16_t    connectionHandle;
        size_t      valueLength = 0;
        uint8_t     value[GattAttributeValueSizeMax];
        bool        isPrimaryService = true;
        bool        withResponse = true;

        if (!(characteristic.GetProperties() & GattAttributeProperty_Write) && !(characteristic.GetProperties() & GattAttributeProperty_WriteWithoutResponse))
        {
            return ResultUnsupportedGattProperty();
        }

        connectionHandle = characteristic.GetConnectionHandle();

        valueLength = characteristic.GetValue(value, NN_ARRAY_SIZE(value));
        NN_ABORT_UNLESS_RANGE(valueLength, 1, GattAttributeValueSizeMax + 1);

        characteristic.GetService(&service);

        serviceId.instanceId    = service.GetInstanceId();
        serviceId.uuid          = service.GetUuid();
        isPrimaryService        = service.IsPrimaryService();

        characteristicId.instanceId = characteristic.GetInstanceId();
        characteristicId.uuid       = characteristic.GetUuid();
        withResponse = ((characteristic.GetProperties() & GattAttributeProperty_Write) != 0);

        auto result = user::LeClientWriteCharacteristic(connectionHandle,
                                                        serviceId, isPrimaryService,
                                                        characteristicId,
                                                        value, static_cast<uint16_t>(valueLength), 0, withResponse);

        return ConvertResult(result);
    }

    nn::Result EnableGattCharacteristicNotification(const GattCharacteristic& characteristic, bool enable) NN_NOEXCEPT
    {
        nn::Result      result = nn::ResultSuccess();
        GattService     service;
        GattId          serviceId;
        GattId          characteristicId;
        uint16_t        connectionHandle;
        bool            isPrimaryService;

        if (!(characteristic.GetProperties() & GattAttributeProperty_Notify) &&
            !(characteristic.GetProperties() & GattAttributeProperty_Indicate))
        {
            return ResultUnsupportedGattProperty();
        }

        connectionHandle = characteristic.GetConnectionHandle();

        characteristic.GetService(&service);

        serviceId.instanceId    = service.GetInstanceId();
        serviceId.uuid          = service.GetUuid();
        isPrimaryService        = service.IsPrimaryService();

        characteristicId.instanceId = characteristic.GetInstanceId();
        characteristicId.uuid       = characteristic.GetUuid();

        if (enable)
        {
            result = user::LeClientRegisterNotification(connectionHandle, serviceId, isPrimaryService, characteristicId);
        }
        else
        {
            result = user::LeClientDeregisterNotification(connectionHandle, serviceId, isPrimaryService, characteristicId);
        }

        return ConvertResult(result);
    }

    nn::Result ReadGattDescriptor(const GattDescriptor& descriptor) NN_NOEXCEPT
    {
        GattService         service;
        GattCharacteristic  characteristic;
        GattId              serviceId;
        GattId              characteristicId;
        GattId              descriptorId;
        uint16_t            connectionHandle;
        bool                isPrimaryService = true;

        connectionHandle = descriptor.GetConnectionHandle();

        descriptor.GetService(&service);
        descriptor.GetCharacteristic(&characteristic);

        serviceId.uuid          = service.GetUuid();
        serviceId.instanceId    = service.GetInstanceId();

        characteristicId.uuid       = characteristic.GetUuid();
        characteristicId.instanceId = characteristic.GetInstanceId();

        descriptorId.uuid       = descriptor.GetUuid();
        descriptorId.instanceId = 0x00;     // Descriptor は常に0x00

        isPrimaryService = service.IsPrimaryService();

        auto result = user::LeClientReadDescriptor(connectionHandle,
                                                   serviceId, isPrimaryService,
                                                   characteristicId,
                                                   descriptorId,
                                                   0);

        return ConvertResult(result);
    }

    nn::Result WriteGattDescriptor(const GattDescriptor& descriptor) NN_NOEXCEPT
    {
        GattService         service;
        GattCharacteristic  characteristic;
        GattId              serviceId;
        GattId              characteristicId;
        GattId              descriptorId;
        uint16_t            connectionHandle;
        size_t              valueLength = 0;
        uint8_t             value[GattAttributeValueSizeMax];
        bool isPrimaryService = true;

        connectionHandle = descriptor.GetConnectionHandle();

        valueLength = descriptor.GetValue(value, NN_ARRAY_SIZE(value));
        NN_ABORT_UNLESS_RANGE(valueLength, 1, GattAttributeValueSizeMax + 1);

        descriptor.GetService(&service);
        descriptor.GetCharacteristic(&characteristic);

        serviceId.uuid          = service.GetUuid();
        serviceId.instanceId    = service.GetInstanceId();

        characteristicId.uuid       = characteristic.GetUuid();
        characteristicId.instanceId = characteristic.GetInstanceId();

        descriptorId.uuid       = descriptor.GetUuid();
        descriptorId.instanceId = 0x00;     // Descriptor は常に0x00

        isPrimaryService = service.IsPrimaryService();

        auto result = user::LeClientWriteDescriptor(connectionHandle,
                                                    serviceId, isPrimaryService,
                                                    characteristicId,
                                                    descriptorId,
                                                    value, static_cast<uint16_t>(valueLength), 0);

        return ConvertResult(result);
    }
}}  // namespace nn::bluetooth
