﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/btm/user/btm_UserApi.h>

#include <nn/bluetooth/bluetooth_GattService.h>
#include <nn/bluetooth/bluetooth_GattCharacteristic.h>
#include <nn/bluetooth/bluetooth_GattDescriptor.h>

namespace nn { namespace bluetooth {

    GattService::GattService()  NN_NOEXCEPT : GattAttribute()
    {
        m_Type = GattAttributeType_Service;

        m_InstanceId = 0x0000;

        m_EndGroupHandle = GattAttributeInvalidHandle;

        m_IsPrimary = true;
    }

    GattService::GattService(const GattAttributeUuid& uuid, uint16_t handle, uint32_t connectionHandle,
        uint16_t instanceId, uint16_t endGroupHandle, bool isPrimary)  NN_NOEXCEPT :
        GattAttribute(uuid, handle, connectionHandle)
    {
        m_Type = GattAttributeType_Service;

        m_InstanceId = instanceId;

        m_EndGroupHandle = endGroupHandle;

        m_IsPrimary = isPrimary;
    }

    GattService::~GattService() NN_NOEXCEPT
    {
        // 何もしない
    }

    uint16_t GattService::GetInstanceId() const NN_NOEXCEPT
    {
        return m_InstanceId;
    }

    uint16_t GattService::GetEndGroupHandle() const NN_NOEXCEPT
    {
        return m_EndGroupHandle;
    }

    bool GattService::IsPrimaryService() const NN_NOEXCEPT
    {
        return m_IsPrimary;
    }

    int GattService::GetIncludedServices(GattService* pOutServices, int count) const  NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pOutServices);

        nn::btm::user::GattService btmService[GattAttributeCountMaxClient];

        uint8_t inNum = static_cast<uint8_t>(count);
        uint8_t includedServiceNum = nn::btm::user::GetGattIncludedServices(btmService, NN_ARRAY_SIZE(btmService), m_ConnectionHandle, m_Handle);

        includedServiceNum = (inNum < includedServiceNum) ? inNum : includedServiceNum;

        for (uint8_t i = 0; i < includedServiceNum; ++i)
        {
            pOutServices[i] = GattService(btmService[i].uuid, btmService[i].handle, m_ConnectionHandle, btmService[i].instanceId, btmService[i].endGroupHandle, btmService[i].isPrimaryService);
        }

        return includedServiceNum;
    }

    int GattService::GetCharacteristics(GattCharacteristic* pOutCharacteristics, int count) const NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pOutCharacteristics);

        nn::btm::user::GattCharacteristic btmCharacteristic[GattAttributeCountMaxClient];

        uint8_t inNum = static_cast<uint8_t>(count);
        uint8_t characteristicNum = nn::btm::user::GetGattCharacteristics(btmCharacteristic, NN_ARRAY_SIZE(btmCharacteristic), m_ConnectionHandle, m_Handle);

        characteristicNum = (inNum < characteristicNum) ? inNum : characteristicNum;

        for (uint8_t i = 0; i < characteristicNum; ++i)
        {
            pOutCharacteristics[i] = GattCharacteristic(btmCharacteristic[i].uuid, btmCharacteristic[i].handle, m_ConnectionHandle,
                                                        btmCharacteristic[i].instanceId, btmCharacteristic[i].property);
        }

        return characteristicNum;
    }

    bool GattService::GetCharacteristic(GattCharacteristic* pOutCharacteristics, const GattAttributeUuid& uuid) const NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pOutCharacteristics);

        nn::btm::user::GattCharacteristic btmCharacteristic[GattAttributeCountMaxClient];

        uint8_t characteristicNum = nn::btm::user::GetGattCharacteristics(btmCharacteristic, NN_ARRAY_SIZE(btmCharacteristic), m_ConnectionHandle, m_Handle);

        for (uint8_t i = 0; i < characteristicNum; ++i)
        {
            if (btmCharacteristic[i].uuid == uuid)
            {
                *pOutCharacteristics = GattCharacteristic(btmCharacteristic[i].uuid, btmCharacteristic[i].handle, m_ConnectionHandle,
                                                          btmCharacteristic[i].instanceId, btmCharacteristic[i].property);

                return true;
            }
        }

        return false;
    }
}}  // namespace nn::bluetooth
