﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os.h>
#include <nn/bpc/bpc_Rtc.h>
#include <nn/util/util_IntrusiveList.h>
#include <mutex>
#include <algorithm>

#include "bgtc_Log.h"
#include "bgtc_ClientBroker.h"
#include "bgtc_RealTimeClock.h"
#include "bgtc_Instance.h"

namespace nn{ namespace bgtc{

ClientBroker::ClientBroker()
    : m_Mutex(true)
    , m_TimerEvent(os::EventClearMode_AutoClear)
    , m_bDispatchable(true)
{

}

ClientBroker::~ClientBroker()
{

}

void ClientBroker::Register(Entry & entry)
{
    std::lock_guard<os::Mutex> lock(m_Mutex);

    m_listEntry.push_back(entry);
}

void ClientBroker::Unregister(Entry & entry)
{
    std::lock_guard<os::Mutex> lock(m_Mutex);

    m_listEntry.erase(m_listEntry.iterator_to(entry));
}

void ClientBroker::ScheduleTask(Entry & entry, Interval interval)
{
    std::lock_guard<os::Mutex> lock(m_Mutex);
    entry.ScheduleTask(interval);
    NN_BGTC_INFO("Scheduled a new task(% 8s):% 9d(now) +% 6d(interval) =% 9d(scheduled) sec\n",
        entry.GetName(), GetNow(), interval, entry.GetScheduledTime());
    OnScheduleChanged();
}

void ClientBroker::SchedulePeriodicTask(Entry & entry, Interval intervalFirst, Interval intervalPeriodic)
{
    std::lock_guard<os::Mutex> lock(m_Mutex);
    entry.SchedulePeriodicTask(intervalFirst, intervalPeriodic);
    NN_BGTC_INFO("Scheduled a new task(% 8s):% 9d(now) +%6d[~% 6d](interval) =% 9d(scheduled) sec\n",
        entry.GetName(), GetNow(), intervalFirst, intervalPeriodic, entry.GetScheduledTime());
    OnScheduleChanged();
}

void ClientBroker::UnscheduleTask(Entry & entry)
{
    std::lock_guard<os::Mutex> lock(m_Mutex);
    entry.UnscheduleTask();
    NN_BGTC_INFO("Unscheduled a task(% 8s):% 9d(now), % 9d(scheduled), ~% 6d(interval) = sec\n",
        entry.GetName(), GetNow(), entry.GetScheduledTime(), entry.GetScheduledTaskInterval());
    OnScheduleChanged();
}

void ClientBroker::SetTaskProcessing(Entry & entry, bool bTaskProcessing)
{
    std::lock_guard<os::Mutex> lock(m_Mutex);
    NN_BGTC_INFO("task(% 8s): processing=% 3s <-% 3s\n",
        entry.GetName(), bTaskProcessing ? "yes" : "no", entry.IsTaskProcessing() ? "yes" : "no");
    entry.SetTaskProcessing(bTaskProcessing);
}

Interval ClientBroker::GetScheduledTaskInterval(const Entry & entry) const
{
    return entry.GetScheduledTaskInterval();
}

bool ClientBroker::IsTaskScheduled(const Entry & entry) const
{
    return entry.IsTaskScheduled();
}

void ClientBroker::SetClientName(Entry & entry, const char * pName, size_t length)
{
    std::lock_guard<os::Mutex> lock(m_Mutex);

    entry.SetName(pName, length);
}

bool ClientBroker::GetNextScheduledTime(Time* pOut) const
{
    std::lock_guard<os::Mutex> lock(m_Mutex);

    Time timeNearestAlarm = INT64_MAX;

    for (auto& entry : m_listEntry)
    {
        if (entry.m_bAlarmActive && entry.m_TimeAlarm < timeNearestAlarm)
        {
            timeNearestAlarm = entry.m_TimeAlarm;
        }
    }

    if (timeNearestAlarm != INT64_MAX)
    {
        *pOut = timeNearestAlarm;
        return true;
    }
    else
    {
        return false;
    }
}

bool ClientBroker::GetNextScheduledTimeAsInterval(Interval* pOut) const
{
    std::lock_guard<os::Mutex> lock(m_Mutex);

    Time scheduled;
    if (!GetNextScheduledTime(&scheduled))
    {
        return false;
    }

    Interval interval = static_cast<Interval>(scheduled - GetNow());
    *pOut = std::max<Interval>(interval, 0);
    return true;
}

bool ClientBroker::IsProcessingClientExists() const
{
    std::lock_guard<os::Mutex> lock(m_Mutex);

    for (auto& entry : m_listEntry)
    {
        if (entry.IsTaskProcessing())
        {
            return true;
        }
    }
    return false;
}

void ClientBroker::ResetTaskProcessingState()
{
    std::lock_guard<os::Mutex> lock(m_Mutex);

    for (auto& entry : m_listEntry)
    {
        entry.SetTaskProcessing(false);
    }
}

void ClientBroker::DispatchScheduleEvent()
{
    std::lock_guard<os::Mutex> lock(m_Mutex);

    SignalScheduleEventIfElapsed();
    m_TimerEvent.Stop();
    Interval interval;
    if (GetNextScheduledTimeAsInterval(&interval))
    {
        m_TimerEvent.StartOneShot(TimeSpan::FromSeconds(interval + 1));
    }
}

void ClientBroker::SetEnableScheduleEvent(bool bDispatchable)
{
    if (!m_bDispatchable && bDispatchable)
    {
        DispatchScheduleEvent();
    }
    m_bDispatchable = bDispatchable;
}

void ClientBroker::ShowSchedule() const
{
    std::lock_guard<os::Mutex> lock(m_Mutex);

    NN_BGTC_TRACE("=== Schedule list ===\n");
    for (auto& entry : m_listEntry)
    {
        NN_BGTC_TRACE("task(% 8s): processing=% 3s, scheduled time=% 9lld(% 6d), periodic=% 3s\n",
            entry.GetName(),
            entry.IsTaskProcessing() ? "yes" : "no",
            entry.GetScheduledTime(), entry.GetScheduledTaskInterval(),
            entry.IsPeriodic() ? "yes" : "no");
        NN_UNUSED(entry);
    }
}

void ClientBroker::SignalScheduleEventIfElapsed()
{
    std::lock_guard<os::Mutex> lock(m_Mutex);

    Time timeNow = GetNow();
    for (auto& entry : m_listEntry)
    {
        entry.SignalScheduleEventIfElapsed(timeNow);
    }
}

void ClientBroker::OnScheduleChanged()
{
    if (m_bDispatchable)
    {
        DispatchScheduleEvent();
    }
}

ClientBroker::Entry::Entry(os::SystemEvent& eventTrigger, os::SystemEvent& eventAlarm)
    : m_TimeAlarm(0), m_PeriodicInterval(0), m_EventTrigger(eventTrigger), m_EventAlarm(eventAlarm)
    , m_bTaskProcessing(false), m_bAlarmActive(false), m_bPeriodic(false)
{
    std::memset(m_ClientName, 0, sizeof(m_ClientName));
}

void ClientBroker::Entry::ScheduleTask(Interval interval)
{
    m_EventAlarm.Clear();
    m_TimeAlarm = GetNow() + interval;
    m_PeriodicInterval = 0;
    m_bAlarmActive = true;
    m_bPeriodic = false;
}

void ClientBroker::Entry::SchedulePeriodicTask(Interval intervalFirst, Interval intervalPeriodic)
{
    m_EventAlarm.Clear();
    m_TimeAlarm = GetNow() + intervalFirst;
    m_PeriodicInterval = intervalPeriodic;
    m_bAlarmActive = true;
    m_bPeriodic = true;
}

void ClientBroker::Entry::UnscheduleTask()
{
    m_TimeAlarm = 0;
    m_PeriodicInterval = 0;
    m_EventAlarm.Clear();
    m_bAlarmActive = false;
    m_bPeriodic = false;
}

Interval ClientBroker::Entry::GetScheduledTaskInterval() const
{
    return std::max<Interval>(static_cast<Interval>(m_TimeAlarm - GetNow()), 0);
}

void ClientBroker::Entry::SignalScheduleEventIfElapsed(Time timeBase)
{
    if (m_bAlarmActive && m_TimeAlarm <= timeBase)
    {
        NN_BGTC_INFO("Notifiy to task(% 8s):% 9d(scheduled) <=% 9d(now)\n", m_ClientName, GetScheduledTime(), timeBase);
        if (m_bPeriodic)
        {
            m_TimeAlarm = GetNow() + m_PeriodicInterval;
        }
        else
        {
            m_bAlarmActive = false;
        }
        m_EventAlarm.Signal();
    }
}

void ClientBroker::Entry::SetTaskProcessing(bool bTaskProcessing)
{
    m_bTaskProcessing = bTaskProcessing;
}

void ClientBroker::Entry::SetName(const char * pName, size_t length)
{
    size_t copyLen = std::min<size_t>(sizeof(m_ClientName) - 1, length);
    std::strncpy(m_ClientName, pName, copyLen);
    m_ClientName[copyLen] = '\0';
}

}}


