﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/bgtc/detail/bgtc_TaskImpl.h>
#include <nn/os.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/nn_SdkLog.h>

#include "bgtc_Config.h"
#include "bgtc_ServiceProxy.h"

#if !NN_BGTC_ENABLE_HIPC
#include "bgtc_Instance.h"
#include "bgtc_TaskServiceImpl.h"
#endif

#include "bgtc_Log.h"

namespace nn { namespace bgtc { namespace detail {

    TaskImpl::TaskImpl() NN_NOEXCEPT
        : m_pTaskService(nullptr)
    {
    }

    TaskImpl::~TaskImpl() NN_NOEXCEPT
    {
        if (IsInitialized())
        {
            Finalize();
        }
    }

    Result TaskImpl::Initialize() NN_NOEXCEPT
    {
        Result result;

#if NN_BGTC_ENABLE_HIPC
        const char* pPortName = PortNames[PortIndex_TaskService];

        result = sf::CreateHipcProxyByName<ITaskService, Allocator::Policy>(&m_pTaskService, pPortName);

        if (result.IsFailure())
        {
            return result;
        }
#else
        typedef nn::sf::ObjectFactory<Allocator::Policy> Factory;

        NN_BGTC_WARN("Windows build is not supported. It may not work correctly.\n");
        m_pTaskService = Factory::CreateSharedEmplaced<ITaskService, TaskServiceImpl>();

        if (!m_pTaskService)
        {
            return ResultUnexpected();
        }
#endif

#if !NN_BGTC_ENABLE_HIPC
        g_Instance.Initialize();
#endif

        result = AttachEventsForTaskService();
        if (result.IsFailure())
        {
            m_pTaskService = nullptr;
#if !NN_BGTC_ENABLE_HIPC
            g_Instance.Finalize();
#endif
            return result;
        }

        SetCurrentModuleName();
        return ResultSuccess();
    }

    void TaskImpl::Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pTaskService);
        FinalizeEventsForTaskService();
        m_pTaskService = nullptr;
#if !NN_BGTC_ENABLE_HIPC
        g_Instance.Finalize();
#endif
    }

    bool TaskImpl::IsInitialized() NN_NOEXCEPT
    {
        return m_pTaskService != nullptr;
    }


    Result TaskImpl::NotifyStarting() NN_NOEXCEPT
    {
        return GetTaskService().NotifyTaskStarting();
    }

    void TaskImpl::NotifyFinished() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            GetTaskService().NotifyTaskFinished()
        );
    }

    nn::os::SystemEvent & TaskImpl::GetTriggerEvent() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_pTaskService != nullptr);
        return m_SystemEventForTrigger;
    }

    bool TaskImpl::IsInHalfAwake() NN_NOEXCEPT
    {
        bool bResult;
        sf::Out<bool> out(&bResult);
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            GetTaskService().IsInHalfAwake(out)
        );
        return bResult;
    }

    bool TaskImpl::IsInFullAwake()
    {
        bool bResult;
        sf::Out<bool> out(&bResult);
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            GetTaskService().IsInFullAwake(out)
        );
        return bResult;
    }

    void TaskImpl::Schedule(Interval intervalSeconds) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            GetTaskService().ScheduleTask(intervalSeconds)
        );
    }

    void TaskImpl::ScheduleUnsafe(Interval intervalSeconds) NN_NOEXCEPT
    {
        (void)GetTaskService().ScheduleTask(intervalSeconds);
    }

    void TaskImpl::SchedulePeriodic(Interval intervalSecondsFirst, Interval intervalSecondsPeriodic) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            GetTaskService().SchedulePeriodicTask(intervalSecondsFirst, intervalSecondsPeriodic)
        );
    }

    Result TaskImpl::GetScheduledInterval(Interval * pOutSeconds) NN_NOEXCEPT
    {
        sf::Out<int32_t> out(pOutSeconds);
        return GetTaskService().GetScheduledTaskInterval(out);
    }

    Result TaskImpl::Unschedule() NN_NOEXCEPT
    {
        return GetTaskService().UnscheduleTask();
    }

    nn::os::SystemEvent & TaskImpl::GetScheduleEvent() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_pTaskService != nullptr);
        return m_SystemEventForSchedule;
    }

    OperationMode TaskImpl::GetOperationMode() NN_NOEXCEPT
    {
        OperationMode operationMode;
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            GetTaskService().GetOperationMode(&operationMode)
        );
        return operationMode;
    }

    bool TaskImpl::WillDisconnectNetworkWhenEnteringSleep() NN_NOEXCEPT
    {
        bool bResult;
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            GetTaskService().WillDisconnectNetworkWhenEnteringSleep(&bResult)
        );
        return bResult;
    }

    bool TaskImpl::WillStayHalfAwakeInsteadSleep() NN_NOEXCEPT
    {
        bool bResult;
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            GetTaskService().WillStayHalfAwakeInsteadSleep(&bResult)
        );
        return bResult;
    }

    void TaskImpl::SetClientName(const char* pName) NN_NOEXCEPT
    {
        nn::sf::InArray<char> name(pName, std::strlen(pName));
        (void)GetTaskService().NotifyClientName(name);
    }

    Result TaskImpl::AttachEventsForTaskService()
    {
        Result result;
        result = AttachEventByService<ITaskService, &ITaskService::GetTriggerEvent>(
            GetTaskService(),
            m_SystemEventForTrigger, os::EventClearMode_AutoClear
            );
        if (result.IsFailure())
        {
            return result;
        }

        result = AttachEventByService<ITaskService, &ITaskService::GetScheduleEvent>(
            GetTaskService(),
            m_SystemEventForSchedule, os::EventClearMode_AutoClear
            );
        if (result.IsFailure())
        {
            return result;
        }
        return ResultSuccess();
    }

    void TaskImpl::FinalizeEventsForTaskService()
    {
        m_SystemEventForSchedule.~SystemEvent();
        m_SystemEventForTrigger.~SystemEvent();
    }

    void TaskImpl::SetCurrentModuleName()
    {
        const char* pThreadName = nullptr;
        if (nn::os::GetCurrentThread())
        {
            pThreadName = nn::os::GetThreadNamePointer(nn::os::GetCurrentThread());
        }
        if (!pThreadName)
        {
            return;
        }

        const char* pModuleName = std::strchr(pThreadName, '.');
        if (!pModuleName)
        {
            return;
        }
        pModuleName += 1;

        const char* pModuleNameEnd = std::strchr(pModuleName, '.');
        if (!pModuleNameEnd)
        {
            return;
        }

        nn::sf::InArray<char> name(pModuleName, static_cast<size_t>(pModuleNameEnd - pModuleName));
        (void)GetTaskService().NotifyClientName(name);
    }

    ITaskService& TaskImpl::GetTaskService() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_pTaskService != nullptr, "bgtc::Initialize() or bgtc::Task::Initialize() should be called before using this API.\n");
        return *m_pTaskService.Get();
    }
}}}
