﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/psc.h>

#include <nn/bgsu/bgsu_Log.h>
#include <nn/bgsu/bgsu_PowerStateEventHandler.h>

namespace nn{ namespace bgsu{

PowerStateEventHandler::PowerStateEventHandler()
    : m_pEvent(nullptr)
    , m_bFromFullAwake(true)
{
}

PowerStateEventHandler::~PowerStateEventHandler()
{
}

Result PowerStateEventHandler::Initialize(psc::PmModuleId moduleId, const psc::PmModuleId* pDependencyList, uint32_t dependencyCount)
{
    Result result = m_PmModule.Initialize(
        moduleId,
        pDependencyList, dependencyCount,
        os::EventClearMode_AutoClear
    );
    if (result.IsFailure())
    {
        NN_BGSU_ERROR("psc::PmModule::Initialize failed.(0x%08x)\n", result.GetInnerValueForDebug());
        return result;
    }

    m_pEvent = m_PmModule.GetEventPointer();

    NN_SDK_ASSERT_NOT_NULL(m_pEvent);

    m_LastPmState = psc::PmState_FullAwake;
    m_bFromFullAwake = true;

    return ResultSuccess();
}

void PowerStateEventHandler::Dispatch()
{
    psc::PmState pmState;
    psc::PmFlagSet pmFlagSet;
    Result result = m_PmModule.GetRequest(&pmState, &pmFlagSet);
    if (result.IsSuccess())
    {
        NN_BGSU_TRACE("Received a request from PmModule: PmState = %d\n", pmState);
        psc::PmState pmStatePrevious = m_LastPmState;
        m_LastPmState = pmState;
        OnReceivePowerStateEvent(pmStatePrevious, pmState, pmFlagSet);

        result = m_PmModule.Acknowledge(pmState, ResultSuccess());
        if (result.IsFailure())
        {
            NN_BGSU_WARN("psc::PmModule::Acknowledge failed.(0x%08x)\n", result.GetInnerValueForDebug());
        }

        if (pmState == psc::PmState_FullAwake)
        {
            m_bFromFullAwake = true;
        }
        else if (pmState == psc::PmState_SleepReady)
        {
            m_bFromFullAwake = false;
        }
    }
    else
    {
        NN_BGSU_WARN("psc::PmModule::GetRequest failed.(0x%08x)\n", result.GetInnerValueForDebug());
    }
}

os::SystemEvent& PowerStateEventHandler::GetEvent()
{
    NN_SDK_REQUIRES_NOT_NULL(m_pEvent);
    return *m_pEvent;
}

void PowerStateEventHandler::Finalize()
{
    m_pEvent = nullptr;
    m_PmModule.Finalize();
}

}
}
