﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/service/bcat_ApiService.h>
#include <nn/bcat/service/bcat_ArchiveDecoder.h>
#include <nn/bcat/service/bcat_ArchiveDecryptionKeyGenerator.h>
#include <nn/bcat/service/bcat_ArchiveVerifier.h>
#include <nn/bcat/bcat_Result.h>
#include <nn/bcat/bcat_ResultPrivate.h>
#include <nn/nn_SdkAssert.h>
#include <nn/crypto/detail/crypto_Clear.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace bcat { namespace service {

nn::Result DecodeArchiveFile(size_t* outSize, void* decoded, size_t decodedSize,
    const void* encoded, size_t encodedSize, nn::ApplicationId appId, const char* passphrase) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outSize);
    NN_SDK_REQUIRES_NOT_NULL(decoded);
    NN_SDK_REQUIRES_GREATER(decodedSize, 0u);
    NN_SDK_REQUIRES_NOT_NULL(encoded);
    NN_SDK_REQUIRES_GREATER(encodedSize, 0u);
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());
    NN_SDK_REQUIRES_NOT_NULL(passphrase);

    ArchiveHeader header = {};

    NN_RESULT_THROW_UNLESS(sizeof (header) < encodedSize, ResultDataVerificationFailed());

    std::memcpy(&header, encoded, sizeof (header));

    NN_RESULT_THROW_UNLESS(header.magic[0] == 'b' && header.magic[1] == 'c' && header.magic[2] == 'a' && header.magic[3] == 't',
        ResultDataVerificationFailed());

    NN_RESULT_THROW_UNLESS(header.version == 1, ResultUnsupportedFormatDetected());

    NN_RESULT_THROW_UNLESS(header.encryptionType == nn::bcat::service::EncryptionType_NoEncrypted ||
        header.encryptionType == nn::bcat::service::EncryptionType_AesCtr128 ||
        header.encryptionType == nn::bcat::service::EncryptionType_AesCtr192 ||
        header.encryptionType == nn::bcat::service::EncryptionType_AesCtr256, ResultUnsupportedFormatDetected());

    NN_RESULT_THROW_UNLESS(header.signatureType == nn::bcat::service::SignatureType_RsaPkcs1Sha1 ||
        header.signatureType == nn::bcat::service::SignatureType_RsaPkcs1Sha256 ||
        header.signatureType == nn::bcat::service::SignatureType_RsaPssSha1 ||
        header.signatureType == nn::bcat::service::SignatureType_RsaPssSha256, ResultUnsupportedFormatDetected());

    NN_RESULT_THROW_UNLESS(header.saltIndex >= nn::bcat::service::ArchiveDecryptionKeyGenerator::SaltIndexMin &&
        header.saltIndex <= nn::bcat::service::ArchiveDecryptionKeyGenerator::SaltIndexMax, ResultUnsupportedFormatDetected());

    size_t actualDecodedSize = encodedSize - sizeof (header);

    NN_RESULT_THROW_UNLESS(actualDecodedSize <= decodedSize, ResultDataVerificationFailed());

    nn::bcat::service::ArchiveDecoder decoder;

    decoder.Initialize(header, passphrase, appId);
    decoder.Decode(decoded, reinterpret_cast<const Bit8*>(encoded) + sizeof (header), actualDecodedSize);

    nn::bcat::service::ArchiveVerifier verifier;

    verifier.Initialize(header.signatureType);
    verifier.Update(decoded, actualDecodedSize);

    if (!verifier.Verify(header.signature, sizeof (header.signature)))
    {
        // 鍵の推測を防ぐため、クリアしておく。
        nn::crypto::detail::ClearMemory(decoded, decodedSize);

        NN_RESULT_THROW(ResultDataVerificationFailed());
    }

    *outSize = actualDecodedSize;

    NN_RESULT_SUCCESS;
}

}}}
