﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/detail/service/core/bcat_TaskManager.h>
#include <nn/npns.h>

namespace nn { namespace bcat { namespace detail { namespace service { namespace core {

namespace
{
    const char* FilePath = "bcat-sys:/dc/task.bin";
}

TaskManager::TaskManager() NN_NOEXCEPT :
    m_Mutex(true),
    m_ScheduleEvent(nn::os::EventClearMode_ManualClear),
    m_SubscriptionEvent(nn::os::EventClearMode_ManualClear),
    m_RecordCount(0),
    m_RecordForUnsubscriptionCount(0),
    m_IsDeleteRequired(false)
{
    std::memset(m_Records, 0, sizeof (m_Records));
    std::memset(m_RecordsForUnsubscription, 0, sizeof (m_RecordsForUnsubscription));

    std::memset(m_VolatileRecords, 0, sizeof (m_VolatileRecords));
}

nn::Result TaskManager::Load() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_TRY(LoadImpl())
        NN_RESULT_CATCH_ALL
        {
            std::memset(m_Records, 0, sizeof (m_Records));
            std::memset(m_RecordsForUnsubscription, 0, sizeof (m_RecordsForUnsubscription));
            m_RecordCount = 0;
            m_RecordForUnsubscriptionCount = 0;

            m_IsDeleteRequired = true;
        }
    NN_RESULT_END_TRY;

    std::memset(m_VolatileRecords, 0, sizeof (m_VolatileRecords));

    NN_RESULT_SUCCESS;
}

nn::Result TaskManager::Clear() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_DETAIL_BCAT_SYSTEM_STORAGE_SCOPED_MOUNT();

    NN_RESULT_TRY(nn::fs::DeleteFile(FilePath))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
        }
    NN_RESULT_END_TRY;

    NN_ABORT_UNLESS_RESULT_SUCCESS(FileSystem::Commit(NN_DETAIL_BCAT_SYSTEM_MOUNT_NAME));

    std::memset(m_Records, 0, sizeof (m_Records));
    std::memset(m_RecordsForUnsubscription, 0, sizeof (m_RecordsForUnsubscription));
    m_RecordCount = 0;
    m_RecordForUnsubscriptionCount = 0;

    std::memset(m_VolatileRecords, 0, sizeof (m_VolatileRecords));

    m_IsDeleteRequired = false;

    NN_RESULT_SUCCESS;
}

void TaskManager::Schedule() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    nn::os::Tick currentTick = nn::os::GetSystemTick();

    for (int i = 0; i < m_RecordCount; i++)
    {
        if (m_Records[i].status == TaskStatus_Wait)
        {
            if (m_VolatileRecords[i].runnableTick <= currentTick)
            {
                SetRunnableWithoutSignal(i);
            }
        }
    }

    SetScheduleTimer();
}

nn::Result TaskManager::Suspend(nn::ApplicationId appId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchRecord(appId);

    NN_RESULT_THROW_UNLESS(index >= 0, ResultNotFound());
    NN_RESULT_THROW_UNLESS(m_VolatileRecords[index].suspendCount < UINT8_MAX, ResultInvalidOperation());

    m_VolatileRecords[index].suspendCount++;

    NN_RESULT_SUCCESS;
}

nn::Result TaskManager::Resume(nn::ApplicationId appId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchRecord(appId);

    NN_RESULT_THROW_UNLESS(index >= 0, ResultNotFound());
    NN_RESULT_THROW_UNLESS(m_VolatileRecords[index].suspendCount > 0, ResultInvalidOperation());

    if (--m_VolatileRecords[index].suspendCount == 0)
    {
        if (m_Records[index].status == TaskStatus_Runnable)
        {
            m_ScheduleEvent.Signal();
        }
        else if (m_Records[index].status == TaskStatus_Wait)
        {
            SetScheduleTimer();
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result TaskManager::AddOrUpdate(nn::ApplicationId appId, uint32_t appVersion) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    Record record = {};

    bool isRunnable = false;

    int index = SearchRecord(appId);

    nn::time::PosixTime now = {};

    // ネットワーク時刻が取得できなかった場合、登録時刻を残さない。
    if (nn::time::StandardNetworkSystemClock::GetCurrentTime(&now).IsFailure())
    {
        now.value = 0;
    }

    if (index >= 0)
    {
        record = m_Records[index];

        if (IsApplicationUpdateRequired(index) && record.appVersion < appVersion)
        {
            isRunnable = true;
        }

        record.appVersion = appVersion;

        if (record.registrationTime < now)
        {
            record.registrationTime = now;
        }
    }
    else
    {
        if (IsFull())
        {
            // 一番古いタスクを購読解除する。
            if (m_Records[TaskCountMax - 1].subscription == SubscriptionStatus_Subscribed)
            {
                CreateUnsubscriptionRecord(m_Records[TaskCountMax - 1].appId);
            }
            m_RecordCount--;
        }

        record.appId = appId;
        record.appVersion = appVersion;
        record.subscription = SubscriptionStatus_Empty;
        record.registrationTime = now;

        m_SubscriptionEvent.Signal();

        isRunnable = true;
    }

    RemoveRecord(appId);

    if (m_RecordCount > 0)
    {
        std::memmove(&m_Records[1], &m_Records[0], sizeof (Record) * m_RecordCount);
        std::memmove(&m_VolatileRecords[1], &m_VolatileRecords[0], sizeof (VolatileRecord) * m_RecordCount);
    }

    m_Records[0] = record;
    m_RecordCount++;

    std::memset(&m_VolatileRecords[0], 0, sizeof (VolatileRecord));

    if (isRunnable)
    {
        SetRunnable(0);
    }

    NN_RESULT_DO(Save());

    NN_RESULT_SUCCESS;
}

nn::Result TaskManager::Remove(nn::ApplicationId appId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchRecord(appId);

    if (index == -1)
    {
        NN_RESULT_SUCCESS;
    }

    if (m_Records[index].subscription == SubscriptionStatus_Subscribed)
    {
        CreateUnsubscriptionRecord(appId);
    }

    RemoveRecord(appId);

    NN_RESULT_DO(Save());

    NN_RESULT_SUCCESS;
}

bool TaskManager::GetTaskRecord(Record* outRecord, nn::ApplicationId appId) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outRecord);
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchRecord(appId);

    if (index != -1)
    {
        *outRecord = m_Records[index];
        return true;
    }

    return false;
}

bool TaskManager::GetRunnableTaskRecord(Record* outRecord) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outRecord);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    for (int i = 0; i < m_RecordCount; i++)
    {
        if (m_Records[i].status == TaskStatus_Runnable && m_VolatileRecords[i].suspendCount == 0)
        {
            *outRecord = m_Records[i];
            return true;
        }
    }

    return false;
}

void TaskManager::GetInfoList(int* outCount, TaskInfo* outInfos, int count) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(outInfos);
    NN_SDK_REQUIRES_GREATER(count, 0);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    nn::time::PosixTime now = {};

    if (nn::time::StandardNetworkSystemClock::GetCurrentTime(&now).IsFailure())
    {
        now.value = 0;
    }

    nn::os::Tick currentTick = nn::os::GetSystemTick();

    int actualCount = 0;

    for (int i = 0; i < m_RecordCount; i++)
    {
        if (count-- == 0)
        {
            break;
        }

        TaskInfo& info = outInfos[actualCount++];

        info.appId = m_Records[i].appId;
        info.appVersion = m_Records[i].appVersion;
        info.status = m_Records[i].status;
        info.subscription = m_Records[i].subscription;
        info.lastError = m_Records[i].lastError;
        info.lastRunTime = m_Records[i].lastRunTime;

        if (m_Records[i].status == TaskStatus_Wait && now.value > 0)
        {
            info.nextRunnableTime = now + (m_VolatileRecords[i].runnableTick - currentTick).ToTimeSpan();
        }
        else
        {
            info.nextRunnableTime.value = 0;
        }

        info.registrationTime = m_Records[i].registrationTime;
    }

    *outCount = actualCount;
}

bool TaskManager::GetNextScheduleInterval(nn::TimeSpan* outInterval) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outInterval);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (!GetScheduleMinInterval(outInterval))
    {
        return false;
    }

    if (*outInterval <= 0)
    {
        return false;
    }

    return true;
}

bool TaskManager::GetNpnsTopicToBeSubscribed(TopicId* outTopicId) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outTopicId);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    for (int i = 0; i < m_RecordCount; i++)
    {
        if (m_Records[i].subscription == SubscriptionStatus_Empty)
        {
            ParameterConverter::Convert(outTopicId, m_Records[i].appId);
            return true;
        }
    }

    return false;
}

bool TaskManager::GetNpnsTopicToBeUnsubscribed(TopicId* outTopicId) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outTopicId);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    for (int i = 0; i < m_RecordForUnsubscriptionCount; i++)
    {
        if (m_RecordsForUnsubscription[i].subscription == SubscriptionStatus_Subscribed)
        {
            ParameterConverter::Convert(outTopicId, m_RecordsForUnsubscription[i].appId);
            return true;
        }
    }

    return false;
}

bool TaskManager::HasNpnsTopicToBeSubscribed(bool includesErrorStatus) const NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    for (int i = 0; i < m_RecordCount; i++)
    {
        if (m_Records[i].subscription == SubscriptionStatus_Empty ||
            (includesErrorStatus && m_Records[i].subscription == SubscriptionStatus_Error))
        {
            return true;
        }
    }

    return false;
}

bool TaskManager::HasNpnsTopicToBeUnsubscribed(bool includesErrorStatus) const NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    for (int i = 0; i < m_RecordForUnsubscriptionCount; i++)
    {
        if (m_RecordsForUnsubscription[i].subscription == SubscriptionStatus_Subscribed ||
            (includesErrorStatus && m_RecordsForUnsubscription[i].subscription == SubscriptionStatus_Error))
        {
            return true;
        }
    }

    return false;
}

nn::os::TimerEvent& TaskManager::GetScheduleEvent() NN_NOEXCEPT
{
    return m_ScheduleEvent;
}

nn::os::Event& TaskManager::GetSubscriptionEvent() NN_NOEXCEPT
{
    return m_SubscriptionEvent;
}

nn::Result TaskManager::Wait(nn::ApplicationId appId, int32_t waitTime) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchRecord(appId);

    NN_RESULT_THROW_UNLESS(index >= 0, ResultNotFound());

    if (m_Records[index].status == TaskStatus_Runnable || m_Records[index].status == TaskStatus_Wait)
    {
        SetWait(index, waitTime);
    }

    NN_RESULT_SUCCESS;
}

nn::Result TaskManager::NotifyNotificationReceived(const TopicId& topicId, const RevisionHash& revisionHash, int32_t waitTime) NN_NOEXCEPT
{
    nn::ApplicationId appId;

    if (!ParameterConverter::Convert(&appId, topicId) || appId == nn::ApplicationId::GetInvalidId())
    {
        NN_DETAIL_BCAT_INFO("[bcat] The invalid topic is specified. Do nothing.\n");
        NN_RESULT_SUCCESS;
    }

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchRecord(appId);

    if (index == -1)
    {
        NN_DETAIL_BCAT_INFO("[bcat] '%s' is unmanaged topic id.\n", topicId);

        CreateUnsubscriptionRecord(appId);

        NN_RESULT_SUCCESS;
    }

    if (std::memcmp(revisionHash.value, m_Records[index].revisionHash.value, sizeof (revisionHash.value)) == 0)
    {
        // リビジョンのハッシュ値が一致したら何もしない。
        NN_DETAIL_BCAT_INFO("[bcat] The revision hash is matched. Do nothing.\n");
        NN_RESULT_SUCCESS;
    }

    m_Records[index].revisionHash = revisionHash;

    SetWait(index, waitTime);

    NN_RESULT_SUCCESS;
}

nn::Result TaskManager::NotifyDone(nn::ApplicationId appId, nn::Result result, const RevisionHash& revisionHash) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchRecord(appId);

    NN_RESULT_THROW_UNLESS(index >= 0, ResultNotFound());

    nn::time::PosixTime now = {};

    if (nn::time::StandardNetworkSystemClock::GetCurrentTime(&now).IsSuccess())
    {
        m_Records[index].lastRunTime = now;
    }

    if (result.IsSuccess())
    {
        // タスク開始から終了までにリビジョンが更新されなかった場合、完了状態にする。
        if (std::memcmp(revisionHash.value, m_Records[index].revisionHash.value, sizeof (revisionHash.value)) == 0)
        {
            NN_SDK_ASSERT(m_Records[index].status == TaskStatus_Runnable);

            m_Records[index].status = TaskStatus_Done;
            m_Records[index].lastError = ParameterConverter::Convert(nn::ResultSuccess());
        }
        else
        {
            NN_DETAIL_BCAT_INFO("[bcat] The revision hash is updated. status = %c\n", m_Records[index].status);
        }

        NN_RESULT_DO(Save());
    }
    else
    {
        if (ResultHttpError::Includes(result))
        {
            // 通信エラー。
            SetWait(index, 5 * 60);
        }
        else if (ResultServerError404::Includes(result))
        {
            // データが存在しない。（データ配信リストの再取得で解消する可能性が高い）
            SetWait(index, 5 * 60);
        }
        else if (ResultServerError500::Includes(result) || ResultServerError503::Includes(result) || ResultServerError509::Includes(result))
        {
            // サーバーが不調。
            SetWait(index, 12 * 3600);
        }
        else if (ResultServerError504::Includes(result))
        {
            // サーバーが不調。（CDN の一時的なエラーである可能性が高いが、5xx は一律 12 時間待ちとする）
            SetWait(index, 12 * 3600);
        }
        else if (ResultCanceledByUser::Includes(result))
        {
            // アプリ起動時の一時停止等によるキャンセル。
            SetWait(index, 0);
        }
        else if (ResultInternetRequestCanceled::Includes(result))
        {
            // タスク実行中の通信切断やスリープ等によるキャンセル。
            SetWait(index, 0);
        }
        else
        {
            // 接続エラー・ストレージロックエラー等。
            m_Records[index].status = TaskStatus_Error;
        }

        m_Records[index].lastError = ParameterConverter::Convert(result);

        // 再起動で復帰しないエラーが発生した時は保存する。
        if (!IsRecoverableOnReboot(result))
        {
            NN_RESULT_DO(Save());
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result TaskManager::NotifyStorageUnmounted(nn::ApplicationId appId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchRecord(appId);

    NN_RESULT_THROW_UNLESS(index >= 0, ResultNotFound());

    if (m_Records[index].status == TaskStatus_Error)
    {
        nn::Result result = ParameterConverter::Convert(m_Records[index].lastError);

        // ストレージロックエラー状態のタスクを実行可能にする。
        if (ResultLocked::Includes(result))
        {
            SetRunnable(index);
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result TaskManager::NotifyNpnsTopicSubscriptionProcessed(const TopicId& topicId, nn::Result result) NN_NOEXCEPT
{
    nn::ApplicationId appId;

    NN_RESULT_THROW_UNLESS(ParameterConverter::Convert(&appId, topicId), ResultInvalidArgument());
    NN_RESULT_THROW_UNLESS(appId != nn::ApplicationId::GetInvalidId(), ResultInvalidArgument());

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    for (int i = 0; i < m_RecordCount; i++)
    {
        if (appId == m_Records[i].appId)
        {
            if (result.IsSuccess())
            {
                m_Records[i].subscription = SubscriptionStatus_Subscribed;
                NN_RESULT_DO(Save());
            }
            else
            {
                m_Records[i].subscription = SubscriptionStatus_Error;
            }
            break;
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result TaskManager::NotifyNpnsTopicUnsubscriptionProcessed(const TopicId& topicId, nn::Result result) NN_NOEXCEPT
{
    nn::ApplicationId appId;

    NN_RESULT_THROW_UNLESS(ParameterConverter::Convert(&appId, topicId), ResultInvalidArgument());
    NN_RESULT_THROW_UNLESS(appId != nn::ApplicationId::GetInvalidId(), ResultInvalidArgument());

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    for (int i = 0; i < m_RecordForUnsubscriptionCount; i++)
    {
        if (appId == m_RecordsForUnsubscription[i].appId)
        {
            if (result.IsSuccess())
            {
                RemoveRecordForUnsubscription(appId);
                NN_RESULT_DO(Save());
            }
            else
            {
                m_RecordsForUnsubscription[i].subscription = SubscriptionStatus_Error;
            }
            break;
        }
    }

    NN_RESULT_SUCCESS;
}

void TaskManager::NotifyNintendoAccountLinked() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    for (int i = 0; i < m_RecordCount; i++)
    {
        if (IsNintendoAccountRequired(i))
        {
            SetRunnable(i);
        }
    }
}

void TaskManager::NotifySleepAwaked() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    bool isSubscriptionRequired = false;

    for (int i = 0; i < m_RecordCount; i++)
    {
        Record& record = m_Records[i];

        if (record.subscription == SubscriptionStatus_Error)
        {
            record.subscription = SubscriptionStatus_Empty;
            isSubscriptionRequired = true;
        }
    }
    for (int i = 0; i < m_RecordForUnsubscriptionCount; i++)
    {
        Record& record = m_RecordsForUnsubscription[i];

        if (record.subscription == SubscriptionStatus_Error)
        {
            record.subscription = SubscriptionStatus_Subscribed;
            isSubscriptionRequired = true;
        }
    }

    if (isSubscriptionRequired)
    {
        m_SubscriptionEvent.Signal();
    }
}

void TaskManager::NotifyNetworkConnected() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    for (int i = 0; i < m_RecordCount; i++)
    {
        Record& record = m_Records[i];

        if (record.status == TaskStatus_Error)
        {
            nn::Result result = ParameterConverter::Convert(record.lastError);

            // 接続エラー状態のタスクを実行可能にする。
            if (ResultInternetRequestNotAccepted::Includes(result))
            {
                SetRunnable(i);
            }
        }
    }
}

nn::Result TaskManager::LoadImpl() NN_NOEXCEPT
{
    NN_DETAIL_BCAT_SYSTEM_STORAGE_SCOPED_MOUNT_READ_ONLY();

    nn::fs::FileHandle handle = {};

    NN_RESULT_TRY(nn::fs::OpenFile(&handle, FilePath, nn::fs::OpenMode_Read))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            NN_RESULT_RETHROW;
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(NN_RESULT_CURRENT_RESULT);
        }
    NN_RESULT_END_TRY;

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    int64_t size = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&size, handle));

    NN_RESULT_THROW_UNLESS(size == sizeof (m_Records) + sizeof (m_RecordsForUnsubscription), ResultSaveVerificationFailed());

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(handle, 0, m_Records, sizeof (m_Records)));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(handle, sizeof (m_Records), m_RecordsForUnsubscription, sizeof (m_RecordsForUnsubscription)));

    m_RecordCount = 0;

    for (auto& record : m_Records)
    {
        if (record.appId == nn::ApplicationId::GetInvalidId())
        {
            break;
        }
        m_RecordCount++;
    }

    m_RecordForUnsubscriptionCount = 0;

    for (auto& record : m_RecordsForUnsubscription)
    {
        if (record.appId == nn::ApplicationId::GetInvalidId())
        {
            break;
        }
        m_RecordForUnsubscriptionCount++;
    }

    if (!Verify())
    {
        NN_DETAIL_BCAT_WARN("[bcat] %s is corrupted. (verification failed)\n", FilePath);
        NN_RESULT_THROW(ResultSaveVerificationFailed());
    }

    Recover();

    NN_RESULT_SUCCESS;
}

void TaskManager::Recover() NN_NOEXCEPT
{
    bool isSubscriptionRequired = false;

    for (int i = 0; i < m_RecordCount; i++)
    {
        Record& record = m_Records[i];

        if (record.status == TaskStatus_Runnable)
        {
            m_ScheduleEvent.Signal();
        }
        else if (record.status == TaskStatus_Wait)
        {
            SetRunnable(i);
        }
        else if (record.status == TaskStatus_Error)
        {
            // 再起動で復帰可能なエラーは復帰させる。
            if (IsRecoverableOnReboot(ParameterConverter::Convert(record.lastError)))
            {
                SetRunnable(i);
            }
        }

        if (record.subscription == SubscriptionStatus_Error)
        {
            record.subscription = SubscriptionStatus_Empty;
            isSubscriptionRequired = true;
        }
    }
    for (int i = 0; i < m_RecordForUnsubscriptionCount; i++)
    {
        Record& record = m_RecordsForUnsubscription[i];

        if (record.subscription == SubscriptionStatus_Error)
        {
            record.subscription = SubscriptionStatus_Subscribed;
            isSubscriptionRequired = true;
        }
    }

    if (isSubscriptionRequired)
    {
        m_SubscriptionEvent.Signal();
    }
}

nn::Result TaskManager::Save() NN_NOEXCEPT
{
    // 待機状態、エラー状態、および、それらの状態からの復帰（実行可能状態への遷移）はシステム再起動時に行われるため、保存処理は行わない。

    NN_DETAIL_BCAT_SYSTEM_STORAGE_SCOPED_MOUNT();

    if (m_IsDeleteRequired)
    {
        NN_RESULT_TRY(nn::fs::DeleteFile(FilePath))
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
            }
        NN_RESULT_END_TRY;
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(FileSystem::CreateFile(FilePath, sizeof (m_Records) + sizeof (m_RecordsForUnsubscription)));

    {
        nn::fs::FileHandle handle = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&handle, FilePath, nn::fs::OpenMode_Write));

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::WriteFile(handle, 0, m_Records, sizeof (m_Records),
            nn::fs::WriteOption::MakeValue(0)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::WriteFile(handle, sizeof (m_Records), m_RecordsForUnsubscription, sizeof (m_RecordsForUnsubscription),
            nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(FileSystem::Commit(NN_DETAIL_BCAT_SYSTEM_MOUNT_NAME));

    m_IsDeleteRequired = false;

    NN_RESULT_SUCCESS;
}

bool TaskManager::Verify() NN_NOEXCEPT
{
    int runningCount = 0;

    for (int i = 0; i < m_RecordCount; i++)
    {
        if (m_Records[i].status != TaskStatus_Empty &&
            m_Records[i].status != TaskStatus_Runnable &&
            m_Records[i].status != TaskStatus_Wait &&
            m_Records[i].status != TaskStatus_Done &&
            m_Records[i].status != TaskStatus_Error)
        {
            return false;
        }
        if (m_Records[i].subscription != SubscriptionStatus_Empty &&
            m_Records[i].subscription != SubscriptionStatus_Error &&
            m_Records[i].subscription != SubscriptionStatus_Subscribed)
        {
            return false;
        }
    }

    for (int i = 0; i < m_RecordForUnsubscriptionCount; i++)
    {
        if (m_RecordsForUnsubscription[i].status != TaskStatus_Empty)
        {
            return false;
        }
        if (m_RecordsForUnsubscription[i].subscription != SubscriptionStatus_Error &&
            m_RecordsForUnsubscription[i].subscription != SubscriptionStatus_Subscribed)
        {
            return false;
        }
    }

    return (runningCount <= 1);
}

int TaskManager::SearchRecord(nn::ApplicationId appId) const NN_NOEXCEPT
{
    for (int i = 0; i < m_RecordCount; i++)
    {
        if (appId == m_Records[i].appId)
        {
            return i;
        }
    }

    return -1;
}

bool TaskManager::IsFull() const NN_NOEXCEPT
{
    return (m_RecordCount == TaskCountMax);
}

void TaskManager::RemoveRecord(nn::ApplicationId appId) NN_NOEXCEPT
{
    for (int i = 0; i < m_RecordCount; i++)
    {
        if (appId == m_Records[i].appId)
        {
            int moveCount = m_RecordCount - 1 - i;

            if (moveCount > 0)
            {
                std::memmove(&m_Records[i], &m_Records[i + 1], sizeof (Record) * moveCount);
                std::memmove(&m_VolatileRecords[i], &m_VolatileRecords[i + 1], sizeof (VolatileRecord) * moveCount);
            }
            m_RecordCount--;

            // メモリダンプを保存するため、末尾をゼロクリアしておく。
            std::memset(&m_Records[m_RecordCount], 0, sizeof (Record));
            return;
        }
    }
}

void TaskManager::RemoveRecordForUnsubscription(nn::ApplicationId appId) NN_NOEXCEPT
{
    for (int i = 0; i < m_RecordForUnsubscriptionCount; i++)
    {
        if (appId == m_RecordsForUnsubscription[i].appId)
        {
            int moveCount = m_RecordForUnsubscriptionCount - 1 - i;

            if (moveCount > 0)
            {
                std::memmove(&m_RecordsForUnsubscription[i], &m_RecordsForUnsubscription[i + 1], sizeof (Record) * moveCount);
            }
            m_RecordForUnsubscriptionCount--;

            // メモリダンプを保存するため、末尾をゼロクリアしておく。
            std::memset(&m_RecordsForUnsubscription[m_RecordForUnsubscriptionCount], 0, sizeof (Record));
            return;
        }
    }
}

void TaskManager::CreateUnsubscriptionRecord(nn::ApplicationId appId) NN_NOEXCEPT
{
    if (m_RecordForUnsubscriptionCount == TaskCountMax)
    {
        return;
    }

    for (int i = 0; i < m_RecordForUnsubscriptionCount; i++)
    {
        if (appId == m_RecordsForUnsubscription[i].appId)
        {
            return;
        }
    }

    Record record = {};

    record.appId = appId;
    record.subscription = SubscriptionStatus_Subscribed;

    m_RecordsForUnsubscription[m_RecordForUnsubscriptionCount++] = record;

    m_SubscriptionEvent.Signal();
}

void TaskManager::SetRunnable(int index) NN_NOEXCEPT
{
    m_Records[index].status = TaskStatus_Runnable;

    m_ScheduleEvent.Signal();
}

void TaskManager::SetRunnableWithoutSignal(int index) NN_NOEXCEPT
{
    m_Records[index].status = TaskStatus_Runnable;
}

void TaskManager::SetWait(int index, int32_t waitTime) NN_NOEXCEPT
{
    if (waitTime == 0)
    {
        SetRunnable(index);
    }
    else
    {
        if (m_Records[index].status == TaskStatus_Wait)
        {
            nn::TimeSpan remainWaitTime = (m_VolatileRecords[index].runnableTick - nn::os::GetSystemTick()).ToTimeSpan();

            // 待機時間が短くならない場合、無視する。
            if (static_cast<int32_t>(remainWaitTime.GetSeconds()) <= waitTime)
            {
                return;
            }
        }
        else
        {
            m_Records[index].status = TaskStatus_Wait;
        }

        if (waitTime != -1)
        {
            m_VolatileRecords[index].runnableTick = nn::os::GetSystemTick() + nn::os::ConvertToTick(nn::TimeSpan::FromSeconds(waitTime));

            NN_DETAIL_BCAT_INFO("[bcat] Wait for %d seconds ...\n", waitTime);

            SetScheduleTimer();
        }
        else
        {
            m_VolatileRecords[index].runnableTick = nn::os::Tick(INT64_MAX);
        }
    }
}

void TaskManager::SetScheduleTimer() NN_NOEXCEPT
{
    nn::TimeSpan interval;

    if (GetScheduleMinInterval(&interval))
    {
        if (interval > 0)
        {
            m_ScheduleEvent.StartOneShot(interval);
        }
        else
        {
            m_ScheduleEvent.Signal();
        }
    }
}

bool TaskManager::GetScheduleMinInterval(nn::TimeSpan* outInterval) const NN_NOEXCEPT
{
    nn::os::Tick nextRunnableTick = nn::os::Tick(INT64_MAX);
    bool hasWaitTask = false;

    for (int i = 0; i < m_RecordCount; i++)
    {
        if (m_Records[i].status == TaskStatus_Wait &&
            m_VolatileRecords[i].suspendCount == 0 && m_VolatileRecords[i].runnableTick < nextRunnableTick)
        {
            nextRunnableTick = m_VolatileRecords[i].runnableTick;
            hasWaitTask = true;
        }
    }

    if (!hasWaitTask)
    {
        return false;
    }

    *outInterval = (nextRunnableTick - nn::os::GetSystemTick()).ToTimeSpan();

    return true;
}

bool TaskManager::IsNintendoAccountRequired(int index) const NN_NOEXCEPT
{
    return (m_Records[index].status == TaskStatus_Error &&
        ResultNintendoAccountNotLinked::Includes(ParameterConverter::Convert(m_Records[index].lastError)));
}

bool TaskManager::IsApplicationUpdateRequired(int index) const NN_NOEXCEPT
{
    return (m_Records[index].status == TaskStatus_Error &&
        ResultApplicationUpdateRequired::Includes(ParameterConverter::Convert(m_Records[index].lastError)));
}

bool TaskManager::IsRecoverableOnReboot(nn::Result result) const NN_NOEXCEPT
{
    return !(ResultNintendoAccountNotLinked::Includes(result) ||
        ResultApplicationUpdateRequired::Includes(result) ||
        ResultServiceExpired::Includes(result));
}

}}}}}
