﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/detail/service/core/bcat_MetaView.h>

namespace nn { namespace bcat { namespace detail { namespace service { namespace core {

namespace
{
    void MakeDownloadFileMetaPath(char* path, size_t size, nn::ApplicationId appId, const char* dirName) NN_NOEXCEPT
    {
        int length = nn::util::SNPrintf(path, size, "bcat-dl:/applications/%016llx/directories/%s/files.meta",
            appId.value, dirName);

        NN_UNUSED(length);
        NN_SDK_ASSERT(length < static_cast<int>(size));
    }
}

nn::Result FileMetaView::Load(const char* path, bool ignoreNotFound) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    nn::fs::FileHandle handle = {};

    NN_RESULT_TRY(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            if (ignoreNotFound)
            {
                m_Count = 0;
                NN_RESULT_SUCCESS;
            }
            else
            {
                NN_RESULT_THROW(ResultNotFound());
            }
        }
    NN_RESULT_END_TRY;

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    m_Count = 0;

    Header header = {};
    size_t size;

    NN_RESULT_DO(nn::fs::ReadFile(&size, handle, 0, &header, sizeof (header)));

    NN_RESULT_THROW_UNLESS(size == sizeof (header), ResultSaveVerificationFailed());
    NN_RESULT_THROW_UNLESS(header.version > 0 && header.version <= Entry::StructureVersion, ResultSaveVerificationFailed());

    if (header.version == FileMeta::StructureVersion)
    {
        NN_RESULT_DO(nn::fs::ReadFile(&size, handle, sizeof (header), m_Records, sizeof (m_Records)));

        NN_SDK_ASSERT(size % sizeof (FileMeta) == 0);

        m_Count = static_cast<int>(size) / sizeof (FileMeta);

        // MEMO: Entry = FileMetaV2 以降になった時、読み込んだデータをコンバートする。
    }
    // else if (header.version == FileMetaV2::StructureVersion) {...}

    NN_RESULT_SUCCESS;
}

nn::Result FileMetaView::Save(const char* path) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int64_t metaSize = sizeof (Entry) * m_Count;
    int64_t fileSize = sizeof (Header) + metaSize;

    nn::fs::FileHandle handle = {};

    NN_RESULT_TRY(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Write))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            NN_RESULT_DO(FileSystem::CreateFile(path, fileSize));
            NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Write));
        }
    NN_RESULT_END_TRY;

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    NN_RESULT_DO(nn::fs::SetFileSize(handle, fileSize));

    Header header = {Entry::StructureVersion};

    NN_RESULT_DO(nn::fs::WriteFile(handle, 0, &header, sizeof (header), nn::fs::WriteOption::MakeValue(0)));

    if (m_Count > 0)
    {
        NN_RESULT_DO(nn::fs::WriteFile(handle, sizeof (header), m_Records, static_cast<size_t>(metaSize),
            nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
    }
    else
    {
        NN_RESULT_DO(nn::fs::FlushFile(handle));
    }

    NN_RESULT_SUCCESS;
}

nn::Result FileMetaView::LoadFromDeliveryCacheStorage(FileMetaView* outView,
    nn::ApplicationId appId, const char* dirName, bool ignoreNotFound) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outView);
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());
    NN_SDK_REQUIRES_NOT_NULL(dirName);

    char path[80] = {};
    DeliveryCacheStorageManager::GetInstance().MakeFileMetaPath(path, sizeof (path), appId, dirName);

    NN_RESULT_DO(outView->Load(path, ignoreNotFound));

    NN_RESULT_SUCCESS;
}

nn::Result FileMetaView::LoadFromDownloadStorage(FileMetaView* outView,
    nn::ApplicationId appId, const char* dirName, bool ignoreNotFound) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outView);
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());
    NN_SDK_REQUIRES_NOT_NULL(dirName);

    char path[160] = {};
    MakeDownloadFileMetaPath(path, sizeof (path), appId, dirName);

    NN_RESULT_DO(outView->Load(path, ignoreNotFound));

    NN_RESULT_SUCCESS;
}

nn::Result FileMetaView::SaveToDeliveryCacheStorage(const FileMetaView& view,
    nn::ApplicationId appId, const char* dirName, bool needsCommit) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());
    NN_SDK_REQUIRES_NOT_NULL(dirName);

    char path[80] = {};
    DeliveryCacheStorageManager::GetInstance().MakeFileMetaPath(path, sizeof (path), appId, dirName);

    NN_RESULT_DO(view.Save(path));

    if (needsCommit)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(DeliveryCacheStorageManager::GetInstance().Commit(appId));
    }

    NN_RESULT_SUCCESS;
}

nn::Result FileMetaView::SaveToDownloadStorage(const FileMetaView& view,
    nn::ApplicationId appId, const char* dirName, bool needsCommit) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());
    NN_SDK_REQUIRES_NOT_NULL(dirName);

    char path[160] = {};
    MakeDownloadFileMetaPath(path, sizeof (path), appId, dirName);

    NN_RESULT_DO(view.Save(path));

    if (needsCommit)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(FileSystem::Commit(NN_DETAIL_BCAT_DOWNLOAD_MOUNT_NAME));
    }

    NN_RESULT_SUCCESS;
}

nn::Result DirectoryMetaView::Load(const char* path, bool ignoreNotFound) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    nn::fs::FileHandle handle = {};

    NN_RESULT_TRY(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            if (ignoreNotFound)
            {
                m_Count = 0;
                NN_RESULT_SUCCESS;
            }
            else
            {
                NN_RESULT_THROW(ResultNotFound());
            }
        }
    NN_RESULT_END_TRY;

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    m_Count = 0;

    Header header = {};
    size_t size;

    NN_RESULT_DO(nn::fs::ReadFile(&size, handle, 0, &header, sizeof (header)));

    NN_RESULT_THROW_UNLESS(size == sizeof (header), ResultSaveVerificationFailed());
    NN_RESULT_THROW_UNLESS(header.version > 0 && header.version <= Entry::StructureVersion, ResultSaveVerificationFailed());

    if (header.version == DirectoryMeta::StructureVersion)
    {
        NN_RESULT_DO(nn::fs::ReadFile(&size, handle, sizeof (header), m_Records, sizeof (m_Records)));

        NN_SDK_ASSERT(size % sizeof (DirectoryMeta) == 0);

        m_Count = static_cast<int>(size) / sizeof (DirectoryMeta);

        // MEMO: Entry = DirectoryMetaV2 以降になった時、読み込んだデータをコンバートする。
    }
    // else if (header.version == DirectoryMetaV2::StructureVersion) {...}

    NN_RESULT_SUCCESS;
}

nn::Result DirectoryMetaView::Save(const char* path) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int64_t metaSize = sizeof (Entry) * m_Count;
    int64_t fileSize = sizeof (Header) + metaSize;

    nn::fs::FileHandle handle = {};

    NN_RESULT_TRY(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Write))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            NN_RESULT_DO(FileSystem::CreateFile(path, fileSize));
            NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Write));
        }
    NN_RESULT_END_TRY;

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    NN_RESULT_DO(nn::fs::SetFileSize(handle, fileSize));

    Header header = {Entry::StructureVersion};

    NN_RESULT_DO(nn::fs::WriteFile(handle, 0, &header, sizeof (header), nn::fs::WriteOption::MakeValue(0)));

    if (m_Count > 0)
    {
        NN_RESULT_DO(nn::fs::WriteFile(handle, sizeof (header), m_Records, static_cast<size_t>(metaSize),
            nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
    }
    else
    {
        NN_RESULT_DO(nn::fs::FlushFile(handle));
    }

    NN_RESULT_SUCCESS;
}

nn::Result DirectoryMetaView::LoadFromDeliveryCacheStorage(DirectoryMetaView* outView,
    nn::ApplicationId appId, bool ignoreNotFound) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outView);
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    char path[80] = {};
    DeliveryCacheStorageManager::GetInstance().MakeDirectoryMetaPath(path, sizeof (path), appId);

    NN_RESULT_DO(outView->Load(path, ignoreNotFound));

    NN_RESULT_SUCCESS;
}

nn::Result DirectoryMetaView::SaveToDeliveryCacheStorage(const DirectoryMetaView& view,
    nn::ApplicationId appId, bool needsCommit) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    char path[80] = {};
    DeliveryCacheStorageManager::GetInstance().MakeDirectoryMetaPath(path, sizeof (path), appId);

    NN_RESULT_DO(view.Save(path));

    if (needsCommit)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(DeliveryCacheStorageManager::GetInstance().Commit(appId));
    }

    NN_RESULT_SUCCESS;
}

}}}}}
