﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/detail/service/core/bcat_DirectoryDownloader.h>
#include <nn/bcat/detail/service/core/bcat_DeliveryListReader.h>
#include <nn/bcat/detail/service/msgpack/bcat_FileInputStream.h>

namespace nn { namespace bcat { namespace detail { namespace service { namespace core {

namespace
{
    struct ProgressCallbackParam
    {
        nn::nifm::NetworkConnection* pConnection;
        nn::os::Event* pCancelEvent;
        DirectoryDownloader::DownloadFileCallback callback;
        void* callbackParam;
    };
}

namespace
{
    bool ProgressCallback(int64_t now, int64_t total, void* param) NN_NOEXCEPT
    {
        NN_UNUSED(total);

        ProgressCallbackParam* cp = reinterpret_cast<ProgressCallbackParam*>(param);

        if (cp->callback)
        {
            int64_t downloaded = now > sizeof (nn::bcat::service::ArchiveHeader) ?
                now - sizeof (nn::bcat::service::ArchiveHeader) : 0;

            cp->callback(downloaded, cp->callbackParam);
        }

        return (cp->pConnection->IsAvailable() && !(cp->pCancelEvent && cp->pCancelEvent->TryWait()));
    }

    void MakeFilePath(char* path, size_t size, nn::ApplicationId appId, const char* dirName, const char* fileName) NN_NOEXCEPT
    {
        int length = nn::util::SNPrintf(path, size, "bcat-dl:/applications/%016llx/directories/%s/files/%s",
            appId.value, dirName, fileName);

        NN_UNUSED(length);
        NN_SDK_ASSERT(length < static_cast<int>(size));
    }

    void MakeDownloadFilePath(char* path, size_t size, nn::ApplicationId appId, const char* dirName, const char* fileName) NN_NOEXCEPT
    {
        int length = nn::util::SNPrintf(path, size, "bcat-dl:/applications/%016llx/directories/%s/download/%s",
            appId.value, dirName, fileName);

        NN_UNUSED(length);
        NN_SDK_ASSERT(length < static_cast<int>(size));
    }

    void MakeDownloadMetaFilePath(char* path, size_t size, nn::ApplicationId appId, const char* dirName, const char* fileName) NN_NOEXCEPT
    {
        int length = nn::util::SNPrintf(path, size, "bcat-dl:/applications/%016llx/directories/%s/download/%s.bafmeta",
            appId.value, dirName, fileName);

        NN_UNUSED(length);
        NN_SDK_ASSERT(length < static_cast<int>(size));
    }

    void MakeDownloadDirectoryPath(char* path, size_t size, nn::ApplicationId appId, const char* dirName) NN_NOEXCEPT
    {
        int length = nn::util::SNPrintf(path, size, "bcat-dl:/applications/%016llx/directories/%s/download",
            appId.value, dirName);

        NN_UNUSED(length);
        NN_SDK_ASSERT(length < static_cast<int>(size));
    }
}

DirectoryDownloader::DirectoryDownloader() NN_NOEXCEPT :
    m_Mutex(true),
    m_AppId(nn::ApplicationId::GetInvalidId()),
    m_pCancelEvent(nullptr),
    m_LastProgressCallback(nullptr),
    m_LastProgressCallbackParam(nullptr),
    m_StartDownloadFileCallback(nullptr),
    m_StartDownloadFileCallbackParam(nullptr),
    m_DownloadFileCallback(nullptr),
    m_DownloadFileCallbackParam(nullptr),
    m_FileCount(0),
    m_IncompleteFileIndex(-1)
{
}

void DirectoryDownloader::SetCancelEvent(nn::os::Event* pCancelEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pCancelEvent);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    m_pCancelEvent = pCancelEvent;
}

void DirectoryDownloader::SetLastProgressCallback(LastProgressCallback callback, void* param) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(callback);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    m_LastProgressCallback = callback;
    m_LastProgressCallbackParam = param;
}

void DirectoryDownloader::SetStartDownloadFileCallback(StartDownloadFileCallback callback, void* param) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(callback);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    m_StartDownloadFileCallback = callback;
    m_StartDownloadFileCallbackParam = param;
}

void DirectoryDownloader::SetDownloadFileCallback(DownloadFileCallback callback, void* param) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(callback);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    m_DownloadFileCallback = callback;
    m_DownloadFileCallbackParam = param;
}

nn::Result DirectoryDownloader::Download(nn::ApplicationId appId, const DirectoryName& dirName, const char* passphrase,
    nn::nifm::NetworkConnection* pConnection, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());
    NN_SDK_REQUIRES_NOT_NULL(passphrase);
    NN_SDK_REQUIRES_NOT_NULL(pConnection);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_GREATER(bufferSize, 0u);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_DETAIL_BCAT_INFO("[bcat] DirectoryDownloader::Download(%016llx, %s) ...\n", appId.value, dirName.value);

    m_AppId = appId;
    m_DirName = dirName;

    nn::util::Strlcpy(m_Passphrase, passphrase, sizeof (m_Passphrase));

    NN_RESULT_DO(StepSetup(buffer, bufferSize));
    NN_RESULT_DO(StepCheckDownloadable());
    NN_RESULT_DO(StepDownloadFiles(pConnection, buffer, bufferSize));

    NN_RESULT_SUCCESS;
}

nn::Result DirectoryDownloader::StepSetup(void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_DETAIL_BCAT_INFO("[bcat] - DirectoryDownloader::StepSetup ...\n");

    char listPath[64] = {};
    DeliveryCacheStorageManager::GetInstance().MakeListPath(listPath, sizeof (listPath), m_AppId);

    detail::service::msgpack::FileInputStream stream;

    stream.SetBuffer(buffer, bufferSize);

    NN_RESULT_DO(stream.Open(listPath));

    DeliveryListReader reader;

    reader.SetupForFileListing(&m_FileCount, m_Files, NN_ARRAY_SIZE(m_Files), m_DirName.value);

    NN_RESULT_DO(reader.Read(nullptr, stream));

    stream.Close();

    NN_RESULT_DO(FileMetaView::LoadFromDeliveryCacheStorage(&m_FileMetaView, m_AppId, m_DirName.value));
    NN_RESULT_DO(m_FileDiffView.Make(m_FileMetaView, m_Files, m_FileCount));

    NN_ABORT_UNLESS_RESULT_SUCCESS(FileMetaView::LoadFromDownloadStorage(&m_DownloadFileMetaView, m_AppId, m_DirName.value));

    NN_RESULT_TRY(GetIncompleteFileIndex(&m_IncompleteFileIndex))
        NN_RESULT_CATCH(ResultNotFound)
        {
            m_IncompleteFileIndex = -1;
        }
        NN_RESULT_CATCH(ResultInvalidState)
        {
            m_IncompleteFileIndex = -1;

            char dlDirPath[160];
            MakeDownloadDirectoryPath(dlDirPath, sizeof (dlDirPath), m_AppId, m_DirName.value);

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::DeleteDirectoryRecursively(dlDirPath));

            NN_DETAIL_BCAT_INFO("[bcat] Cleanup file download directory.\n");
        }
    NN_RESULT_END_TRY;

    NN_RESULT_DO(GetLastProgress());

    NN_RESULT_SUCCESS;
}

nn::Result DirectoryDownloader::StepCheckDownloadable() NN_NOEXCEPT
{
    NN_DETAIL_BCAT_INFO("[bcat] - DirectoryDownloader::StepCheckDownloadable ...\n");

    int indexes[DeliveryCacheFileCountMaxPerDirectory] = {};
    int count = 0;

    m_FileDiffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), DiffType_Remove);

    int64_t increaseSize = 0;
    int64_t decreaseSize = 0;

    for (int i = 0; i < count; i++)
    {
        FileMeta entry = {};
        m_FileMetaView.GetEntry(&entry, indexes[i]);

        decreaseSize += FileSystem::CalculateFileSizeOnStorage(entry.size);
    }

    int64_t oldTotal = 0;
    int64_t newTotal = 0;

    m_FileDiffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), DiffType_Update);

    for (int i = 0; i < count; i++)
    {
        FileMeta entry = {};
        m_FileMetaView.GetEntry(&entry, indexes[i]);

        oldTotal += FileSystem::CalculateFileSizeOnStorage(entry.size);
    }

    m_FileDiffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), DiffType_Update);

    for (int i = 0; i < count; i++)
    {
        const ListFile& entry = m_Files[indexes[i]];

        newTotal += FileSystem::CalculateFileSizeOnStorage(entry.size);
    }

    if (oldTotal < newTotal)
    {
        increaseSize += newTotal - oldTotal;
    }
    else
    {
        decreaseSize += oldTotal - newTotal;
    }

    m_FileDiffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), DiffType_Add);

    for (int i = 0; i < count; i++)
    {
        const ListFile& entry = m_Files[indexes[i]];

        increaseSize += FileSystem::CalculateFileSizeOnStorage(entry.size);
    }

    int64_t freeSpaceSize = 0;
    NN_RESULT_DO(DeliveryCacheStorageManager::GetInstance().GetFreeSpaceSize(&freeSpaceSize, m_AppId));

    // FS の管理テーブル分を計上しておく。
    int64_t remainSize = freeSpaceSize + decreaseSize - increaseSize - BlockSize * 2;

    NN_DETAIL_BCAT_INFO("[bcat] StorageInfo:\n");
    NN_DETAIL_BCAT_INFO("[bcat]     FreeSpaceSize = %lld\n", freeSpaceSize);
    NN_DETAIL_BCAT_INFO("[bcat]     Reserved = %lld\n", BlockSize * 2);
    NN_DETAIL_BCAT_INFO("[bcat]     + = %lld\n", increaseSize);
    NN_DETAIL_BCAT_INFO("[bcat]     - = %lld\n", decreaseSize);
    NN_DETAIL_BCAT_INFO("[bcat]     RemainSize = %lld\n", remainSize);

    NN_RESULT_THROW_UNLESS(remainSize >= 0, ResultUsableSpaceNotEnough());

    NN_RESULT_SUCCESS;
}

nn::Result DirectoryDownloader::StepDownloadFiles(nn::nifm::NetworkConnection* pConnection, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_DETAIL_BCAT_INFO("[bcat] - DirectoryDownloader::StepDownloadFiles ...\n");

    int indexes[DeliveryCacheFileCountMaxPerDirectory] = {};
    int count = 0;

    m_FileDiffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), DiffType_Update);

    if (count < NN_ARRAY_SIZE(indexes))
    {
        int c = 0;

        m_FileDiffView.GetIndexListFromRemote(&c, indexes + count, NN_ARRAY_SIZE(indexes) - count, DiffType_Add);
        count += c;
    }

    if (count == 0 && m_IncompleteFileIndex == -1)
    {
        NN_DETAIL_BCAT_INFO("[bcat] Do nothing.\n");
        NN_RESULT_SUCCESS;
    }

    if (m_IncompleteFileIndex != -1)
    {
        NN_RESULT_DO(DownloadFile(m_Files[m_IncompleteFileIndex], pConnection, buffer, bufferSize));
    }

    for (int i = 0; i < count; i++)
    {
        FileMeta entry = {};

        // ダウンロード済みならスキップする。
        if (m_DownloadFileMetaView.Search(nullptr, &entry, m_Files[indexes[i]].name.value).IsSuccess())
        {
            // NN_DETAIL_BCAT_INFO("[bcat] The file(%s/%s) was already downloaded.\n", m_DirName.value, entry.name.value);
        }
        else
        {
            NN_RESULT_DO(DownloadFile(m_Files[indexes[i]], pConnection, buffer, bufferSize));
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result DirectoryDownloader::GetIncompleteFileIndex(int* outIndex) NN_NOEXCEPT
{
    char path[160];
    MakeDownloadDirectoryPath(path, sizeof (path), m_AppId, m_DirName.value);

    nn::fs::DirectoryEntry entries[3] = {};
    int64_t count = 0;

    nn::fs::DirectoryHandle handle = {};

    NN_RESULT_TRY(nn::fs::OpenDirectory(&handle, path, nn::fs::OpenDirectoryMode_File))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            NN_RESULT_THROW(ResultNotFound());
        }
        NN_RESULT_CATCH(nn::fs::ResultDataCorrupted)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(NN_RESULT_CURRENT_RESULT);
        }
    NN_RESULT_END_TRY;

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseDirectory(handle);
    };

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadDirectory(&count, entries, handle, NN_ARRAY_SIZE(entries)));

    // 正常な状態なら、中間データの数は 0 か 2 のはず。
    if (count == 0)
    {
        NN_RESULT_THROW(ResultNotFound());
    }

    NN_RESULT_THROW_UNLESS(count == 2, ResultInvalidState());

    int index = -1;

    for (int i = 0; i < m_FileCount; i++)
    {
        int length = nn::util::Strnlen(m_Files[i].name.value, sizeof (m_Files[i].name.value));

        if (nn::util::Strnicmp(entries[0].name, m_Files[i].name.value, length) == 0 &&
            nn::util::Strnicmp(entries[1].name, m_Files[i].name.value, length) == 0)
        {
            const char* extension0 = &entries[0].name[length];
            const char* extension1 = &entries[1].name[length];

            if ((nn::util::Strncmp(extension0, ".incomplete", sizeof (".incomplete")) == 0 &&
                 nn::util::Strncmp(extension1, ".bafmeta", sizeof (".bafmeta")) == 0) ||
                (nn::util::Strncmp(extension1, ".incomplete", sizeof (".incomplete")) == 0 &&
                 nn::util::Strncmp(extension0, ".bafmeta", sizeof (".bafmeta")) == 0))
            {
                index = i;
                break;
            }
        }
    }

    NN_RESULT_THROW_UNLESS(index != -1, ResultInvalidState());

    *outIndex = index;

    NN_RESULT_SUCCESS;
}

nn::Result DirectoryDownloader::GetLastProgress() NN_NOEXCEPT
{
    int count = m_DownloadFileMetaView.GetCount();

    if (count == 0)
    {
        NN_RESULT_SUCCESS;
    }

    int64_t downloaded = 0;

    FileMeta entry = {};

    for (int i = 0; i < count; i++)
    {
        m_DownloadFileMetaView.GetEntry(&entry, i);

        downloaded += entry.size;
    }

    NN_DETAIL_BCAT_INFO("[bcat] LastDownloadInfo:\n");
    NN_DETAIL_BCAT_INFO("[bcat]     DownloadedSize = %lld\n", downloaded);
    NN_DETAIL_BCAT_INFO("[bcat]     DirectoryName = %s\n", m_DirName.value);

    if (m_LastProgressCallback)
    {
        m_LastProgressCallback(downloaded, m_DirName, m_LastProgressCallbackParam);
    }

    NN_RESULT_SUCCESS;
}

nn::Result DirectoryDownloader::DownloadFile(const ListFile& file, nn::nifm::NetworkConnection* pConnection, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    if (m_StartDownloadFileCallback)
    {
        m_StartDownloadFileCallback(m_DirName, file.name, file.size, m_StartDownloadFileCallbackParam);
    }

    NN_DETAIL_BCAT_INFO("[bcat] Download file(%s/%s) ...\n", m_DirName.value, file.name.value);

    char dlFilePath[160] = {};
    MakeDownloadFilePath(dlFilePath, sizeof (dlFilePath), m_AppId, m_DirName.value, file.name.value);

    char dlMetaFilePath[160] = {};
    MakeDownloadMetaFilePath(dlMetaFilePath, sizeof (dlMetaFilePath), m_AppId, m_DirName.value, file.name.value);

    ProgressCallbackParam callbackParam = {pConnection, m_pCancelEvent, m_DownloadFileCallback, m_DownloadFileCallbackParam};

    m_ArchiveDownloader.Reset();
    m_ArchiveDownloader.SetPath(dlFilePath, dlMetaFilePath);
    m_ArchiveDownloader.SetPassphrase(m_AppId, m_Passphrase);
    m_ArchiveDownloader.SetProgressCallback(ProgressCallback, &callbackParam);
    m_ArchiveDownloader.SetJournalSize(DownloadStorageJournalSize);
    m_ArchiveDownloader.SetFileWorkBuffer(buffer, bufferSize);

    NN_DETAIL_BCAT_INFO("[bcat] Url = %s\n", file.url.value);

    NN_RESULT_TRY(m_ArchiveDownloader.Download(file.url.value, m_pCancelEvent))
        NN_RESULT_CATCH(ResultServerError404)
        {
            NN_DETAIL_BCAT_WARN("[bcat] The file is not found. (status 404)\n");
            NN_RESULT_RETHROW;
        }
        NN_RESULT_CATCH(ResultServerError410)
        {
            NN_DETAIL_BCAT_WARN("[bcat] Service expired. (status 410)\n");
            NN_RESULT_THROW(ResultPlatformServiceExpired());
        }
        NN_RESULT_CATCH(ResultDataVerificationFailed)
        {
            NN_DETAIL_BCAT_WARN("[bcat] The file verification failed.\n");
            NN_RESULT_RETHROW;
        }
        NN_RESULT_CATCH(nn::fs::ResultDataCorrupted)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(NN_RESULT_CURRENT_RESULT);
        }
        NN_RESULT_CATCH_ALL
        {
            // ダウンロード済みのデータを確定させる。
            NN_ABORT_UNLESS_RESULT_SUCCESS(FileSystem::Commit(NN_DETAIL_BCAT_DOWNLOAD_MOUNT_NAME));

            NN_RESULT_THROW_UNLESS(!IsCanceled(), ResultCanceledByUser());
            NN_RESULT_THROW_UNLESS(pConnection->IsAvailable(), ResultInternetRequestCanceled());
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY;

    NN_ABORT_UNLESS_RESULT_SUCCESS(FileSystem::Commit(NN_DETAIL_BCAT_DOWNLOAD_MOUNT_NAME));

    char filePath[160] = {};
    MakeFilePath(filePath, sizeof (filePath), m_AppId, m_DirName.value, file.name.value);

    NN_ABORT_UNLESS_RESULT_SUCCESS(FileSystem::RenameFile(dlFilePath, filePath));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::DeleteFile(dlMetaFilePath));

    FileMeta meta = {};

    meta.name = file.name;
    meta.dataId = file.dataId;
    meta.size = file.size;
    meta.digest = file.digest;

    NN_RESULT_DO(m_DownloadFileMetaView.AddOrUpdate(meta));
    NN_ABORT_UNLESS_RESULT_SUCCESS(FileMetaView::SaveToDownloadStorage(m_DownloadFileMetaView, m_AppId, m_DirName.value, false));

    NN_ABORT_UNLESS_RESULT_SUCCESS(FileSystem::Commit(NN_DETAIL_BCAT_DOWNLOAD_MOUNT_NAME));

    NN_RESULT_THROW_UNLESS(!IsCanceled(), ResultCanceledByUser());

    NN_RESULT_SUCCESS;
}

bool DirectoryDownloader::IsCanceled() NN_NOEXCEPT
{
    return (m_pCancelEvent && m_pCancelEvent->TryWait());
}

}}}}}
