﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/detail/service/core/bcat_DeliveryCacheStorageManager.h>
#include <nn/bcat/detail/service/core/bcat_PassphraseManager.h>
#include <nn/fs/fs_BcatSaveData.h>
#include <nn/fs/fs_FileSystemPrivate.h>

namespace nn { namespace bcat { namespace detail { namespace service { namespace core {

namespace
{
    void MakeBcatSaveMountName(char* name, size_t size, int index) NN_NOEXCEPT
    {
        nn::util::SNPrintf(name, size, "bcat-dc-%02d", index);
    }

    nn::Result MountBcatSaveData(const char* name, nn::ApplicationId appId) NN_NOEXCEPT
    {
        nn::ncm::ApplicationId ncmAppId = {appId.value};

        NN_RESULT_TRY(nn::fs::MountBcatSaveData(name, ncmAppId))
            NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
            {
                NN_DETAIL_BCAT_INFO("[bcat] BCAT savedata (%016llx) is not created yet.\n", appId.value);
                NN_RESULT_THROW(ResultStorageNotFound());
            }
        NN_RESULT_END_TRY;

        NN_RESULT_SUCCESS;
    }
}

DeliveryCacheStorageManager::DeliveryCacheStorageManager() NN_NOEXCEPT :
    m_Mutex(true),
    m_IsTestModeEnabled(false)
{
    std::memset(m_Records, 0, sizeof (m_Records));
}

void DeliveryCacheStorageManager::EnableTestMode() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    m_IsTestModeEnabled = true;
}

nn::Result DeliveryCacheStorageManager::Mount(nn::ApplicationId appId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchRecord(appId);

    if (index == -1)
    {
        index = SearchEmptyRecord();
        NN_RESULT_THROW_UNLESS(index != -1, ResultMountLimitReached());
    }

    if (m_Records[index].referenceCount == 0)
    {
        char mountName[nn::fs::MountNameLengthMax + 1] = {};
        MakeBcatSaveMountName(mountName, sizeof (mountName), index);

        if (!m_IsTestModeEnabled)
        {
            NN_RESULT_DO(MountBcatSaveData(mountName, appId));
        }

        m_Records[index].appId = appId;
    }
    else
    {
        // MEMO: 複数のアプリケーションから同じアプリケーション ID のストレージのマウントを許可する場合、ロックエラーにしない。（バックグラウンド処理とは排他）
        NN_RESULT_THROW(ResultLocked());
    }

    m_Records[index].referenceCount++;

    NN_RESULT_SUCCESS;
}

void DeliveryCacheStorageManager::Unmount(nn::ApplicationId appId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchRecord(appId);

    NN_SDK_ASSERT(index != -1);
    NN_SDK_ASSERT(m_Records[index].referenceCount > 0);

    m_Records[index].referenceCount--;

    if (m_Records[index].referenceCount == 0)
    {
        char mountName[nn::fs::MountNameLengthMax + 1] = {};
        MakeBcatSaveMountName(mountName, sizeof (mountName), index);

        if (!m_IsTestModeEnabled)
        {
            nn::fs::Unmount(mountName);
        }

        m_Records[index].appId = nn::ApplicationId::GetInvalidId();

        PassphraseManager::GetInstance().RemoveTemporary(appId);
    }
}

nn::Result DeliveryCacheStorageManager::Commit(nn::ApplicationId appId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchRecord(appId);

    NN_SDK_ASSERT(index != -1);
    NN_SDK_ASSERT(m_Records[index].referenceCount > 0);

    char mountName[nn::fs::MountNameLengthMax + 1] = {};
    MakeBcatSaveMountName(mountName, sizeof (mountName), index);

    if (!m_IsTestModeEnabled)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::CommitSaveData(mountName));
    }

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryCacheStorageManager::RequestSavePassphrase(nn::ApplicationId appId, const char* passphrase) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());
    NN_SDK_REQUIRES_NOT_NULL(passphrase);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchRecord(appId);

    if (index != -1)
    {
        NN_ABORT_UNLESS(PassphraseManager::GetInstance().AddTemporary(appId, passphrase));
    }
    else
    {
        NN_RESULT_DO(Mount(appId));

        NN_UTIL_SCOPE_EXIT
        {
            Unmount(appId);
        };

        NN_RESULT_DO(PassphraseManager::GetInstance().Save(appId, passphrase));
    }

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryCacheStorageManager::GetFreeSpaceSize(int64_t* outSize, nn::ApplicationId appId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outSize);
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchRecord(appId);

    NN_SDK_ASSERT(index != -1);
    NN_SDK_ASSERT(m_Records[index].referenceCount > 0);

    char mountName[nn::fs::MountNameLengthMax + 1] = {};
    MakeBcatSaveMountName(mountName, sizeof (mountName), index);

    if (m_IsTestModeEnabled)
    {
        *outSize = DownloadStorageSize;
    }
    else
    {
        char root[nn::fs::MountNameLengthMax + 3] = {};
        nn::util::SNPrintf(root, sizeof (root), "%s:/", mountName);

        NN_RESULT_DO(nn::fs::GetFreeSpaceSize(outSize, root));
    }

    NN_RESULT_SUCCESS;
}

void DeliveryCacheStorageManager::MakePassphrasePath(char* path, size_t size, nn::ApplicationId appId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);
    NN_SDK_REQUIRES_GREATER(size, 0u);
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchRecord(appId);

    NN_SDK_ASSERT(index != -1);
    NN_SDK_ASSERT(m_Records[index].referenceCount > 0);

    char mountName[nn::fs::MountNameLengthMax + 1] = {};
    MakeBcatSaveMountName(mountName, sizeof (mountName), index);

    int length = nn::util::SNPrintf(path, size, "%s:/passphrase.bin", mountName);

    NN_UNUSED(length);
    NN_SDK_ASSERT(length < static_cast<int>(size));
}

void DeliveryCacheStorageManager::MakeListPath(char* path, size_t size, nn::ApplicationId appId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);
    NN_SDK_REQUIRES_GREATER(size, 0u);
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchRecord(appId);

    NN_SDK_ASSERT(index != -1);
    NN_SDK_ASSERT(m_Records[index].referenceCount > 0);

    char mountName[nn::fs::MountNameLengthMax + 1] = {};
    MakeBcatSaveMountName(mountName, sizeof (mountName), index);

    int length = nn::util::SNPrintf(path, size, "%s:/list.msgpack", mountName);

    NN_UNUSED(length);
    NN_SDK_ASSERT(length < static_cast<int>(size));
}

void DeliveryCacheStorageManager::MakeETagPath(char* path, size_t size, nn::ApplicationId appId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);
    NN_SDK_REQUIRES_GREATER(size, 0u);
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchRecord(appId);

    NN_SDK_ASSERT(index != -1);
    NN_SDK_ASSERT(m_Records[index].referenceCount > 0);

    char mountName[nn::fs::MountNameLengthMax + 1] = {};
    MakeBcatSaveMountName(mountName, sizeof (mountName), index);

    int length = nn::util::SNPrintf(path, size, "%s:/etag.bin", mountName);

    NN_UNUSED(length);
    NN_SDK_ASSERT(length < static_cast<int>(size));
}

void DeliveryCacheStorageManager::MakeNintendoAccountRequiredPath(char* path, size_t size, nn::ApplicationId appId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);
    NN_SDK_REQUIRES_GREATER(size, 0u);
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchRecord(appId);

    NN_SDK_ASSERT(index != -1);
    NN_SDK_ASSERT(m_Records[index].referenceCount > 0);

    char mountName[nn::fs::MountNameLengthMax + 1] = {};
    MakeBcatSaveMountName(mountName, sizeof (mountName), index);

    int length = nn::util::SNPrintf(path, size, "%s:/na_required", mountName);

    NN_UNUSED(length);
    NN_SDK_ASSERT(length < static_cast<int>(size));
}

void DeliveryCacheStorageManager::MakeTransactionPath(char* path, size_t size, nn::ApplicationId appId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);
    NN_SDK_REQUIRES_GREATER(size, 0u);
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchRecord(appId);

    NN_SDK_ASSERT(index != -1);
    NN_SDK_ASSERT(m_Records[index].referenceCount > 0);

    char mountName[nn::fs::MountNameLengthMax + 1] = {};
    MakeBcatSaveMountName(mountName, sizeof (mountName), index);

    int length = nn::util::SNPrintf(path, size, "%s:/index.lock", mountName);

    NN_UNUSED(length);
    NN_SDK_ASSERT(length < static_cast<int>(size));
}

void DeliveryCacheStorageManager::MakeFilePath(char* path, size_t size, nn::ApplicationId appId, const char* dirName, const char* fileName) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);
    NN_SDK_REQUIRES_GREATER(size, 0u);
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());
    NN_SDK_REQUIRES_NOT_NULL(dirName);
    NN_SDK_REQUIRES_NOT_NULL(fileName);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchRecord(appId);

    NN_SDK_ASSERT(index != -1);
    NN_SDK_ASSERT(m_Records[index].referenceCount > 0);

    char mountName[nn::fs::MountNameLengthMax + 1] = {};
    MakeBcatSaveMountName(mountName, sizeof (mountName), index);

    int length = nn::util::SNPrintf(path, size, "%s:/directories/%s/files/%s", mountName, dirName, fileName);

    NN_UNUSED(length);
    NN_SDK_ASSERT(length < static_cast<int>(size));
}

void DeliveryCacheStorageManager::MakeFileMetaPath(char* path, size_t size, nn::ApplicationId appId, const char* dirName) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);
    NN_SDK_REQUIRES_GREATER(size, 0u);
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());
    NN_SDK_REQUIRES_NOT_NULL(dirName);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchRecord(appId);

    NN_SDK_ASSERT(index != -1);
    NN_SDK_ASSERT(m_Records[index].referenceCount > 0);

    char mountName[nn::fs::MountNameLengthMax + 1] = {};
    MakeBcatSaveMountName(mountName, sizeof (mountName), index);

    int length = nn::util::SNPrintf(path, size, "%s:/directories/%s/files.meta", mountName, dirName);

    NN_UNUSED(length);
    NN_SDK_ASSERT(length < static_cast<int>(size));
}

void DeliveryCacheStorageManager::MakeRootDirectoryPath(char* path, size_t size, nn::ApplicationId appId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);
    NN_SDK_REQUIRES_GREATER(size, 0u);
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchRecord(appId);

    NN_SDK_ASSERT(index != -1);
    NN_SDK_ASSERT(m_Records[index].referenceCount > 0);

    char mountName[nn::fs::MountNameLengthMax + 1] = {};
    MakeBcatSaveMountName(mountName, sizeof (mountName), index);

    int length = nn::util::SNPrintf(path, size, "%s:/directories", mountName);

    NN_UNUSED(length);
    NN_SDK_ASSERT(length < static_cast<int>(size));
}

void DeliveryCacheStorageManager::MakeDirectoryPath(char* path, size_t size, nn::ApplicationId appId, const char* dirName) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);
    NN_SDK_REQUIRES_GREATER(size, 0u);
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());
    NN_SDK_REQUIRES_NOT_NULL(dirName);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchRecord(appId);

    NN_SDK_ASSERT(index != -1);
    NN_SDK_ASSERT(m_Records[index].referenceCount > 0);

    char mountName[nn::fs::MountNameLengthMax + 1] = {};
    MakeBcatSaveMountName(mountName, sizeof (mountName), index);

    int length = nn::util::SNPrintf(path, size, "%s:/directories/%s", mountName, dirName);

    NN_UNUSED(length);
    NN_SDK_ASSERT(length < static_cast<int>(size));
}

void DeliveryCacheStorageManager::MakeDirectoryMetaPath(char* path, size_t size, nn::ApplicationId appId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);
    NN_SDK_REQUIRES_GREATER(size, 0u);
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchRecord(appId);

    NN_SDK_ASSERT(index != -1);
    NN_SDK_ASSERT(m_Records[index].referenceCount > 0);

    char mountName[nn::fs::MountNameLengthMax + 1] = {};
    MakeBcatSaveMountName(mountName, sizeof (mountName), index);

    int length = nn::util::SNPrintf(path, size, "%s:/directories.meta", mountName);

    NN_UNUSED(length);
    NN_SDK_ASSERT(length < static_cast<int>(size));
}

int DeliveryCacheStorageManager::SearchRecord(nn::ApplicationId appId) NN_NOEXCEPT
{
    for (int i = 0; i < MountCountMax; i++)
    {
        if (m_Records[i].appId == appId)
        {
            return i;
        }
    }

    return -1;
}

int DeliveryCacheStorageManager::SearchEmptyRecord() NN_NOEXCEPT
{
    for (int i = 0; i < MountCountMax; i++)
    {
        if (m_Records[i].appId == nn::ApplicationId::GetInvalidId())
        {
            return i;
        }
    }

    return -1;
}

}}}}}
