﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/detail/service/bcat_DeliveryCacheDirectoryService.generic.h>
#include <nn/bcat/detail/service/bcat_DeliveryCacheStorageService.generic.h>
#include <nn/bcat/bcat_Result.h>
#include <nn/bcat/bcat_ResultPrivate.h>
#include <nn/bcat/detail/bcat_Log.h>
#include <mutex>
#include <nn/fs.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>

#define CHECK_IPC_STRING(value, maxSize) \
    do                                                              \
    {                                                               \
        size_t size = value.GetLength();                            \
        if (size == 0 || size > maxSize || value[size - 1] != '\0') \
        {                                                           \
            NN_RESULT_THROW(ResultInvalidArgument());               \
        }                                                           \
    }                                                               \
    while (NN_STATIC_CONDITION(false))

namespace nn { namespace bcat { namespace detail { namespace service {

DeliveryCacheDirectoryServiceForGeneric::DeliveryCacheDirectoryServiceForGeneric(nn::sf::SharedPointer<DeliveryCacheStorageServiceForGeneric>&& parent) NN_NOEXCEPT :
    m_Mutex(true),
    m_Parent(std::move(parent)),
    m_IsOpened(false)
{
}

DeliveryCacheDirectoryServiceForGeneric::~DeliveryCacheDirectoryServiceForGeneric() NN_NOEXCEPT
{
}

nn::Result DeliveryCacheDirectoryServiceForGeneric::Open(const nn::bcat::DirectoryName& dirName) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(dirName.IsValid(), ResultInvalidArgument());

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_THROW_UNLESS(!m_IsOpened, ResultAlreadyOpened());

    char path[64] = {};
    nn::util::SNPrintf(path, sizeof (path), "bcat-dc:/%s", dirName.value);

    nn::fs::DirectoryHandle handle = {};

    // 開くことができるかどうかの確認のみ。
    NN_RESULT_TRY(nn::fs::OpenDirectory(&handle, path, nn::fs::OpenDirectoryMode_File))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            NN_RESULT_THROW(ResultNotFound());
        }
    NN_RESULT_END_TRY;

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseDirectory(handle);
    };

    m_DirName = dirName;
    m_IsOpened = true;

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryCacheDirectoryServiceForGeneric::Read(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::bcat::DeliveryCacheDirectoryEntry>& outEntries) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_IsOpened, ResultNotOpened());

    char path[64] = {};
    nn::util::SNPrintf(path, sizeof (path), "bcat-dc:/%s", m_DirName.value);

    nn::fs::DirectoryHandle handle = {};

    NN_RESULT_DO(nn::fs::OpenDirectory(&handle, path, nn::fs::OpenDirectoryMode_File));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseDirectory(handle);
    };

    int actualCount = 0;

    while (actualCount < static_cast<int>(outEntries.GetLength()))
    {
        nn::fs::DirectoryEntry entry = {};
        int64_t count;

        NN_RESULT_DO(nn::fs::ReadDirectory(&count, &entry, handle, 1));

        if (count == 0)
        {
            break;
        }

        DeliveryCacheDirectoryEntry& bcatEntry = outEntries[actualCount++];

        nn::util::Strlcpy(bcatEntry.name.value, entry.name, sizeof (bcatEntry.name.value));
        bcatEntry.size = entry.fileSize;

        // ダイジェスト値はサポートしない。
        std::memset(bcatEntry.digest.value, 0, sizeof (bcatEntry.digest.value));
    }

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryCacheDirectoryServiceForGeneric::GetCount(nn::sf::Out<std::int32_t> outCount) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_IsOpened, ResultNotOpened());

    char path[64] = {};
    nn::util::SNPrintf(path, sizeof (path), "bcat-dc:/%s", m_DirName.value);

    nn::fs::DirectoryHandle handle = {};

    NN_RESULT_DO(nn::fs::OpenDirectory(&handle, path, nn::fs::OpenDirectoryMode_File));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseDirectory(handle);
    };

    int64_t count = 0;
    NN_RESULT_DO(nn::fs::GetDirectoryEntryCount(&count, handle));

    NN_RESULT_THROW_UNLESS(count <= DeliveryCacheFileCountMaxPerDirectory, ResultEntryLimitReached());

    *outCount = static_cast<int32_t>(count);

    NN_RESULT_SUCCESS;
}

}}}}
