﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm> // std::min, std::max
#include <nn/settings/system/settings_Audio.h>
#include <nn/os.h>
#include <nn/hid/system/hid_AudioControl.h>
#include <nn/hid/system/hid_InputDetection.h>
#include <nn/audioctrl/audioctrl_AudioControllerTypes.h>
#include "audioctrl_AudioControllerService.h"
#include "audioctrl_UserInputHandler.h"

namespace nn { namespace audio {
// For CodecController
nn::os::SystemEvent* RegisterJackEvent() NN_NOEXCEPT;
void UnregisterJackEvent() NN_NOEXCEPT;
void ClearJackEvent() NN_NOEXCEPT;
bool IsJackPlugged() NN_NOEXCEPT;
bool IsCodecRequested() NN_NOEXCEPT;
void SetSpeakerMute(bool mute) NN_NOEXCEPT;
void NotifyHeadphoneOutSelect(bool isEnable) NN_NOEXCEPT;
void NotifyHeadphoneInEnabled(bool isEnable) NN_NOEXCEPT;
nn::os::EventType* GetUacSpeakerAttachEvent() NN_NOEXCEPT;
nn::os::EventType* GetUacSpeakerDetachEvent() NN_NOEXCEPT;
nn::os::EventType* GetUacUnsupportedSpeakerAttachEvent() NN_NOEXCEPT;
int GetNumUacSpeakers() NN_NOEXCEPT;
void NotifyUacSpeakerSwitch(bool isEnable) NN_NOEXCEPT;
}}

#define NN_AUDIOCTRL_TRACE(...)
#if !defined(NN_AUDIOCTRL_TRACE)
#include <nn/audioctrl/detail/audioctrl_Log.h>
#include <nn/nn_TimeSpan.h>
#define NN_AUDIOCTRL_TRACE(format, ...) NN_DETAIL_STRUCTURED_SDK_LOG(audioctrl, Trace, 0, "[audioctrl:%06lld] " format, nn::os::ConvertToTimeSpan(nn::os::GetSystemTick()).GetMilliSeconds(), ##__VA_ARGS__)
#endif

namespace nn { namespace audioctrl { namespace server { namespace detail {

namespace {
const nn::TimeSpan ButtonRepeatFirstIntervalTime = nn::TimeSpan::FromMilliSeconds(300);
const nn::TimeSpan ButtonRepeatNormalIntervalTime = nn::TimeSpan::FromMilliSeconds(50);
const nn::TimeSpan HeadphoneJackDetectIgnoreTime = nn::TimeSpan::FromMilliSeconds(100);
const nn::TimeSpan VolumeSaveWaitTime = nn::TimeSpan::FromMilliSeconds(1000);
const nn::TimeSpan SwitchHeadphoneToSpeakerIgnoreTime = nn::TimeSpan::FromMilliSeconds(500);
} // anonymous namespace

UserInputHandler::UserInputHandler() NN_NOEXCEPT
    : m_ButtonRepeatTimerEvent(nn::os::EventClearMode_AutoClear),
      m_HeadphoneMicJackDetectTimerEvent(nn::os::EventClearMode_AutoClear),
      m_VolumeSettingsSaveTimerEvent(nn::os::EventClearMode_AutoClear),
      m_pHeadphoneMicJackDetectEvent(nullptr),
      m_HidAudioControlStateChangeEvent(),
      m_PreviousTarget(AudioTarget_Invalid),
      m_HidAudioControlStateLatestSamplingNumber(0)
{

}

void UserInputHandler::Initialize() NN_NOEXCEPT
{
    // GPIO ライブラリの初期化
    nn::gpio::Initialize();

    // HID オーディオコントロール (USB 音声出力機器の HID) を初期化とイベントのバインド
    nn::hid::system::InitializeAudioControl();
    nn::hid::system::BindAudioControlStateChangeEvent(&m_HidAudioControlStateChangeEvent, nn::os::EventClearMode_AutoClear);

    // 「両ボタン押し→片方のボタンを離す」の操作を受けたときにボリューム変更を行うため
    const auto VolumeButtonInterruptMode = nn::gpio::InterruptMode_AnyEdge;

    // ボリュームボタン GPIO ハンドラの初期化
    m_ButtonUpHandler.Initialize(nn::gpio::GpioPadName_ButtonVolUp, nn::gpio::Direction_Input, VolumeButtonInterruptMode);
    m_ButtonDownHandler.Initialize(nn::gpio::GpioPadName_ButtonVolDn, nn::gpio::Direction_Input, VolumeButtonInterruptMode);

    // ヘッドホンジャック挿抜イベントの登録と取得
    m_pHeadphoneMicJackDetectEvent = nn::audio::RegisterJackEvent();

    // 各種パラメータの初期化
    m_IsHeadphoneJackDetectionEnabled = true;
    m_IsVolumeButtonEnabled = true;
    m_PreviousTarget = GetCurrentTarget();

    // ハンドラースレッドの起動
    m_IsAwake = false;
    Wake();

//     // 無操作状態の解除のための InputDetector の初期化
//     //   - タイムスタンプが必要である場合のみ呼べばよい
//     //   - 別途 hid (client) の権限が必要になることに注意
//     nn::hid::system::InitializeInputDetector();
}

void UserInputHandler::Finalize() NN_NOEXCEPT
{
    // ハンドラースレッドの終了
    Sleep();

    // ヘッドホンジャック挿抜イベントの登録解除
    nn::audio::UnregisterJackEvent();

    // ボリュームボタン GPIO ハンドラの終了
    m_ButtonUpHandler.Finalize();
    m_ButtonDownHandler.Finalize();

    // GPIO ライブラリの終了
    nn::gpio::Finalize();
}

void UserInputHandler::Sleep() NN_NOEXCEPT
{
    if(m_IsAwake == false)
    {
        return;
    }

    m_IsAwake = false;
}

void UserInputHandler::Wake() NN_NOEXCEPT
{
    if(m_IsAwake)
    {
        return;
    }

    // スリープ中に出力先が切り替わったことをハンドリング
    const auto CurrentTarget = GetCurrentTarget();
    SwitchTarget(m_PreviousTarget, CurrentTarget);
    m_PreviousTarget = CurrentTarget;

    m_IsAwake = true;
}

bool UserInputHandler::IsHeadphoneMicJackInserted() NN_NOEXCEPT
{
    return nn::audio::IsJackPlugged();
}

void UserInputHandler::VolumeButtonHandler() NN_NOEXCEPT
{
    // ミュート時などに長押しボリュームアップが動かないように止める
    m_ButtonRepeatTimerEvent.Stop();
    m_ButtonRepeatTimerEvent.Clear();

    // イベントのクリア
    m_ButtonUpHandler.Clear();
    m_ButtonDownHandler.Clear();

    // 現在の出力先を取得
    const auto CurrentTarget = GetCurrentTarget();

    // スピーカー・ヘッドホン・USB 出力機器の時のみ処理を行う
    if(!(CurrentTarget == AudioTarget_Speaker || CurrentTarget == AudioTarget_Headphone || CurrentTarget == AudioTarget_UsbOutputDevice))
    {
        return;
    }

    // ボリュームボタンの状態を取得
    const auto Up = m_ButtonUpHandler.IsLow();
    const auto Down = m_ButtonDownHandler.IsLow();

    // 現在のミュート状態とボリューム値を取得
    const auto CurrentMute = nn::audioctrl::server::detail::IsTargetMute(CurrentTarget);
    const auto CurrentVolume = nn::audioctrl::server::detail::GetTargetVolume(CurrentTarget);

    // ボリュームボタンのどちらも押されていない時はなにもしない
    if(Up == false && Down == false)
    {
        return;
    }

    // ボリュームボタン両押しかつ非ミュート時はなにもしない
    if(Up && Down && CurrentMute == false)
    {
        return;
    }

    int8_t nextVolume = CurrentVolume;

    if(CurrentMute) // ミュート解除時
    {
        NN_AUDIOCTRL_TRACE("Unmute.\n");
    }
    else // ボリューム変更時
    {
        if (Up)
        {
            nextVolume = static_cast<int8_t>(std::min(CurrentVolume + 1, GetTargetVolumeMax()));
        }
        else if (Down)
        {
            nextVolume = static_cast<int8_t>(std::max(CurrentVolume - 1, GetTargetVolumeMin()));
        }

        NN_AUDIOCTRL_TRACE("Volume is changed.\n");
    }

    // 最大・最小音量に達しているかをチェック
    if(!(Up && nextVolume == nn::audioctrl::server::detail::GetTargetVolumeMax()) &&
        !(Down && nextVolume == nn::audioctrl::server::detail::GetTargetVolumeMin()))
    {
        // ボリュームボタンのリピートタイマーを開始
        m_ButtonRepeatTimerEvent.StartPeriodic(ButtonRepeatFirstIntervalTime, ButtonRepeatNormalIntervalTime);

        NN_AUDIOCTRL_TRACE("Start volume repeat timer.\n");
    }

    // ボリュームボタンを押した時は常にミュート解除状態
    const auto NextMute = false;

    // ミュート設定
    nn::audioctrl::server::detail::SetTargetMute(CurrentTarget, NextMute);

    // ボリューム設定
    nn::audioctrl::server::detail::SetTargetVolume(CurrentTarget, static_cast<int8_t>(nextVolume));

    // オーバレイ通知を送信
    // 音量変更のオーバレイ通知
    OverlaySenderController::VolumeChangeDetailReasonFlagSet detailReason = {};

    detailReason.Set<OverlaySenderController::VolumeChangeDetailReasonFlag::upAndDownButtonPressed>(Up && Down && NextMute);
    detailReason.Set<OverlaySenderController::VolumeChangeDetailReasonFlag::upButtonPressed>(Up && !NextMute);
    detailReason.Set<OverlaySenderController::VolumeChangeDetailReasonFlag::downButtonPressed>(Down && !NextMute);

    const auto Reason = (CurrentMute != NextMute) ?
                          OverlaySenderController::VolumeChangeReason::MuteChanged
                        : OverlaySenderController::VolumeChangeReason::VolumeChanged;

    auto& overlaySenderController = nn::audioctrl::server::detail::GetOverlaySenderController();
    overlaySenderController.SendVolumeChangeMessage(nextVolume, NextMute, CurrentTarget, Reason, detailReason);

    // ヘッドホン音量警告のオーバレイ通知
    if( CurrentTarget == AudioTarget_Headphone && IsHeadphoneVolumeWarningTransmissionNeeded() )
    {
        overlaySenderController.SendHeadphoneVolumeWarningNoticeMessage();
    }

    // ボリューム設定保存のタイマーを開始
    m_VolumeSettingsSaveTimerEvent.StartOneShot(VolumeSaveWaitTime);

    // 無操作状態の解除のための通知
    nn::hid::system::NotifyInputDetector(nn::hid::system::InputSourceId::Pad::Mask);
}

void UserInputHandler::VolumeButtonTimerEventHandler() NN_NOEXCEPT
{
    // イベントをクリア
    m_ButtonRepeatTimerEvent.Clear();

    // ボリュームボタンの状態を取得
    const auto Up = m_ButtonUpHandler.IsLow();
    const auto Down = m_ButtonDownHandler.IsLow();

    // どちらのボリュームボタンも押されていなければ止める
    if(Up == false && Down == false)
    {
        m_ButtonRepeatTimerEvent.Stop();
        NN_AUDIOCTRL_TRACE("Stop button repeat timer.\n");
    }
    else
    {
        const auto CurrentTarget = GetCurrentTarget();
        // スピーカー・ヘッドホン・USB 出力機器の時のみ処理を行う
        if(!(CurrentTarget == AudioTarget_Speaker || CurrentTarget == AudioTarget_Headphone || CurrentTarget == AudioTarget_UsbOutputDevice))
        {
            return;
        }

        int8_t volume = nn::audioctrl::server::detail::GetTargetVolume(CurrentTarget);

        if (Up)
        {
            volume = static_cast<int8_t>(std::min(volume + 1, GetTargetVolumeMax()));
        }
        else if (Down)
        {
            volume = static_cast<int8_t>(std::max(volume - 1, GetTargetVolumeMin()));
        }

        // ボリューム値が最大・最小値になった時にボタンリピートを止める
        if(volume == nn::audioctrl::server::detail::GetTargetVolumeMax() || volume == nn::audioctrl::server::detail::GetTargetVolumeMin())
        {
            m_ButtonRepeatTimerEvent.Stop();
        }

        // ボリューム設定
        nn::audioctrl::server::detail::SetTargetVolume(CurrentTarget, static_cast<int8_t>(volume));

        // オーバレイ通知を送信
        // 音量変更のオーバレイ通知
        OverlaySenderController::VolumeChangeDetailReasonFlagSet detailReason = {};
        detailReason.Set<OverlaySenderController::VolumeChangeDetailReasonFlag::upButtonPressed>(Up);
        detailReason.Set<OverlaySenderController::VolumeChangeDetailReasonFlag::downButtonPressed>(Down);

        const auto mute = false;
        const auto reason =  OverlaySenderController::VolumeChangeReason::VolumeChanged;

        auto& overlaySenderController = nn::audioctrl::server::detail::GetOverlaySenderController();
        overlaySenderController.SendVolumeChangeMessage(volume, mute, CurrentTarget, reason, detailReason);

        // ヘッドホン音量警告のオーバレイ通知
        if( CurrentTarget == AudioTarget_Headphone && IsHeadphoneVolumeWarningTransmissionNeeded() )
        {
            overlaySenderController.SendHeadphoneVolumeWarningNoticeMessage();
        }

        // ボリューム設定保存のタイマーを開始
        m_VolumeSettingsSaveTimerEvent.StartOneShot(VolumeSaveWaitTime);

        NN_AUDIOCTRL_TRACE("Volume changed.\n");
    }
}

void UserInputHandler::HeadphoneJackDetectEventHandler() NN_NOEXCEPT
{
    // イベントをクリア
    m_pHeadphoneMicJackDetectEvent->Clear();

    // 挿抜の割り込みをクリア
    nn::audio::ClearJackEvent();

    // ヘッドホン挿抜検出が無効化されていた時は切り替え処理などを行わない
    if(m_IsHeadphoneJackDetectionEnabled == false)
    {
        return;
    }

    // 挿抜検出のタイマーを停止 (既にタイマーが発動している可能性がある)
    m_HeadphoneMicJackDetectTimerEvent.Stop();

    // 挿抜検出のタイマーを開始
    m_HeadphoneMicJackDetectTimerEvent.StartOneShot(HeadphoneJackDetectIgnoreTime);

    NN_AUDIOCTRL_TRACE("Start headphone detect timer\n");
}

void UserInputHandler::HeadphoneJackDetectTimerEventHandler() NN_NOEXCEPT
{
    const auto HeadphoneMicInserted = IsHeadphoneMicJackInserted();
    const auto CurrentTarget = GetCurrentTarget();

    // イベントをクリア
    m_HeadphoneMicJackDetectTimerEvent.Clear();

    // ヘッドホン挿抜の検出を有効にする
    m_IsHeadphoneJackDetectionEnabled = true;

    // ボリュームボタンのリピートタイマーを止めておく
    // (ボリュームボタンを押されながらヘッドホンジャック挿抜を行われた時は、
    // 再度、押しなおさないとボリュームが変化しないようにする)
    m_ButtonRepeatTimerEvent.Stop();
    m_ButtonRepeatTimerEvent.Clear();

    // アナログマイク入力の有効・無効を設定
    nn::audio::NotifyHeadphoneInEnabled(HeadphoneMicInserted);

    // 現在の出力先と変更が無ければなにもしない
    if(CurrentTarget == m_PreviousTarget)
    {
        return;
    }

    // ジャック挿抜状態を通知
    nn::audioctrl::server::detail::SetTargetConnected(nn::audioctrl::AudioTarget_Headphone, HeadphoneMicInserted);

    SwitchTarget(m_PreviousTarget, CurrentTarget);
    m_PreviousTarget = CurrentTarget;

    // ヘッドホンジャック挿入時のチャタリング防止対応のために再度チェックを走らせる
    // ヘッドホンジャックが刺さった時は、SwitchHeadphoneToSpeakerIgnoreTime の間はスピーカへの遷移を無視する
    if(CurrentTarget == AudioTarget_Headphone)
    {
        // ヘッドホン挿抜の検出を無効にする
        m_IsHeadphoneJackDetectionEnabled = false;

        // 挿抜検出のタイマーを開始
        m_HeadphoneMicJackDetectTimerEvent.StartOneShot(SwitchHeadphoneToSpeakerIgnoreTime);
    }
}

void UserInputHandler::VolumeSettingsSaveTimerEventHandler() NN_NOEXCEPT
{
    // イベントをクリア
    m_VolumeSettingsSaveTimerEvent.Clear();

    // 音量設定を保存
    SaveTargetVolumeSetting(nn::audioctrl::AudioTarget_Headphone);
    SaveTargetVolumeSetting(nn::audioctrl::AudioTarget_Speaker);
    SaveTargetVolumeSetting(nn::audioctrl::AudioTarget_UsbOutputDevice);

    NN_AUDIOCTRL_TRACE("Volume settings is saved.\n");
}

void UserInputHandler::SetHeadphoneJackDetectionEnabled(bool isEnabled) NN_NOEXCEPT
{
    // 念の為、ヘッドホンジャック検出のタイマーを止めておく
    m_HeadphoneMicJackDetectTimerEvent.Stop();
    m_HeadphoneMicJackDetectTimerEvent.Clear();

    m_IsHeadphoneJackDetectionEnabled = isEnabled;
}

bool UserInputHandler::IsHeadphoneJackDetectionEnabled() const NN_NOEXCEPT
{
    return m_IsHeadphoneJackDetectionEnabled;
}

void UserInputHandler::SetVolumeButtonEnabled(bool isEnabled) NN_NOEXCEPT
{
    // 念の為、ボリュームボタンのリピートタイマーを止めておく
    m_ButtonRepeatTimerEvent.Stop();
    m_ButtonRepeatTimerEvent.Clear();

    m_IsVolumeButtonEnabled = isEnabled;
}

bool UserInputHandler::IsVolumeButtonEnabled() const NN_NOEXCEPT
{
    return m_IsVolumeButtonEnabled;
}

nn::os::SystemEventType& UserInputHandler::GetButtonUpEvent() NN_NOEXCEPT
{
    return *m_ButtonUpHandler.GetSystemEvent();
}

nn::os::SystemEventType& UserInputHandler::GetButtonDownEvent() NN_NOEXCEPT
{
    return *m_ButtonDownHandler.GetSystemEvent();
}

nn::os::TimerEvent& UserInputHandler::GetButtonRepeatTimerEvent() NN_NOEXCEPT
{
    return m_ButtonRepeatTimerEvent;
}

nn::os::SystemEvent& UserInputHandler::GetHeadpphoneJackDetect() NN_NOEXCEPT
{
    return *m_pHeadphoneMicJackDetectEvent;
}

nn::os::TimerEvent& UserInputHandler::GetHeadphoneJackDetectTimerEvent() NN_NOEXCEPT
{
    return m_HeadphoneMicJackDetectTimerEvent;
}

nn::os::TimerEvent& UserInputHandler::GetVolumeSettingsTimerEvent() NN_NOEXCEPT
{
    return m_VolumeSettingsSaveTimerEvent;
}

nn::os::EventType* UserInputHandler::GetUacSpeakerAttachEvent() NN_NOEXCEPT
{
    return nn::audio::GetUacSpeakerAttachEvent();
}

nn::os::EventType* UserInputHandler::GetUacSpeakerDetachEvent() NN_NOEXCEPT
{
    return nn::audio::GetUacSpeakerDetachEvent();
}

nn::os::EventType* UserInputHandler::GetUacUnsupportedSpeakerAttachEvent() NN_NOEXCEPT
{
    return nn::audio::GetUacUnsupportedSpeakerAttachEvent();
}

void UserInputHandler::UacSpeakerAttachHandler() NN_NOEXCEPT
{
    NN_AUDIOCTRL_TRACE("UacAttachHandler\n");
    nn::os::ClearEvent(GetUacSpeakerAttachEvent());

    const auto CurrentTarget = GetCurrentTarget();
    SwitchTarget(m_PreviousTarget, CurrentTarget);
    m_PreviousTarget = CurrentTarget;
}

AudioTarget UserInputHandler::GetCurrentTarget() NN_NOEXCEPT
{
    const auto HeadphoneMicInserted = IsHeadphoneMicJackInserted();
    const auto CodecDeviceRequested = nn::audio::IsCodecRequested();
    const auto UacSpeakerCount = nn::audio::GetNumUacSpeakers();

    // Priority:
    // USB Speaker > BulitinHeadphone > TV > BulitinSpeaker
    if(UacSpeakerCount > 0)
    {
        return AudioTarget_UsbOutputDevice;
    }
    else if(HeadphoneMicInserted)
    {
        return AudioTarget_Headphone;
    }
    else if(!CodecDeviceRequested)
    {
        return AudioTarget_Tv;
    }
    else
    {
        return AudioTarget_Speaker;
    }
}

void UserInputHandler::SwitchTarget(AudioTarget targetFrom, AudioTarget targetTo) NN_NOEXCEPT
{
    NN_AUDIOCTRL_TRACE("from: %d to: %d\n", targetFrom, targetTo);

    bool isSendOverlayNotifiacationNeeded = false;

    // 遷移元と遷移先が同じ時は何もしない
    if(targetFrom == targetTo)
    {
        return;
    }

    // 遷移元共通処理
    if(targetFrom == AudioTarget_Speaker)
    {
        // 何もしない
    }
    else if(targetFrom == AudioTarget_Headphone)
    {
        // 本体内蔵ヘッドホンジャックを使用しないことを通知
        nn::audio::NotifyHeadphoneOutSelect(false);
    }
    else if(targetFrom == AudioTarget_Tv)
    {
        // 何もしない
    }
    else if(targetFrom == AudioTarget_UsbOutputDevice)
    {
        // USB スピーカーを使わないことを通知
        nn::audio::NotifyUacSpeakerSwitch(false);
    }

    // 遷移先共通処理
    if(targetTo == AudioTarget_Speaker)
    {
        // 本体内蔵スピーカーを使用することを通知
        nn::audio::NotifyHeadphoneOutSelect(false);

        // オーバレイ通知送信フラグをオン
        isSendOverlayNotifiacationNeeded = true;
    }
    else if(targetTo == AudioTarget_Headphone)
    {
        // 本体内蔵ヘッドホンジャックを使用することを通知
        nn::audio::NotifyHeadphoneOutSelect(true);

        // オーバレイ通知送信フラグをオン
        isSendOverlayNotifiacationNeeded = true;
    }
    else if(targetTo == AudioTarget_Tv)
    {
        // 何もしない
    }
    else if(targetTo == AudioTarget_UsbOutputDevice)
    {
        // USB スピーカーを使うことを通知
        nn::audio::NotifyUacSpeakerSwitch(true);

        // オーバレイ通知送信フラグをオン
        isSendOverlayNotifiacationNeeded = true;
    }

    // 遷移元・遷移先の組み合わせごとの処理
    if(targetFrom == AudioTarget_Headphone && targetTo == AudioTarget_Speaker)
    {
        // ヘッドホンジャック抜けミュート
        nn::audioctrl::server::detail::SetTargetMute(AudioTarget_Speaker, nn::settings::system::IsForceMuteOnHeadphoneRemoved());
    }

    if(targetFrom == AudioTarget_Tv && targetTo == AudioTarget_Speaker)
    {
        // オーバレイ通知送信フラグをオフ
        isSendOverlayNotifiacationNeeded = false;
    }

    // オーバレイ通知
    if(isSendOverlayNotifiacationNeeded)
    {
        OverlaySenderController::VolumeChangeDetailReasonFlagSet detailReason = {};

        // ヘッドホンジャックが抜けて遷移する時
        if(targetFrom == AudioTarget_Headphone && (targetTo == AudioTarget_Speaker || targetTo == AudioTarget_Tv))
        {
            detailReason.Set<OverlaySenderController::VolumeChangeDetailReasonFlag::headphoneMicJackUnplugged>(true);
        }

        // ヘッドホンジャックが挿入されて遷移する時
        if((targetFrom == AudioTarget_Speaker ||  targetTo == AudioTarget_Tv) && targetTo == AudioTarget_Headphone)
        {
            detailReason.Set<OverlaySenderController::VolumeChangeDetailReasonFlag::headphoneMicJackPlugged>(true);
        }

        // USB スピーカーが切断されて遷移する時
        if(targetFrom == AudioTarget_UsbOutputDevice)
        {
            detailReason.Set<OverlaySenderController::VolumeChangeDetailReasonFlag::usbOutputDeviceDisconnected>(true);
        }

        // USB スピーカーが接続されて遷移する時
        if(targetTo == AudioTarget_UsbOutputDevice)
        {
            detailReason.Set<OverlaySenderController::VolumeChangeDetailReasonFlag::usbOutputDeviceConnected>(true);
        }

        // 遷移先のミュート設定を取得
        const auto Mute = nn::audioctrl::server::detail::IsTargetMute(targetTo);
        // 遷移先のボリューム設定を取得
        const int8_t Volume = nn::audioctrl::server::detail::GetTargetVolume(targetTo);

        const auto Reason =  OverlaySenderController::VolumeChangeReason::OutputTargetChanged;

        auto& overlaySenderController = nn::audioctrl::server::detail::GetOverlaySenderController();
        overlaySenderController.SendVolumeChangeMessage(Volume, Mute, targetTo, Reason, detailReason);
    }
}

void UserInputHandler::UacSpeakerDetachHandler() NN_NOEXCEPT
{
    nn::os::ClearEvent(GetUacSpeakerDetachEvent());
    NN_AUDIOCTRL_TRACE("UacDetachHandler\n");

    // ボリュームボタンのリピートタイマーを止めておく
    // (ボリュームボタンを押されながら USB スピーカー挿抜を行われた時は、
    // 再度、押しなおさないとボリュームが変化しないようにする)
    m_ButtonRepeatTimerEvent.Stop();
    m_ButtonRepeatTimerEvent.Clear();

    const auto CurrentTarget = GetCurrentTarget();
    SwitchTarget(m_PreviousTarget, CurrentTarget);
    m_PreviousTarget = CurrentTarget;
}

void UserInputHandler::UacUnsupportedSpeakerAttachHandler() NN_NOEXCEPT
{
    NN_AUDIOCTRL_TRACE("UacUnsupportedSpeakerAttachHandler\n");

    nn::os::ClearEvent(GetUacUnsupportedSpeakerAttachEvent());
    NotifyUnsupportedUsbOutputDeviceAttached();
}

void UserInputHandler::HidAudioControlStateChangeEventHandler() NN_NOEXCEPT
{
    NN_AUDIOCTRL_TRACE("HidAudioControlStateChangeEventHandler\n");
    nn::os::ClearSystemEvent(&m_HidAudioControlStateChangeEvent);

    // 出力先が USB スピーカー以外になっていた時はなにもしない
    const auto currentTarget = GetCurrentTarget();
    if(currentTarget != AudioTarget_UsbOutputDevice)
    {
        return;
    }

    nn::hid::system::AudioControlState audioControlStates[nn::hid::system::AudioControlStateCountMax];
    const auto availableStatesCount = nn::hid::system::GetAudioControlStates(audioControlStates, NN_ARRAY_SIZE(audioControlStates));

    // State を一個も取れなかったときは何もしない (基本的にはないはず)
    if(availableStatesCount <= 0)
    {
        return;
    }

    int volumeDiff = 0;
    bool up = false;
    bool down = false;
    for(int i = 0; i < availableStatesCount; ++i)
    {
        const auto& audioControlState = audioControlStates[i];
        const auto& audioControls = audioControlState.audioControls;

        if(m_HidAudioControlStateLatestSamplingNumber >= audioControlState.samplingNumber)
        {
            break;
        }

        if(audioControls.Test<nn::hid::system::AudioControl::VolumeIncrement>())
        {
            ++volumeDiff;
            up = true;
        }

        if(audioControls.Test<nn::hid::system::AudioControl::VolumeDecrement>())
        {
            --volumeDiff;
            down = true;
        }
    }

    // 最新の samplingNumber を保存
    const auto& audioControlState = audioControlStates[0];
    m_HidAudioControlStateLatestSamplingNumber = audioControlState.samplingNumber;

    // 音量が変化していない場合は何もしない
    if(volumeDiff == 0)
    {
        return;
    }

    const auto currentVolume = nn::audioctrl::server::detail::GetTargetVolume(currentTarget);
    const auto newVolume = std::max(std::min(currentVolume + volumeDiff, GetTargetVolumeMax()), GetTargetVolumeMin());

    // ボリューム設定
    nn::audioctrl::server::detail::SetTargetVolume(currentTarget, static_cast<int8_t>(newVolume));

    // 音量変更のオーバレイ通知を送信
    OverlaySenderController::VolumeChangeDetailReasonFlagSet detailReason = {};
    detailReason.Set<OverlaySenderController::VolumeChangeDetailReasonFlag::upButtonPressed>(up);
    detailReason.Set<OverlaySenderController::VolumeChangeDetailReasonFlag::downButtonPressed>(down);

    const auto mute = false;
    const auto reason =  OverlaySenderController::VolumeChangeReason::VolumeChanged;
    auto& overlaySenderController = nn::audioctrl::server::detail::GetOverlaySenderController();
    overlaySenderController.SendVolumeChangeMessage(newVolume, mute, currentTarget, reason, detailReason);

    // ボリューム設定保存のタイマーを開始
    m_VolumeSettingsSaveTimerEvent.StartOneShot(VolumeSaveWaitTime);
}

nn::os::SystemEventType& UserInputHandler::GetHidAudioControlChangeEvent() NN_NOEXCEPT
{
    return m_HidAudioControlStateChangeEvent;
}

}}}}  // namespace nn::audioctrl::server::detail
