﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/audioctrl.h>
#include <nn/audio.h>
#include <nn/settings/system/settings_Audio.h>

#include "audioctrl_AudioControllerService.h"
#include "audioctrl_AudioControllerImpl.h"

#include <nn/nn_SdkLog.h>
namespace nn { namespace audioctrl { namespace server { namespace detail {

namespace
{

} // anonymous namespace

AudioControllerImpl::AudioControllerImpl() NN_NOEXCEPT
{
}

AudioControllerImpl::~AudioControllerImpl() NN_NOEXCEPT
{
}

Result AudioControllerImpl::GetTargetVolume(int32_t target, nn::sf::Out<std::int32_t> outVolume) NN_NOEXCEPT
{
    *outVolume = nn::audioctrl::server::detail::GetTargetVolume(static_cast<AudioTarget>(target));
    NN_RESULT_SUCCESS;
}

Result AudioControllerImpl::SetTargetVolume(int32_t target, int32_t volume) NN_NOEXCEPT
{
    const auto audioTarget = static_cast<AudioTarget>(target);
    const auto audioVolume = static_cast<int8_t>(volume);

    nn::audioctrl::server::detail::SetTargetVolume(audioTarget, audioVolume);
    nn::audioctrl::server::detail::SaveTargetVolumeSetting(audioTarget);

    if( audioTarget == AudioTarget_Headphone && IsHeadphoneVolumeWarningTransmissionNeeded() )
    {
        GetOverlaySenderController().SendHeadphoneVolumeWarningNoticeMessage();
    }

    NN_RESULT_SUCCESS;
}

Result AudioControllerImpl::GetTargetVolumeMin(nn::sf::Out<std::int32_t> outVolume) NN_NOEXCEPT
{
    *outVolume = nn::audioctrl::server::detail::GetTargetVolumeMin();
    NN_RESULT_SUCCESS;
}

Result AudioControllerImpl::GetTargetVolumeMax(nn::sf::Out<std::int32_t> outVolume) NN_NOEXCEPT
{
    *outVolume = nn::audioctrl::server::detail::GetTargetVolumeMax();
    NN_RESULT_SUCCESS;
}

Result AudioControllerImpl::IsTargetMute(int32_t target, nn::sf::Out<bool> isMute) NN_NOEXCEPT
{
    *isMute = nn::audioctrl::server::detail::IsTargetMute(static_cast<AudioTarget>(target));
    NN_RESULT_SUCCESS;
}

Result AudioControllerImpl::SetTargetMute(int32_t target, bool isMute) NN_NOEXCEPT
{
    const auto audioTarget = static_cast<AudioTarget>(target);

    nn::audioctrl::server::detail::SetTargetMute(audioTarget, isMute);
    nn::audioctrl::server::detail::SaveTargetVolumeSetting(audioTarget);

    if( audioTarget == AudioTarget_Headphone && IsHeadphoneVolumeWarningTransmissionNeeded() )
    {
        GetOverlaySenderController().SendHeadphoneVolumeWarningNoticeMessage();
    }

    NN_RESULT_SUCCESS;
}

Result AudioControllerImpl::IsTargetConnected(int32_t target, nn::sf::Out<bool> isConnected) NN_NOEXCEPT
{
    *isConnected = nn::audioctrl::server::detail::IsTargetConnected(static_cast<AudioTarget>(target));
    NN_RESULT_SUCCESS;
}

Result AudioControllerImpl::SetDefaultTarget(std::uint32_t target, nn::TimeSpanType fadeOutTime, nn::TimeSpanType fadeInTime) NN_NOEXCEPT
{
    if (!nn::audioctrl::server::detail::SetDefaultTarget(static_cast<AudioTarget>(target), fadeOutTime, fadeInTime))
    {
        // TODO: enable this line after omm handle error cases
//         NN_RESULT_THROW(ResultNotFound());
    }
    NN_RESULT_SUCCESS;
}

Result AudioControllerImpl::GetDefaultTarget(nn::sf::Out<std::int32_t> defaultTarget) NN_NOEXCEPT
{
    *defaultTarget = nn::audioctrl::server::detail::GetDefaultTarget();
    NN_RESULT_SUCCESS;
}

Result AudioControllerImpl::GetOutputModeSetting(nn::sf::Out<std::uint32_t> mode, uint32_t target) NN_NOEXCEPT
{
    *mode = nn::audioctrl::server::detail::GetOutputModeSetting(static_cast<AudioTarget>(target));
    NN_RESULT_SUCCESS;
}

Result AudioControllerImpl::SetOutputModeSetting(uint32_t target, uint32_t mode) NN_NOEXCEPT
{
    nn::audioctrl::server::detail::SetOutputModeSetting(static_cast<AudioTarget>(target), static_cast<AudioOutputMode>(mode));
    NN_RESULT_SUCCESS;
}

Result AudioControllerImpl::GetAudioOutputMode(nn::sf::Out<std::uint32_t> mode, uint32_t target) NN_NOEXCEPT
{
    *mode = nn::audioctrl::server::detail::GetAudioOutputMode(static_cast<AudioTarget>(target));
    NN_RESULT_SUCCESS;
}

Result AudioControllerImpl::SetAudioOutputMode(uint32_t target, uint32_t mode) NN_NOEXCEPT
{
    nn::audioctrl::server::detail::SetAudioOutputMode(static_cast<AudioTarget>(target), static_cast<AudioOutputMode>(mode));
    NN_RESULT_SUCCESS;
}

Result AudioControllerImpl::SetForceMutePolicy(std::int32_t policy) NN_NOEXCEPT
{
    nn::audioctrl::server::detail::SetForceMutePolicy(static_cast<ForceMutePolicy>(policy));
    NN_RESULT_SUCCESS;
}

Result AudioControllerImpl::GetForceMutePolicy(nn::sf::Out<std::int32_t> policy) NN_NOEXCEPT
{
    *policy = nn::audioctrl::server::detail::GetForceMutePolicy();
    NN_RESULT_SUCCESS;
}

Result AudioControllerImpl::SetOutputTarget(std::uint32_t target) NN_NOEXCEPT
{
    nn::audioctrl::server::detail::SetOutputTarget(static_cast<AudioTarget>(target));
    NN_RESULT_SUCCESS;
}

Result AudioControllerImpl::SetInputTargetForceEnabled(bool isEnabled) NN_NOEXCEPT
{
    nn::audioctrl::server::detail::SetInputTargetForceEnabled(isEnabled);
    NN_RESULT_SUCCESS;
}

Result AudioControllerImpl::SetHeadphoneOutputLevelMode(std::uint32_t mode) NN_NOEXCEPT
{
    nn::audioctrl::server::detail::SetHeadphoneOutputLevelMode(static_cast<nn::audioctrl::HeadphoneOutputLevelMode>(mode));
    NN_RESULT_SUCCESS;
}

Result AudioControllerImpl::GetHeadphoneOutputLevelMode(nn::sf::Out<std::uint32_t> mode) NN_NOEXCEPT
{
    *mode = nn::audioctrl::server::detail::GetHeadphoneOutputLevelMode();
    NN_RESULT_SUCCESS;
}

Result AudioControllerImpl::AcquireAudioVolumeUpdateEventForPlayReport(nn::sf::Out<nn::sf::NativeHandle> pOutHandle) NN_NOEXCEPT
{
    nn::os::NativeHandle handle;
    nn::audioctrl::server::detail::AcquireAudioVolumeUpdateEventForPlayReport(&handle);
    pOutHandle.Set(nn::sf::NativeHandle(handle, false));

    NN_RESULT_SUCCESS;
}

Result AudioControllerImpl::AcquireAudioOutputDeviceUpdateEventForPlayReport(nn::sf::Out<nn::sf::NativeHandle> pOutHandle) NN_NOEXCEPT
{
    nn::os::NativeHandle handle;
    nn::audioctrl::server::detail::AcquireAudioOutputDeviceUpdateEventForPlayReport(&handle);
    pOutHandle.Set(nn::sf::NativeHandle(handle, false));

    NN_RESULT_SUCCESS;
}

Result AudioControllerImpl::GetAudioOutputTargetForPlayReport(nn::sf::Out<std::uint32_t> target) NN_NOEXCEPT
{
    *target = static_cast<uint32_t>(nn::audioctrl::server::detail::GetAudioOutputTargetForPlayReport());
    NN_RESULT_SUCCESS;
}

Result AudioControllerImpl::NotifyHeadphoneVolumeWarningDisplayedEvent() NN_NOEXCEPT
{
    nn::audioctrl::server::detail::NotifyHeadphoneVolumeWarningDisplayedEvent();
    NN_RESULT_SUCCESS;
}

Result AudioControllerImpl::SetSystemOutputMasterVolume(float volume) NN_NOEXCEPT
{
    nn::audioctrl::server::detail::SetSystemOutputMasterVolume(volume);
    NN_RESULT_SUCCESS;
}

Result AudioControllerImpl::GetSystemOutputMasterVolume(nn::sf::Out<float> volume) NN_NOEXCEPT
{
    *volume = nn::audioctrl::server::detail::GetSystemOutputMasterVolume();
    NN_RESULT_SUCCESS;
}

Result AudioControllerImpl::GetAudioVolumeDataForPlayReport(nn::sf::Out<::nn::audioctrl::PlayReportAudioVolumeData> data) NN_NOEXCEPT
{
    PlayReportAudioVolumeData tmp;
    nn::audioctrl::server::detail::GetAudioVolumeDataForPlayReport(&tmp);
    *data = tmp;
    NN_RESULT_SUCCESS;
}

Result AudioControllerImpl::UpdateHeadphoneSettings(bool isRestricted) NN_NOEXCEPT
{
    auto isHighPowerAvailable = bool();
    auto volume = int8_t();

    if (nn::audioctrl::server::detail::NeedsToUpdateHeadphoneVolume(
            &isHighPowerAvailable, &volume, isRestricted))
    {
        nn::audioctrl::server::detail::SetTargetVolume(
            audioctrl::AudioTarget::AudioTarget_Headphone, volume);

        if (isHighPowerAvailable)
        {
            nn::audioctrl::server::detail::SetHeadphoneOutputLevelMode(
                audioctrl::HeadphoneOutputLevelMode_HighPower);
        }
    }

    NN_RESULT_SUCCESS;
}

}}}}  // namespace nn::audioctrl::server::server::detail
