﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os/os_Thread.h>
#include "audioctrl_UserInputHandler.h"
#include "audioctrl_PlayReportController.h"

namespace nn { namespace audioctrl { namespace server { namespace detail {

// 非同期処理コントローラ
class AsyncTaskController
{
    NN_DISALLOW_COPY( AsyncTaskController );
    NN_DISALLOW_MOVE( AsyncTaskController );

protected:
    UserInputHandler& m_UserInputHandler;
    PlayReportController& m_PlayReportController;
    bool m_HasCodecIc;
    bool m_HasHdmi;
    bool m_IsUacEnabled;
    bool m_IsAwake;
    bool m_IsInitialized;
    nn::os::Mutex m_Mutex;
    nn::os::Event m_ExitEvent;
    nn::os::ThreadType m_Thread;
    NN_OS_ALIGNAS_THREAD_STACK int8_t m_ThreadStack[12 * 1024];

protected:
    void ThreadFunctionImpl() NN_NOEXCEPT;
    static void ThreadFunction(void* arg) NN_NOEXCEPT;

public:
    /**
    * @brief       コンストラクタ
    * @pre         なし
    */
    AsyncTaskController(UserInputHandler& userInputHandler, PlayReportController& playReportController) NN_NOEXCEPT;

    /**
    * @brief       非同期処理コントローラの初期化
    * @param[in]   hasCodecIc コーデック IC が存在するプラットフォームか
    * @param[in]   hasHdmi HDMI が存在するプラットフォームか
    * @param[in]   isUacEnabled USB Audio Class 機能が有効か
    * @pre
    *  - コンストラクタで渡された UserInputHandler と PlayReportController は初期化済み
    *  - 非同期処理コントローラは未初期化
    * @post
    *  - 非同期処理コントローラは初期化済み
    * @details
    * コントローラは初期化カウンタを持たないので、多重に初期化することはできません。
    * 本 API はスレッドセーフです。
    */
    void Initialize(bool hasCodecIc, bool hasHdmi, bool isUacEnabled) NN_NOEXCEPT;

    /**
    * @brief       非同期処理コントローラの終了
    * @pre
    *  - 非同期処理コントローラは初期化済み
    * @post
    *  - 非同期処理コントローラは未初期化
    * @details
    * 本 API はスレッドセーフです。
    */
    void Finalize() NN_NOEXCEPT;

    /**
    * @brief       スリープ処理の実行
    * @pre
    *  - 非同期処理コントローラは初期化済み
    * @post
    *  - 非同期処理コントローラはスリープ状態
    * @details
    * スリープ状態時に本 API を呼び出した場合、何もしません。
    * コントローラは内部で本 API が呼び出された回数を保持していません。
    * そのため、本 API を二回以上呼んだ場合でも、Wake() を呼んだ場合は即座にウェイク状態になります。
    * 本 API はスレッドセーフです。
    */
    void Sleep() NN_NOEXCEPT;

    /**
    * @brief       ウェイク処理の実行
    * @pre
    *  - 非同期処理コントローラは初期化済み
    * @post
    *  - 非同期処理コントローラはウェイク状態
    * @details
    * ウェイク状態時に本 API を呼び出した場合、何もしません。
    * コントローラは内部で本 API が呼び出された回数を保持していません。
    * そのため、本 API を二回以上呼んだ場合でも、Sleep() を呼んだ場合は即座にスリープ状態になります。
    * 本 API はスレッドセーフです。
    */
    void Wake() NN_NOEXCEPT;
};

}}}}  // namespace nn::audioctrl::server::detail
