﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/dd/dd_DeviceAddressSpaceApi.h>
#include <nn/audio/audio_Profiler.h>
#include <nn/audio/detail/audio_Log.h>

#include "audio_CreateAudioDebugManager.h"

namespace nn {
namespace audio {

#if defined(NN_AUDIO_ENABLE_PROFILER)
namespace
{
#define NN_AUDIO_USE_SYSTEM_PROCESS
    nn::sf::SharedPointer<detail::IAudioDebugManager> CreateAudioDebugManager() NN_NOEXCEPT
    {
    #if defined(NN_BUILD_CONFIG_ADDRESS_64) && !defined(NN_AUDIO_USE_SYSTEM_PROCESS)
    #error "nn_audio with DFC mode can't be used on 64 bit environment currently."
    #endif

    #if defined(NN_BUILD_CONFIG_OS_HORIZON)
    #if defined(NN_AUDIO_USE_SYSTEM_PROCESS)
        return CreateAudioDebugManagerByHipc();
    #else
        // DFC に切り替えれば audio process なしでも動作可能
        return CreateAudioDebugManagerByDfc();
    #endif // #if defined(NN_AUDIO_USE_SYSTEM_PROCESS)
    #else
        return CreateAudioDebugManagerByDfc();
    #endif
    }
}
#endif //#if defined(NN_AUDIO_ENABLE_PROFILER)

void StartDspProfiler(DspProfilerParameter* params) NN_NOEXCEPT
{
#if defined(NN_AUDIO_ENABLE_PROFILER)
    NN_SDK_REQUIRES_GREATER(params->numSamples, 0);
    NN_SDK_REQUIRES_NOT_NULL(params->buffer);
    //NN_SDK_REQUIRES(params->sampleFrequencyUsec > 100); //How big does this need to be? Can't halt adsp
    DspProfilerHeader* header = reinterpret_cast<DspProfilerHeader*>(params->buffer);
    header->sampleFrequency = params->sampleFrequencyUsec;
    header->maxSamples = params->numSamples;
    header->curSample = 0;
    uint32_t bufferSize = GetBufferSizeForDspProfiler(params);

    auto processHandle = nn::os::InvalidNativeHandle;

    #if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
        processHandle = nn::dd::GetCurrentProcessHandle();
        NN_ABORT_UNLESS(processHandle != nn::os::InvalidNativeHandle);
    #endif

    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateAudioDebugManager()->StartDspProfiler(nn::sf::NativeHandle(processHandle, false), reinterpret_cast<uint64_t>(header), bufferSize));
#else
    NN_UNUSED( params );
    NN_DETAIL_AUDIO_WARN("Audio Debug server is not enabled. Make sure to set \"NN_AUDIO_ENABLE_PROFILER=1\" and rebuild audio library\n");
    NN_SDK_ASSERT(0);
#endif // #if defined(NN_AUDIO_ENABLE_PROFILER)
}

void StopDspProfiler() NN_NOEXCEPT
{
#if defined(NN_AUDIO_ENABLE_PROFILER)
    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateAudioDebugManager()->StopDspProfiler());
#else
    NN_DETAIL_AUDIO_WARN("Audio Debug server is not enabled. Make sure to set \"NN_AUDIO_ENABLE_PROFILER=1\" and rebuild audio library\n");
    NN_SDK_ASSERT(0);
#endif// #if defined(NN_AUDIO_ENABLE_PROFILER)
}

uint32_t GetBufferSizeForDspProfiler(DspProfilerParameter* params) NN_NOEXCEPT
{
    return static_cast<uint32_t>(sizeof(DspProfilerHeader) + (params->numSamples * sizeof(DspProfilerSample)));
}

void StartCpuProfiler() NN_NOEXCEPT
{
#if defined(NN_AUDIO_ENABLE_PROFILER)
    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateAudioDebugManager()->StartCpuProfiler());
#else
    NN_DETAIL_AUDIO_WARN("Audio Debug server is not enabled. Make sure to set \"NN_AUDIO_ENABLE_PROFILER=1\" and rebuild audio library\n");
    NN_SDK_ASSERT(0);
#endif// #if defined(NN_ENABLE_PROFILER)

}

void StopCpuProfiler() NN_NOEXCEPT
{
#if defined(NN_AUDIO_ENABLE_PROFILER)
    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateAudioDebugManager()->StopCpuProfiler());
#else
    NN_DETAIL_AUDIO_WARN("Audio Debug server is not enabled. Make sure to set \"NN_AUDIO_ENABLE_PROFILER=1\" and rebuild audio library\n");
    NN_SDK_ASSERT(0);
#endif// #if defined(NN_ENABLE_PROFILER)
}

}}


