﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/dd/dd_DeviceAddressSpace.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_Types.h>
#include <nn/applet/applet_Apis.h>
#include <nn/audio/audio_FinalOutputRecorder.h>
#include <nn/audio/audio_AudioInTypes.h>
#include "audio_CreateFinalOutputRecorderManager.h"
namespace nn{
namespace audio {

namespace {

static const nn::applet::AppletResourceUserId RecordAll = { 0xFFFFFFFFFFFFFFFFllu };
class FinalOutputRecorderMap
{
public:
    FinalOutputRecorderMap()
    : m_totalUsed(0)
    {
        os::InitializeMutex(&m_mutex, true, 0);
        std::memset(m_finalOutputRecorderPairs, 0, sizeof(m_finalOutputRecorderPairs));
    }

    ~FinalOutputRecorderMap()
    {
        os::FinalizeMutex(&m_mutex);
    }

    void AddPair(const FinalOutputRecorder* key, void* handle) NN_NOEXCEPT
    {
        os::LockMutex(&m_mutex);
        NN_SDK_ASSERT(m_totalUsed!=nn::audio::FinalOutputRecorderCountMax,"Opening too many FinalOutputRecorder sessions, can only open %d\n", nn::audio::FinalOutputRecorderCountMax);

        for(int i = 0; i < nn::audio::FinalOutputRecorderCountMax; ++i)
        {
            if(m_finalOutputRecorderPairs[i].pFinalOutputRecorderKey == nullptr)
            {
                NN_SDK_ASSERT( m_finalOutputRecorderPairs[i].pFinalOutputRecorderHandle == nullptr );
                m_finalOutputRecorderPairs[i].pFinalOutputRecorderKey = key;
                m_finalOutputRecorderPairs[i].pFinalOutputRecorderHandle = handle;
                ++m_totalUsed;
                break;
            }
        }
        os::UnlockMutex(&m_mutex);
    }

    void RemovePair(const FinalOutputRecorder* key) NN_NOEXCEPT
    {
        os::LockMutex(&m_mutex);
        for( int i = 0; i < nn::audio::FinalOutputRecorderCountMax; ++i )
        {
            if( m_finalOutputRecorderPairs[i].pFinalOutputRecorderKey == key )
            {
                m_finalOutputRecorderPairs[i].pFinalOutputRecorderKey = nullptr;
                m_finalOutputRecorderPairs[i].pFinalOutputRecorderHandle = nullptr;
                --m_totalUsed;
                break;
            }
        }
        os::UnlockMutex(&m_mutex);
    }

    void* FindPair(const FinalOutputRecorder* key) NN_NOEXCEPT
    {
        os::LockMutex(&m_mutex);
        for( int i = 0; i < nn::audio::FinalOutputRecorderCountMax; ++i )
        {
            if( m_finalOutputRecorderPairs[i].pFinalOutputRecorderKey == key )
            {
                os::UnlockMutex(&m_mutex);
                return m_finalOutputRecorderPairs[i].pFinalOutputRecorderHandle;
            }
        }
        os::UnlockMutex(&m_mutex);
        return nullptr;
    }

private:
    struct FinalOutputRecorderPair
    {
        const FinalOutputRecorder* pFinalOutputRecorderKey;
        void* pFinalOutputRecorderHandle;
    };

    FinalOutputRecorderPair m_finalOutputRecorderPairs[nn::audio::FinalOutputRecorderCountMax];
    int m_totalUsed;
    os::MutexType m_mutex;
};

FinalOutputRecorderMap g_FinalOutputRecorderMap;
}
#define NN_AUDIO_USE_SYSTEM_PROCESS

nn::sf::SharedPointer<detail::IFinalOutputRecorderManager> CreateFinalOutputRecorderManager() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#if defined(NN_AUDIO_USE_SYSTEM_PROCESS)
    return CreateFinalOutputRecorderManagerByHipc();
#else
    return CreateFinalOutputRecorderManagerByDfc();
#endif
#else
    return CreateFinalOutputRecorderManagerByDfc();
#endif
}

Result OpenFinalOutputRecorder(nn::audio::FinalOutputRecorder* pFinalOutputRecorder, const nn::audio::FinalOutputRecorderParameter& parameter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFinalOutputRecorder);
    NN_SDK_REQUIRES(parameter.channelCount == 0 || parameter.channelCount == 2 || parameter.channelCount == 6);

    bool canOpen = g_FinalOutputRecorderMap.FindPair(pFinalOutputRecorder) == nullptr;
    NN_SDK_REQUIRES(canOpen, "pFinalOutputRecorder %p has already been opened", pFinalOutputRecorder);
    NN_RESULT_THROW_UNLESS(canOpen, ResultAlreadyOpen());

    nn::audio::detail::FinalOutputRecorderParameterInternal finalOutputRecorderParam;

    nn::sf::SharedPointer<detail::IFinalOutputRecorder> finalOutputRecorderShared;
    auto processHandle = nn::os::InvalidNativeHandle;

    #if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
        processHandle = nn::dd::GetCurrentProcessHandle();
        NN_ABORT_UNLESS(processHandle != nn::os::InvalidNativeHandle);
    #endif

    auto finalOutputRecorderManager = CreateFinalOutputRecorderManager();
    NN_RESULT_DO(finalOutputRecorderManager->OpenFinalOutputRecorder(&finalOutputRecorderShared, parameter, nn::sf::NativeHandle(processHandle, false), &finalOutputRecorderParam, RecordAll) );

    g_FinalOutputRecorderMap.AddPair(pFinalOutputRecorder, finalOutputRecorderShared.Detach());

    pFinalOutputRecorder->sampleRate = finalOutputRecorderParam.sampleRate;
    pFinalOutputRecorder->channelCount = finalOutputRecorderParam.channelCount;
    pFinalOutputRecorder->sampleFormat = static_cast<nn::audio::SampleFormat>(finalOutputRecorderParam.sampleFormat);
    pFinalOutputRecorder->state = static_cast<nn::audio::FinalOutputRecorderState>(finalOutputRecorderParam.state);
    NN_RESULT_SUCCESS;
}

Result OpenFinalOutputRecorder(nn::audio::FinalOutputRecorder* pOutFinalOutputRecorder, nn::os::SystemEvent* pOutSystemEvent, const FinalOutputRecorderParameter& parameter) NN_NOEXCEPT
{
    NN_RESULT_DO(OpenFinalOutputRecorder(pOutFinalOutputRecorder, parameter));

    nn::sf::NativeHandle systemEventHandle;
    NN_RESULT_DO(static_cast<detail::IFinalOutputRecorder*>(g_FinalOutputRecorderMap.FindPair(pOutFinalOutputRecorder))->RegisterBufferEvent(&systemEventHandle));
    pOutSystemEvent->AttachReadableHandle(systemEventHandle.GetOsHandle(), systemEventHandle.IsManaged(), nn::os::EventClearMode_AutoClear);
    systemEventHandle.Detach();

    NN_RESULT_SUCCESS;
}

nn::audio::FinalOutputRecorderState GetFinalOutputRecorderState(const nn::audio::FinalOutputRecorder* pFinalOutputRecorder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFinalOutputRecorder);
    uint32_t audioState;

    void* handle = g_FinalOutputRecorderMap.FindPair(pFinalOutputRecorder);
    if( handle == nullptr )
    {
        return pFinalOutputRecorder->state;
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS( static_cast<detail::IFinalOutputRecorder*>(handle)->GetFinalOutputRecorderState(&audioState) );

    return static_cast<FinalOutputRecorderState>(audioState);
}

void CloseFinalOutputRecorder(nn::audio::FinalOutputRecorder* pFinalOutputRecorder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFinalOutputRecorder);
    void* handle = g_FinalOutputRecorderMap.FindPair(pFinalOutputRecorder);
    if( handle == nullptr )
    {
        return;
    }

    NN_SDK_REQUIRES_EQUAL(GetFinalOutputRecorderState(pFinalOutputRecorder), FinalOutputRecorderState_Stopped);
    pFinalOutputRecorder->state = FinalOutputRecorderState_Stopped;
    nn::sf::ReleaseSharedObject(static_cast<detail::IFinalOutputRecorder*>(handle));
    g_FinalOutputRecorderMap.RemovePair(pFinalOutputRecorder);
}

Result StartFinalOutputRecorder(nn::audio::FinalOutputRecorder* pFinalOutputRecorder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFinalOutputRecorder);
    NN_SDK_REQUIRES_EQUAL(GetFinalOutputRecorderState(pFinalOutputRecorder), FinalOutputRecorderState_Stopped);
    void* handle = g_FinalOutputRecorderMap.FindPair(pFinalOutputRecorder);
    NN_SDK_ASSERT(handle);
    return static_cast<detail::IFinalOutputRecorder*>(handle)->Start();
}

void StopFinalOutputRecorder(nn::audio::FinalOutputRecorder* pFinalOutputRecorder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFinalOutputRecorder);
    NN_SDK_REQUIRES_EQUAL(GetFinalOutputRecorderState(pFinalOutputRecorder), nn::audio::FinalOutputRecorderState_Started);
    void* handle = g_FinalOutputRecorderMap.FindPair(pFinalOutputRecorder);
    NN_SDK_ASSERT(handle);
    static_cast<detail::IFinalOutputRecorder*>(handle)->Stop();
}

int GetFinalOutputRecorderSampleRate(const nn::audio::FinalOutputRecorder* pFinalOutputRecorder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFinalOutputRecorder);
    return pFinalOutputRecorder->sampleRate;
}

int GetFinalOutputRecorderChannelCount(const nn::audio::FinalOutputRecorder* pFinalOutputRecorder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFinalOutputRecorder);
    return pFinalOutputRecorder->channelCount;
}

SampleFormat GetFinalOutputRecorderSampleFormat(const nn::audio::FinalOutputRecorder* pFinalOutputRecorder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFinalOutputRecorder);
    return pFinalOutputRecorder->sampleFormat;
}

void SetFinalOutputRecorderBufferInfo(FinalOutputRecorderBuffer* pOutFinalOutputRecorderBuffer, void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutFinalOutputRecorderBuffer);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
#if defined(NN_BUILD_CONFIG_ADDRESS_32)
    pOutFinalOutputRecorderBuffer->_bufferU64 = 0;
    pOutFinalOutputRecorderBuffer->_sizeU64 = 0;
#endif
    pOutFinalOutputRecorderBuffer->buffer = buffer;
    pOutFinalOutputRecorderBuffer->size = size;
}

void SetFinalOutputRecorderBufferInfo(FinalOutputRecorderBuffer* pOutFinalOutputRecorderBuffer, void* buffer, size_t bufferSize, size_t dataSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutFinalOutputRecorderBuffer);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_ALIGNED(buffer, FinalOutputRecorderBuffer::AddressAlignment);
    NN_SDK_REQUIRES_ALIGNED(bufferSize, FinalOutputRecorderBuffer::SizeGranularity);
    NN_SDK_REQUIRES_LESS_EQUAL(dataSize, bufferSize);
#if defined(NN_BUILD_CONFIG_ADDRESS_32)
    pOutFinalOutputRecorderBuffer->_bufferU64 = 0;
    pOutFinalOutputRecorderBuffer->_bufferSizeU64 = 0;
    pOutFinalOutputRecorderBuffer->_sizeU64 = 0;
#endif
    pOutFinalOutputRecorderBuffer->buffer = buffer;
    pOutFinalOutputRecorderBuffer->bufferSize = bufferSize;
    pOutFinalOutputRecorderBuffer->size = dataSize;
}

void* GetFinalOutputRecorderBufferDataPointer(const FinalOutputRecorderBuffer* pFinalOutputRecorderBuffer) NN_NOEXCEPT
{
    return pFinalOutputRecorderBuffer->buffer;
}

size_t GetFinalOutputRecorderBufferDataSize(const FinalOutputRecorderBuffer* pFinalOutputRecorderBuffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFinalOutputRecorderBuffer);
    return pFinalOutputRecorderBuffer->size;
}

void AppendFinalOutputRecorderBuffer(nn::audio::FinalOutputRecorder* pFinalOutputRecorder, FinalOutputRecorderBuffer* pFinalOutputRecorderBuffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFinalOutputRecorder);
    NN_SDK_REQUIRES_NOT_NULL(pFinalOutputRecorderBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pFinalOutputRecorderBuffer->buffer);
    NN_SDK_REQUIRES_ALIGNED(pFinalOutputRecorderBuffer->buffer, nn::audio::FinalOutputRecorderBuffer::AddressAlignment);
    NN_SDK_REQUIRES_ALIGNED(pFinalOutputRecorderBuffer->bufferSize, nn::audio::FinalOutputRecorderBuffer::SizeGranularity);
    NN_SDK_REQUIRES(pFinalOutputRecorderBuffer->size %
        (nn::audio::GetFinalOutputRecorderChannelCount(pFinalOutputRecorder) * nn::audio::GetSampleByteSize(nn::audio::GetFinalOutputRecorderSampleFormat(pFinalOutputRecorder))) == 0);

    void* handle = g_FinalOutputRecorderMap.FindPair(pFinalOutputRecorder);
    NN_SDK_ASSERT(handle);
    nn::sf::InBuffer inBuf( reinterpret_cast<char*>(pFinalOutputRecorderBuffer), sizeof( *pFinalOutputRecorderBuffer ) );
    pFinalOutputRecorderBuffer->released = nn::TimeSpan::FromNanoSeconds(0);
    static_cast<detail::IFinalOutputRecorder*>(handle)->AppendFinalOutputRecorderBufferAuto(inBuf, reinterpret_cast<uint64_t>(pFinalOutputRecorderBuffer));
}

FinalOutputRecorderBuffer* GetReleasedFinalOutputRecorderBuffer(nn::audio::FinalOutputRecorder* pFinalOutputRecorder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFinalOutputRecorder);
    int countRet = 0;
    int count = 1;

    uint64_t finalOutputRecorderArray;
    nn::sf::OutBuffer inArray(reinterpret_cast<char *>(&finalOutputRecorderArray), count*sizeof(finalOutputRecorderArray));
    int64_t released;
    void* handle = g_FinalOutputRecorderMap.FindPair(pFinalOutputRecorder);
    NN_SDK_ASSERT(handle);
    static_cast<detail::IFinalOutputRecorder*>(handle)->GetReleasedFinalOutputRecorderBuffersAuto(inArray, &countRet, &released);

    FinalOutputRecorderBuffer* inBuffer = reinterpret_cast<FinalOutputRecorderBuffer*>(finalOutputRecorderArray);
    for(int i = 0; i < countRet; ++i)
    {
        inBuffer[i].released = nn::TimeSpan::FromNanoSeconds(released);
    }

    return inBuffer;
}

bool ContainsFinalOutputRecorderBuffer(const FinalOutputRecorder* pFinalOutputRecorder, const FinalOutputRecorderBuffer* pFinalOutputRecorderBuffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFinalOutputRecorder);
    NN_SDK_REQUIRES_NOT_NULL(pFinalOutputRecorderBuffer);
    bool contains;

    void* handle = g_FinalOutputRecorderMap.FindPair(pFinalOutputRecorder);
    NN_SDK_ASSERT(handle);
    NN_ABORT_UNLESS_RESULT_SUCCESS( static_cast<detail::IFinalOutputRecorder*>(handle)->ContainsFinalOutputRecorderBuffer(reinterpret_cast<uint64_t>(pFinalOutputRecorderBuffer), &contains) );

    return contains;
}

nn::Result RegisterBufferEvent(FinalOutputRecorder* pFinalOutputRecorder, nn::os::SystemEvent* pBufferEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFinalOutputRecorder);
    nn::sf::NativeHandle systemEventHandle;

    void* handle = g_FinalOutputRecorderMap.FindPair(pFinalOutputRecorder);
    NN_SDK_ASSERT(handle);
    NN_RESULT_DO(static_cast<detail::IFinalOutputRecorder*>(handle)->RegisterBufferEvent(&systemEventHandle));

    pBufferEvent->AttachReadableHandle(systemEventHandle.GetOsHandle(), systemEventHandle.IsManaged(), nn::os::EventClearMode_AutoClear);
    systemEventHandle.Detach();
    NN_RESULT_SUCCESS;
}

void InitializeFinalOutputRecorderParameter(FinalOutputRecorderParameter* param) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(param);
    param->sampleRate = 0;
    param->channelCount = 0;
}

nn::TimeSpan GetFinalOutputRecorderBufferEndTime(const FinalOutputRecorderBuffer* pFinalOutputRecorderBuffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFinalOutputRecorderBuffer);

    return pFinalOutputRecorderBuffer->released;
}

int GetAdspLoad() NN_NOEXCEPT
{
    auto finalOutputRecorderManager = CreateFinalOutputRecorderManager();
    int32_t value = 0;
    auto result = finalOutputRecorderManager->GetAdspLoad(&value);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    return value;
}

}} //namespace audio / nn
