﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <mutex>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/sf/sf_Types.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/dd/dd_DeviceAddressSpace.h>
#include <nn/applet/applet_Apis.h>
#include <nn/os/os_SdkSystemEventApi.h>

#include <nn/audio/server/audio_UserServiceHipcServer.h>
#include <nn/audio/audio_AudioRenderer.h>
#include <nn/audio/audio_Result.private.h>
#include "audio_CreateAudioRendererManager.h"
#include "../audio_AudioRendererUtil.h"
#include "../audio_VoiceInfoManager.h"
#include "../audio_MixManager.h"
#include "../audio_EffectManager.h"
#include "../audio_SinkManager.h"
#include "../audio_PerformanceBufferManager.h"
#include "../audio_MemoryPoolManager.h"
#include "../audio_BehaviorManager.h"
#include "../common/audio_AudioRendererParameterInternal.h"
#include "../audio_ResourceExclusionChecker.h"
#include "../audio_RendererInfo.h"


// These criteria must be same as IsValidAudioRendererParameter()
#if defined(NN_BUILD_CONFIG_OS_WIN)
#define NN_AUDIO_SDK_REQUIRES_RENDERING_METHOD(renderingDevice, executionMode) \
    NN_SDK_REQUIRES(renderingDevice == AudioRendererRenderingDevice_Cpu);
#else
#define NN_AUDIO_SDK_REQUIRES_RENDERING_METHOD(renderingDevice, executionMode) \
    NN_SDK_REQUIRES((renderingDevice == AudioRendererRenderingDevice_Cpu && executionMode == AudioRendererExecutionMode_ManualExecution) \
                 || (renderingDevice == AudioRendererRenderingDevice_AudioCoprocessor && executionMode == AudioRendererExecutionMode_AutoExecution));
#endif

#define NN_AUDIO_SDK_REQUIRES_FOR_AUDIO_RENDERER_PARAMETER(parameter)                                  \
    NN_SDK_REQUIRES(parameter._magic == common::GetCurrentRevision(),                                  \
    "AudioRendererParmeter must be initialize with nn::audio::InitializeAudioRendererParameter()");    \
    NN_SDK_REQUIRES(parameter.sampleRate == 32000 || parameter.sampleRate == 48000);                   \
    NN_SDK_REQUIRES(IsValidMixBufferSampleCount(parameter.sampleCount, parameter.sampleRate));         \
    NN_SDK_REQUIRES_GREATER(parameter.mixBufferCount, 0);                                              \
    NN_SDK_REQUIRES_GREATER(parameter.voiceCount, 0);                                                  \
    NN_SDK_REQUIRES_GREATER(parameter.sinkCount, 0);                                                   \
    NN_SDK_REQUIRES_GREATER_EQUAL(parameter.effectCount, 0);                                           \
    NN_SDK_REQUIRES_GREATER_EQUAL(parameter.subMixCount, 0);                                           \
    NN_SDK_REQUIRES_LESS_EQUAL(parameter.mixBufferCount, AudioRendererParameter::MixBufferCountMax);   \
    NN_SDK_REQUIRES_LESS_EQUAL(parameter.voiceCount, AudioRendererParameter::VoiceCountMax);           \
    NN_SDK_REQUIRES_LESS_EQUAL(parameter.sinkCount, AudioRendererParameter::SinkCountMax);             \
    NN_SDK_REQUIRES_LESS_EQUAL(parameter.effectCount, AudioRendererParameter::EffectCountMax);         \
    NN_SDK_REQUIRES_LESS_EQUAL(parameter.subMixCount, AudioRendererParameter::SubMixCountMax);         \
    NN_AUDIO_SDK_REQUIRES_RENDERING_METHOD(parameter.renderingDevice, parameter.executionMode);

namespace nn {
namespace audio {
namespace detail {
std::atomic<uint32_t> g_AudioRendererCount;
} // detail

namespace {
#if !defined(NN_SDK_BUILD_RELEASE)
bool IsValidMixBufferSampleCount(int sampleCount, int sampleRate) NN_NOEXCEPT
{
    // TODO
    return (sampleCount == 240 || sampleCount == 160) && sampleRate / 200 == sampleCount;
}
#endif  // #if !defined(NN_SDK_BUILD_RELEASE)

bool IsDfc(const AudioRendererParameter& parameter) NN_NOEXCEPT
{
    return parameter.renderingDevice == AudioRendererRenderingDevice_Cpu;
}

size_t UpdateRendererInfoOutStatus(const AudioRendererConfig* pConfig, const void* pData) NN_NOEXCEPT
{
    auto pRendererInfo = reinterpret_cast<const RendererInfo*>(pData);

    memcpy(pConfig->_pRendererInfo, pRendererInfo, sizeof(*pRendererInfo));

    return sizeof(RendererInfo);
}

} // anonymous namespace

nn::sf::SharedPointer<detail::IAudioRendererManager> CreateAudioRendererManager(bool isDfc) NN_NOEXCEPT
{
    if(isDfc)
    {
        return CreateAudioRendererManagerByDfc();
    }
    else
    {
        return CreateAudioRendererManagerByHipc();
    }
}

size_t GetAudioRendererWorkBufferSize(const AudioRendererParameter& parameter) NN_NOEXCEPT
{
    NN_AUDIO_SDK_REQUIRES_FOR_AUDIO_RENDERER_PARAMETER(parameter);
    auto audioRendererManager = CreateAudioRendererManager(IsDfc(parameter));
    int64_t size = 0; // for klocwork inspection
    NN_ABORT_UNLESS_RESULT_SUCCESS(audioRendererManager->GetWorkBufferSize(&size, detail::ConvertToAudioRendererParameterInternal(parameter)));
    return static_cast<size_t>(size);
}

nn::Result OpenAudioRenderer(AudioRendererHandle* outHandle, const AudioRendererParameter& parameter, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outHandle);
    NN_AUDIO_SDK_REQUIRES_FOR_AUDIO_RENDERER_PARAMETER(parameter);
    NN_SDK_REQUIRES_NOT_NULL(workBuffer);
    NN_SDK_REQUIRES_GREATER_EQUAL(workBufferSize, GetAudioRendererWorkBufferSize(parameter));
    NN_SDK_REQUIRES_ALIGNED(reinterpret_cast<uintptr_t>(workBuffer), nn::os::MemoryPageSize);

    // Update the number of AudioRenderer clients open.
    if(detail::g_AudioRendererCount++ >= static_cast<uint32_t>(AudioRendererCountMax))
    {
        detail::g_AudioRendererCount--;
        NN_RESULT_THROW(ResultOutOfResource());
    }

    auto audioRendererManager = CreateAudioRendererManager(IsDfc(parameter));

    nn::sf::SharedPointer<detail::IAudioRenderer> audioRenderer;
    auto processHandle = nn::os::InvalidNativeHandle;
    auto appletResourceUserId = nn::applet::AppletResourceUserId::GetInvalidId();
    nn::Result result;

    const auto IsManualExecutionCpuRerindering = (parameter.executionMode == AudioRendererExecutionMode_ManualExecution && parameter.renderingDevice == AudioRendererRenderingDevice_Cpu);

    if(IsManualExecutionCpuRerindering)
    {
        result = audioRendererManager->OpenAudioRendererForManualExecution(&audioRenderer, detail::ConvertToAudioRendererParameterInternal(parameter), reinterpret_cast<uint64_t>(workBuffer), nn::sf::NativeHandle(processHandle, false), workBufferSize, appletResourceUserId);
    }
    else
    {
        nn::os::TransferMemory transferMemory(workBuffer, workBufferSize, nn::os::MemoryPermission_None);
#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
        processHandle = dd::GetCurrentProcessHandle();
        NN_ABORT_UNLESS(processHandle != nn::os::InvalidNativeHandle);

        appletResourceUserId = applet::GetAppletResourceUserId();
        // NN_ABORT_UNLESS(appletResourceUserId != nn::applet::AppletResourceUserId::GetInvalidId());
#endif
        result = audioRendererManager->OpenAudioRenderer(&audioRenderer, detail::ConvertToAudioRendererParameterInternal(parameter), nn::sf::NativeHandle(transferMemory.Detach(), true), nn::sf::NativeHandle(processHandle, false), workBufferSize, appletResourceUserId);
    }

    NN_RESULT_TRY(result)
        NN_RESULT_CATCH(ResultInvalidClientProcessHandle)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
    NN_RESULT_END_TRY

    outHandle->_handle = audioRenderer.Detach();
    outHandle->_context = nullptr;

    if(auto pChecker = nn::audio::detail::AcquireResourceExclusionChecker())
    {
        pChecker->RegisterInternalHandle(outHandle->_handle);
    }

    NN_RESULT_SUCCESS;
}

nn::Result OpenAudioRenderer(AudioRendererHandle* outHandle, nn::os::SystemEvent* pOutSystemEvent, const AudioRendererParameter& parameter, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outHandle);
    NN_SDK_REQUIRES_NOT_NULL(pOutSystemEvent);
    NN_SDK_REQUIRES_NOT_NULL(workBuffer);

    NN_RESULT_DO(OpenAudioRenderer(outHandle, parameter, workBuffer, workBufferSize));

    const auto IsManualExecutionCpuRerindering = (parameter.executionMode == AudioRendererExecutionMode_ManualExecution && parameter.renderingDevice == AudioRendererRenderingDevice_Cpu);

    if(IsManualExecutionCpuRerindering)
    {
        NN_RESULT_DO(nn::os::CreateSystemEvent(pOutSystemEvent->GetBase(), nn::os::EventClearMode_AutoClear, false));
        outHandle->_context= pOutSystemEvent;

    }
    else
    {
        nn::sf::NativeHandle systemEventHandle;
        NN_RESULT_DO(static_cast<detail::IAudioRenderer*>(outHandle->_handle)->QuerySystemEvent(&systemEventHandle));
        outHandle->_context = nullptr;
        pOutSystemEvent->AttachReadableHandle(systemEventHandle.GetOsHandle(), systemEventHandle.IsManaged(), nn::os::EventClearMode_AutoClear);
        systemEventHandle.Detach();
    }

    NN_RESULT_SUCCESS;
}

void CloseAudioRenderer(AudioRendererHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(handle._handle);
    NN_SDK_REQUIRES_EQUAL(GetAudioRendererState(handle), AudioRendererState_Stopped);
    nn::sf::ReleaseSharedObject(static_cast<detail::IAudioRenderer*>(handle._handle));

    if(auto pChecker = nn::audio::detail::FindResourceExclusionCheckerFromInternalHandle(handle._handle))
    {
        detail::ReleaseResourceExclusionChecker(pChecker);
    }

    // Update the number of AudioRenderer clients open.
    if(detail::g_AudioRendererCount-- == 0u)
    {
        NN_SDK_ASSERT(false, "g_AudioRendererCount == 0");
    }
}

nn::Result StartAudioRenderer(AudioRendererHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(handle._handle);
    NN_SDK_REQUIRES_EQUAL(GetAudioRendererState(handle), AudioRendererState_Stopped);
    return static_cast<detail::IAudioRenderer*>(handle._handle)->Start();
}

void StopAudioRenderer(AudioRendererHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(handle._handle);
    NN_SDK_REQUIRES_EQUAL(GetAudioRendererState(handle), AudioRendererState_Started);
    NN_ABORT_UNLESS_RESULT_SUCCESS(static_cast<detail::IAudioRenderer*>(handle._handle)->Stop());
}

int GetAudioRendererSampleRate(AudioRendererHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(handle._handle);
    int32_t value = 0; // for klocwork inspection
    NN_ABORT_UNLESS_RESULT_SUCCESS(static_cast<detail::IAudioRenderer*>(handle._handle)->GetSampleRate(&value));
    return value;
}

int GetAudioRendererSampleCount(AudioRendererHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(handle._handle);
    int32_t value = 0; // for klocwork inspection
    NN_ABORT_UNLESS_RESULT_SUCCESS(static_cast<detail::IAudioRenderer*>(handle._handle)->GetSampleCount(&value));
    return value;
}

int GetAudioRendererMixBufferCount(AudioRendererHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(handle._handle);
    int32_t value = 0; // for klocwork inspection
    NN_ABORT_UNLESS_RESULT_SUCCESS(static_cast<detail::IAudioRenderer*>(handle._handle)->GetMixBufferCount(&value));
    return value;
}

AudioRendererState GetAudioRendererState(AudioRendererHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(handle._handle);
    int32_t value = 0; // for klocwork inspection
    NN_ABORT_UNLESS_RESULT_SUCCESS(static_cast<detail::IAudioRenderer*>(handle._handle)->GetState(&value));
    return static_cast<AudioRendererState>(value);
}

nn::Result RequestUpdateAudioRenderer(AudioRendererHandle handle, const AudioRendererConfig* pConfig) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(handle._handle);
    NN_SDK_REQUIRES_NOT_NULL(pConfig);

    auto pChecker = nn::audio::detail::FindResourceExclusionCheckerFromInternalHandle(handle._handle);
    NN_SDK_ASSERT(pChecker == nullptr || pChecker->RegisterConfig(pConfig->_pConfigBuffer, pConfig->_configBufferSize));
    detail::ScopedHandleAccessChecker marker(pChecker);

#if defined(NN_SDK_BUILD_DEBUG)
    // Dirty InParameter to improve debuggability.
    memset(pConfig->_pInParameter, 0xFF, pConfig->_pInParameterSize);
#endif

    // skip outStatus clear

    uintptr_t pInParameter = reinterpret_cast<uintptr_t>(pConfig->_pInParameter);
    auto inParamHeader = new(reinterpret_cast<void*>(pInParameter)) nn::audio::common::UpdateDataHeader();
    inParamHeader->Initialize(pConfig->_pBehaviorManager->GetRevision());

    inParamHeader->SetSize(&common::UpdateDataSizes::behavior,
        pConfig->_pBehaviorManager->UpdateInParameter(inParamHeader->GetCurrentPtr(pInParameter)));

    inParamHeader->SetSize(&common::UpdateDataSizes::memoryPool,
        pConfig->_pMemoryPoolManager->UpdateInParameter(reinterpret_cast<nn::audio::MemoryPoolInfo::InParameter*>(inParamHeader->GetCurrentPtr(pInParameter))));

    size_t voiceChInfoSize = 0;
    size_t voiceInfoSize = 0;
    pConfig->_pVoiceInfoManager->UpdateVoiceRelatedParameters(inParamHeader->GetCurrentPtr(pInParameter), &voiceInfoSize, &voiceChInfoSize);
    inParamHeader->SetSize(&common::UpdateDataSizes::voiceChannelResource, voiceChInfoSize);
    inParamHeader->SetSize(&common::UpdateDataSizes::voice, voiceInfoSize);

    inParamHeader->SetSize(&common::UpdateDataSizes::effect,
        pConfig->_pEffectManager->UpdateEffectsInParameter(inParamHeader->GetCurrentPtr(pInParameter)));

    inParamHeader->SetSize(&common::UpdateDataSizes::splitter,
        pConfig->_pSplitterInfoManager->UpdateInParameter(inParamHeader->GetCurrentPtr(pInParameter)));

    inParamHeader->SetSize(&common::UpdateDataSizes::mix,
        pConfig->_pMixManager->UpdateMixInParameter(inParamHeader->GetCurrentPtr(pInParameter)));

    inParamHeader->SetSize(&common::UpdateDataSizes::sink,
        pConfig->_pSinkManager->UpdateSinkInParameter(reinterpret_cast<nn::audio::common::SinkInParameter*>(inParamHeader->GetCurrentPtr(pInParameter))));

    inParamHeader->SetSize(&common::UpdateDataSizes::performance,
        pConfig->_pPerformanceBufferManager->UpdateInParameter(reinterpret_cast<nn::audio::PerformanceBufferInfo::InParameter*>(inParamHeader->GetCurrentPtr(pInParameter))));

    NN_ABORT_UNLESS(inParamHeader->GetTotalSize() <= pConfig->_pInParameterSize);

    // TODO: Insert check routine here to validate addresses in MemoryPool.

    nn::sf::InBuffer inBuffer(static_cast<char*>(pConfig->_pInParameter), inParamHeader->GetTotalSize());
    nn::sf::OutBuffer outBuffer(static_cast<char*>(pConfig->_pOutStatus), pConfig->_pOutStatusSize);
    auto perfManager = pConfig->_pPerformanceBufferManager;
    nn::sf::OutBuffer performanceBuffer(static_cast<char*>(perfManager->GetBufferBase()), perfManager->GetBufferSize());

    NN_RESULT_TRY(static_cast<detail::IAudioRenderer*>(handle._handle)->RequestUpdate(outBuffer, performanceBuffer, inBuffer))
        NN_RESULT_CATCH(ResultInvalidUpdateInfo)
            {
                return ResultInvalidUpdateInfo();
            }
        NN_RESULT_CATCH(ResultCycleDetected)
            {
                return ResultCycleDetected();
            }
    NN_RESULT_END_TRY

    auto outStatusHeader = reinterpret_cast<nn::audio::common::UpdateDataHeader*>(pConfig->_pOutStatus);
    uintptr_t pOutStatus = reinterpret_cast<uintptr_t>(pConfig->_pOutStatus);
    size_t size = 0;
    size_t offset = sizeof(nn::audio::common::UpdateDataHeader);

    size = pConfig->_pMemoryPoolManager->UpdateOutStatus(reinterpret_cast<nn::audio::MemoryPoolInfo::OutStatus*>(pOutStatus + offset));
    NN_RESULT_THROW_UNLESS(outStatusHeader->CheckSize(&common::UpdateDataSizes::memoryPool, size), ResultInvalidUpdateInfo());
    offset += size;

    size = pConfig->_pVoiceInfoManager->UpdateVoiceOutStatus(reinterpret_cast<VoiceInfo::OutStatus*>(pOutStatus + offset));
    NN_RESULT_THROW_UNLESS(outStatusHeader->CheckSize(&common::UpdateDataSizes::voice, size), ResultInvalidUpdateInfo());
    offset += size;

    size = pConfig->_pEffectManager->UpdateEffectOutStatus(reinterpret_cast<EffectInfo::OutStatus*>(pOutStatus + offset));
    NN_RESULT_THROW_UNLESS(outStatusHeader->CheckSize(&common::UpdateDataSizes::effect, size), ResultInvalidUpdateInfo());
    offset += size;

    size = pConfig->_pSinkManager->UpdateSinkOutStatus(reinterpret_cast<common::SinkOutStatus*>(pOutStatus + offset));
    NN_RESULT_THROW_UNLESS(outStatusHeader->CheckSize(&common::UpdateDataSizes::sink, size), ResultInvalidUpdateInfo());
    offset += size;

    size = pConfig->_pPerformanceBufferManager->UpdateOutStatus(reinterpret_cast<void*>(pOutStatus + offset));
    NN_RESULT_THROW_UNLESS(outStatusHeader->CheckSize(&common::UpdateDataSizes::performance, size), ResultInvalidUpdateInfo());
    offset += size;

    size = pConfig->_pBehaviorManager->UpdateOutStatus(reinterpret_cast<void*>(pOutStatus + offset));
    NN_RESULT_THROW_UNLESS(outStatusHeader->CheckSize(&common::UpdateDataSizes::behavior, size), ResultInvalidUpdateInfo());
    offset += size;

    size = UpdateRendererInfoOutStatus(pConfig, reinterpret_cast<void*>(pOutStatus + offset));
    NN_RESULT_THROW_UNLESS(outStatusHeader->CheckSize(&common::UpdateDataSizes::rendererInfo, size), ResultInvalidUpdateInfo());
    offset += size;

    NN_ABORT_UNLESS(outStatusHeader->GetTotalSize() == pConfig->_pOutStatusSize);

    // Warning prints.
    return pConfig->_pBehaviorManager->CheckErrors();
}

nn::Result SetAudioRendererRenderingTimeLimit(AudioRendererHandle handle, int limitPercent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(handle._handle);
    NN_SDK_REQUIRES_MINMAX(limitPercent, 0, 100);

    NN_RESULT_DO(static_cast<detail::IAudioRenderer*>(handle._handle)->SetRenderingTimeLimit(limitPercent));

    NN_RESULT_SUCCESS;
}

int GetAudioRendererRenderingTimeLimit(AudioRendererHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(handle._handle);
    int32_t limitPercent = 0;
    NN_RESULT_DO(static_cast<detail::IAudioRenderer*>(handle._handle)->GetRenderingTimeLimit(&limitPercent));
    return limitPercent;
}

Result ExecuteAudioRendererRendering(AudioRendererHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(handle._handle);
    NN_RESULT_DO(static_cast<detail::IAudioRenderer*>(handle._handle)->ExecuteAudioRendererRendering());

    if(handle._context != nullptr)
    {
        static_cast<nn::os::SystemEvent*>(handle._context)->Signal();
    }

    NN_RESULT_SUCCESS;
}

}  // namespace audio
}  // namespace nn
