﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/os/os_SystemEventApi.h>
#include <nn/nn_SdkAssert.h>
#include <nn/audio/audio_Applet.h>
#include "audio_CreateAudioRendererManager.h"

namespace nn {
namespace audio {

namespace {

nn::sf::SharedPointer<detail::IAudioRendererManagerForApplet> CreateAudioRendererManagerForApplet() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    return CreateAudioRendererManagerForAppletByHipc();
#else
    return CreateAudioRendererManagerForAppletByDfc();
#endif
}

} // anonymous namespace

nn::Result RequestSuspendAudioRenderers(nn::applet::AppletResourceUserId id, nn::TimeSpan fadeTime) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(fadeTime, nn::TimeSpan::FromMilliSeconds(0), nn::TimeSpan::FromMilliSeconds(10));
    return CreateAudioRendererManagerForApplet()->RequestSuspend(id, fadeTime.GetNanoSeconds());
}

nn::Result RequestResumeAudioRenderers(nn::applet::AppletResourceUserId id, nn::TimeSpan fadeTime) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(fadeTime, nn::TimeSpan::FromMilliSeconds(0), nn::TimeSpan::FromMilliSeconds(10));
    return CreateAudioRendererManagerForApplet()->RequestResume(id, fadeTime.GetNanoSeconds());
}

nn::Result GetAudioRenderersProcessMasterVolume(float* pOutVolume, nn::applet::AppletResourceUserId id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutVolume);

    NN_RESULT_DO(CreateAudioRendererManagerForApplet()->GetProcessMasterVolume(pOutVolume, id));
    NN_RESULT_SUCCESS;
}

nn::Result SetAudioRenderersProcessMasterVolume(nn::applet::AppletResourceUserId id, float volume, nn::TimeSpan fadeTime) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(volume, nn::audio::ProcessMasterVolumeMin, nn::audio::ProcessMasterVolumeMax);
    NN_SDK_REQUIRES_MINMAX(fadeTime, nn::TimeSpan::FromMilliSeconds(0), nn::TimeSpan::FromMilliSeconds(1000));

    NN_RESULT_DO(CreateAudioRendererManagerForApplet()->SetProcessMasterVolume(id, volume, fadeTime.GetNanoSeconds()));
    NN_RESULT_SUCCESS;
}

nn::Result RegisterAppletResourceUserId(nn::applet::AppletResourceUserId id) NN_NOEXCEPT
{
    NN_RESULT_DO(CreateAudioRendererManagerForApplet()->RegisterAppletResourceUserId(id));
    NN_RESULT_SUCCESS;
}

nn::Result UnregisterAppletResourceUserId(nn::applet::AppletResourceUserId id) NN_NOEXCEPT
{
    NN_RESULT_DO(CreateAudioRendererManagerForApplet()->UnregisterAppletResourceUserId(id));
    NN_RESULT_SUCCESS;
}

nn::Result GetAudioRenderersProcessRecordVolume(float* pOutVolume, nn::applet::AppletResourceUserId id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutVolume);

    NN_RESULT_DO(CreateAudioRendererManagerForApplet()->GetProcessRecordVolume(pOutVolume, id));
    NN_RESULT_SUCCESS;
}

nn::Result SetAudioRenderersProcessRecordVolume(nn::applet::AppletResourceUserId id, float volume, nn::TimeSpan fadeTime) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(volume, nn::audio::ProcessRecordVolumeMin, nn::audio::ProcessRecordVolumeMax);
    NN_SDK_REQUIRES_MINMAX(fadeTime, nn::TimeSpan::FromMilliSeconds(0), nn::TimeSpan::FromMilliSeconds(1000));

    NN_RESULT_DO(CreateAudioRendererManagerForApplet()->SetProcessRecordVolume(id, volume, fadeTime.GetNanoSeconds()));
    NN_RESULT_SUCCESS;
}

}  // namespace audio
}  // namespace nn
